package com.af.v4.system.common.plugins.wechat;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.plugins.http.RestTools;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * 微信接口工具类
 */
@Component
public class WechatApiUtils {

    static final Logger LOGGER = LoggerFactory.getLogger(WechatApiUtils.class);

    private final WechatAuthUtils wechatAuthUtils;

    public WechatApiUtils(WechatAuthUtils wechatAuthUtils) {
        this.wechatAuthUtils = wechatAuthUtils;
    }

    /**
     * 获取所有关注了公众号的所有用户OpenId
     *
     * @param openIdArray 接受openId结果的列表
     * @param authInfo 鉴权信息
     * @param nextOpenIds 第一个拉取的OPENID，不填默认从头开始拉取
     * @return OpenId列表
     */
    private List<String> getAllOpenId(List<String> openIdArray, JSONObject authInfo, String nextOpenIds) {
        String orgId = authInfo.getString("appid");
        String accessToken = wechatAuthUtils.authorization(orgId, authInfo);
        String url = WechatApi.getGetOpenidUrl() + "?access_token=" + accessToken;
        if (nextOpenIds != null) {
            url = url + "&next_openid=" + nextOpenIds;
        }
        JSONObject result = new JSONObject(RestTools.get(url));
        if (result.has("data")) {
            String nextOpenId = result.getString("next_openid");
            JSONObject data = result.getJSONObject("data");
            JSONArray openIds = data.getJSONArray("openid");
            for (Object openId : openIds) {
                openIdArray.add((String) openId);
            }
            if (nextOpenId.isEmpty()) {
                return openIdArray;
            } else {
                // 继续拉取
                return getAllOpenId(openIdArray, authInfo, nextOpenId);
            }
        } else {
            throw new ServiceException("获取所有用户的OpenId失败，原因：" + result);
        }
    }

    public List<String> getAllOpenId(JSONObject authInfo) {
        List<String> allOpenIds = new ArrayList<>(10000);
        return getAllOpenId(allOpenIds, authInfo, null);
    }

    /**
     * 发送模板消息
     */
    public JSONObject sendTemplateMessage(JSONObject authInfo, String openId, String templateId, String url, String miniprogram, JSONObject data) {
        JSONObject body = new JSONObject();
        body.put("touser", openId);
        body.put("template_id", templateId);
        body.put("url", url);
        body.put("miniprogram", miniprogram);
        body.put("data", data);
        String orgId = authInfo.getString("appid");
        String accessToken = wechatAuthUtils.authorization(orgId, authInfo);
        String requestUrl = WechatApi.getSendTemplateMessageUrl() + "?access_token=" + accessToken;
        return new JSONObject(RestTools.post(requestUrl, body));
    }

    /**
     * 清空api的调用quota
     *
     * @apiNote 本接口用于清空公众号/小程序/第三方平台等接口的每日调用接口次数
     */
    public JSONObject clearQuota(JSONObject authInfo) {
        String orgId = authInfo.getString("appid");
        JSONObject body = new JSONObject();
        body.put("appid", orgId);
        String accessToken = wechatAuthUtils.authorization(orgId, authInfo);
        String requestUrl = WechatApi.getClearQuotaUrl() + "?access_token=" + accessToken;
        return new JSONObject(RestTools.post(requestUrl, body));
    }
}
