package com.af.v4.system.common.plugins.http.core;

import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.plugins.http.config.HttpClientConfig;
import com.af.v4.system.common.plugins.http.config.HttpLogSuppressor;
import org.apache.hc.client5.http.ConnectTimeoutException;
import org.apache.hc.client5.http.ConnectionKeepAliveStrategy;
import org.apache.hc.client5.http.HttpHostConnectException;
import org.apache.hc.client5.http.classic.methods.HttpUriRequestBase;
import org.apache.hc.client5.http.config.ConnectionConfig;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.DefaultHttpRequestRetryStrategy;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManagerBuilder;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.client5.http.ssl.DefaultClientTlsStrategy;
import org.apache.hc.client5.http.ssl.NoopHostnameVerifier;
import org.apache.hc.client5.http.ssl.TlsSocketStrategy;
import org.apache.hc.client5.http.ssl.TrustSelfSignedStrategy;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.HeaderElement;
import org.apache.hc.core5.http.HttpHeaders;
import org.apache.hc.core5.http.HttpStatus;
import org.apache.hc.core5.http.config.Registry;
import org.apache.hc.core5.http.config.RegistryBuilder;
import org.apache.hc.core5.http.message.BasicHeaderElementIterator;
import org.apache.hc.core5.ssl.SSLContexts;
import org.apache.hc.core5.util.TimeValue;
import org.apache.hc.core5.util.Timeout;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StopWatch;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;

/**
 * HttpClient连接池实现
 *
 * @author Mr.river
 * @see RestTools
 */
public class HttpConnectionPoolUtil extends BaseHttpPoolUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpConnectionPoolUtil.class);
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static ReentrantLock SYNC_LOCK = new ReentrantLock();
    /**
     * 发送HTTP请求的客户端单例
     */
    private static volatile CloseableHttpClient httpClient;

    /**
     * 获取httpclient的https实例（需要SSL证书）
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient(TlsSocketStrategy sslSocketFactory) {
        return createHttpClient(sslSocketFactory);
    }

    /**
     * 构建httpclient实例
     *
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient() {
        return createHttpClient(null);
    }

    /**
     * 构建httpclient实例
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient(TlsSocketStrategy sslSocketFactory) {
        HttpClientConfig config = new HttpClientConfig.Builder().build();
        //设置https相关信息
        if (sslSocketFactory == null) {
            try {
                sslSocketFactory = new DefaultClientTlsStrategy(
                        SSLContexts.custom().loadTrustMaterial(null, new TrustSelfSignedStrategy()).build(),
                        NoopHostnameVerifier.INSTANCE
                );
            } catch (NoSuchAlgorithmException | KeyManagementException | KeyStoreException e) {
                throw new RuntimeException(e);
            }
        }
        Registry<TlsSocketStrategy> registry = RegistryBuilder.<TlsSocketStrategy>create()
                .register("https", sslSocketFactory)
                .build();
        /* 设置连接参数 **/
        PoolingHttpClientConnectionManager manager =
                PoolingHttpClientConnectionManagerBuilder.create()
                        .setTlsSocketStrategy(registry.lookup("https"))
                        .build();
        // 最大连接数
        manager.setMaxTotal(HttpClientConfig.HTTP_MAX_POOL_SIZE);
        // 路由最大连接数
        manager.setDefaultMaxPerRoute(HttpClientConfig.HTTP_MAX_POOL_SIZE);
        // 连接配置参数
        ConnectionConfig connectionConfig = ConnectionConfig.custom()
                .setConnectTimeout(Timeout.ofSeconds(config.getConnectTimeout())).build();
        manager.setDefaultConnectionConfig(connectionConfig);
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(Timeout.ofSeconds(config.getConnectTimeout()))
                .setResponseTimeout(Timeout.ofSeconds(config.getSocketTimeout()))
                .build();

        //设置keepalive
        ConnectionKeepAliveStrategy myStrategy = (response, context) -> {
            // Honor 'keep-alive' header
            BasicHeaderElementIterator it = new BasicHeaderElementIterator(
                    response.headerIterator(HttpHeaders.KEEP_ALIVE));
            while (it.hasNext()) {
                HeaderElement he = it.next();
                String param = he.getName();
                String value = he.getValue();
                if (value != null && "timeout".equalsIgnoreCase(param)) {
                    try {
                        return TimeValue.ofSeconds(Long.parseLong(value) * 1000);
                    } catch (NumberFormatException ignore) {
                    }
                }
            }
            // Keep alive for 5 seconds only
            return TimeValue.ofSeconds(5 * 1000);
        };

        return HttpClients.custom()
                .setKeepAliveStrategy(myStrategy)
                .setConnectionManagerShared(true)
                .setConnectionManager(manager)
                .setRetryStrategy(new DefaultHttpRequestRetryStrategy(0, TimeValue.ofSeconds(1L)))
                .evictExpiredConnections()
                .evictIdleConnections(TimeValue.ofMilliseconds(HttpClientConfig.HTTP_IDEL_TIME_OUT))
                .setDefaultRequestConfig(requestConfig)
                .build();
    }

    /**
     * 获取HTTP请求 httpclient实例
     *
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient() {
        if (httpClient == null) {
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            SYNC_LOCK.lock();
            try {
                if (httpClient == null) {
                    httpClient = createHttpClient();
                }
            } finally {
                SYNC_LOCK.unlock();
            }
        }
        return httpClient;
    }

    /**
     * 发送通用HTTP请求
     *
     * @param value              请求参数
     * @param headersStr         请求头
     * @param config             请求配置
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @param base               请求类型
     * @return 请求结果
     */
    public static String request(String value, String headersStr, HttpClientConfig config, Boolean isStandardResponse, HttpUriRequestBase base) {
        return request(value, headersStr, config, base, isStandardResponse, null);
    }

    public static String request(String value, String headersStr, HttpClientConfig config, HttpUriRequestBase base) {
        return request(value, headersStr, config, false, base);
    }

    public static String requestFormUrlEncoded(String value, String headersStr, HttpClientConfig config, Boolean isStandardResponse, HttpUriRequestBase base) {
        return requestFormUrlEncoded(value, headersStr, config, base, isStandardResponse, null);
    }

    public static String requestFormUrlEncoded(String value, String headersStr, HttpClientConfig config, HttpUriRequestBase base) {
        return requestFormUrlEncoded(value, headersStr, config, false, base);
    }

    /**
     * 发送通用HTTP请求
     *
     * @param value              请求参数
     * @param headersStr         请求头
     * @param base               请求类型
     * @param httpClient         client对象
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @return 请求结果
     */
    public static String request(String value,
                                 String headersStr,
                                 HttpClientConfig config,
                                 HttpUriRequestBase base,
                                 Boolean isStandardResponse,
                                 CloseableHttpClient httpClient) {
        String url = base.getScheme() + "://" + base.getAuthority() + base.getPath();
        if (httpClient == null) {
            httpClient = getHttpClient();
        }
        //配置请求参数
        setRequestConfig(config, base);
        //设置请求头
        setHeaders(base, headersStr);
        //设置请求体
        if (base.getEntity() == null) {
            setBody(base, value);
        }
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        final AtomicReference<String> result = new AtomicReference<>();
        try {
            httpClient.execute(base, HttpClientContext.create(), response -> {
                stopWatch.stop();
                logExecutionTime(stopWatch);
                String body = getResponseData(response);
                if (isStandardResponse) {
                    JSONObject headerObject = new JSONObject();
                    Header[] headerArray = response.getHeaders();
                    for (Header header : headerArray) {
                        headerObject.put(header.getName(), header.getValue());
                    }
                    result.set(new JSONObject().put("body", body).put("header", headerObject).toString());
                } else {
                    result.set(body);
                }
                return result.get();
            });
        } catch (IOException e) {
            // 返回错误信息
            int code;
            if (e instanceof HttpHostConnectException | e instanceof ConnectTimeoutException) {
                code = HttpStatus.SC_GATEWAY_TIMEOUT;
            } else {
                code = HttpStatus.SC_SERVICE_UNAVAILABLE;
            }
            JSONObject errorResult = new JSONObject();
            errorResult.put("hasError", true);
            errorResult.put("code", code);
            errorResult.put("msg", e.getMessage());
            result.set(errorResult.toString());
        } finally {
            if (!HttpLogSuppressor.isEnabled()) {
                LOGGER.info("api: {}，header: {}，body: {}，response: {}，type: sync", url, headersStr, value, result);
            }
        }
        return result.get();
    }

    public static String requestFormUrlEncoded(String value,
                                               String headersStr,
                                               HttpClientConfig config,
                                               HttpUriRequestBase base,
                                               Boolean isStandardResponse,
                                               CloseableHttpClient httpClient) {
        setUrlEncodedBody(base, value);
        return request(value, headersStr, config, base, isStandardResponse, httpClient);
    }

    public static String requestFormUrlEncoded(String value,
                                               String headersStr,
                                               HttpClientConfig config,
                                               HttpUriRequestBase base,
                                               CloseableHttpClient httpClient) {
        return requestFormUrlEncoded(value, headersStr, config, base, false, httpClient);
    }
}
