package com.af.v4.system.common.plugins.security;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * 基于固定密码表的单表代换加密工具类
 * 用于对字符串进行可逆加密与解密，主要用于对数据库敏感字段进行可检索（支持 LIKE）的加密存储
 * <p>
 * 特性：
 * - 高性能、线程安全
 * - 固定映射表（JSON 格式）
 * - 支持汉字、数字、英文
 * - 未匹配字符保持原样
 * - 密文固定为 4 位字符
 * </p>
 */
public class SubstitutionCipherUtil {

    private static final Logger log = LoggerFactory.getLogger(SubstitutionCipherUtil.class);

    /**
     * 默认密码表路径
     */
    private static final String DEFAULT_CIPHER_TABLE_PATH = "cipher_table.json";

    /**
     * 密文固定长度
     */
    private static final int CIPHER_LENGTH = 4;

    /**
     * 明文到密文的映射表（加密用）
     */
    private static volatile Map<String, String> encryptMap;

    /**
     * 密文到明文的映射表（解密用）
     */
    private static volatile Map<String, String> decryptMap;

    /**
     * 读写锁，用于初始化时的线程安全
     */
    private static final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();

    /**
     * 初始化标识
     */
    private static volatile boolean initialized = false;

    /**
     * Jackson ObjectMapper
     */
    private static final ObjectMapper objectMapper = new ObjectMapper();

    // 静态初始化块，使用默认路径加载密码表
    static {
        try {
            init(DEFAULT_CIPHER_TABLE_PATH);
        } catch (Exception e) {
            log.error("使用默认路径加载密码表失败: {}", DEFAULT_CIPHER_TABLE_PATH, e);
            // 不抛出异常，允许后续手动初始化
        }
    }

    /**
     * 私有构造函数，防止实例化
     */
    private SubstitutionCipherUtil() {
        throw new UnsupportedOperationException("Utility class");
    }

    /**
     * 从指定路径初始化密码表
     *
     * @param passwordTablePath 密码表文件路径（相对于 classpath）
     */
    public static void init(String passwordTablePath) {
        lock.writeLock().lock();
        try {
            log.info("开始初始化密码表，路径: {}", passwordTablePath);

            // 加载 JSON 文件
            InputStream inputStream = SubstitutionCipherUtil.class.getClassLoader()
                    .getResourceAsStream(passwordTablePath);

            if (inputStream == null) {
                throw new RuntimeException("密码表文件不存在: " + passwordTablePath);
            }

            // 解析 JSON 为 Map
            Map<String, String> loadedMap;
            try {
                loadedMap = objectMapper.readValue(inputStream, new TypeReference<Map<String, String>>() {
                });
            } catch (IOException e) {
                throw new RuntimeException("密码表文件格式错误: " + passwordTablePath, e);
            }

            if (loadedMap == null || loadedMap.isEmpty()) {
                throw new RuntimeException("密码表文件为空: " + passwordTablePath);
            }

            // 验证密码表格式
            validateCipherTable(loadedMap);

            // 生成加密映射表（不可变）
            encryptMap = Map.copyOf(loadedMap);

            // 生成解密映射表（反转映射，不可变）
            Map<String, String> reverseMap = new HashMap<>(loadedMap.size());
            for (Map.Entry<String, String> entry : loadedMap.entrySet()) {
                String plaintext = entry.getKey();
                String ciphertext = entry.getValue();

                // 检查是否有重复的密文
                if (reverseMap.containsKey(ciphertext)) {
                    throw new RuntimeException(
                            String.format("密码表中存在重复的密文: %s 对应多个明文: %s 和 %s",
                                    ciphertext, reverseMap.get(ciphertext), plaintext));
                }

                reverseMap.put(ciphertext, plaintext);
            }
            decryptMap = Collections.unmodifiableMap(reverseMap);

            initialized = true;
            log.info("密码表初始化成功，加载 {} 个映射", encryptMap.size());

        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            throw new RuntimeException("密码表初始化失败: " + passwordTablePath, e);
        } finally {
            lock.writeLock().unlock();
        }
    }

    /**
     * 验证密码表格式
     *
     * @param cipherTable 密码表
     */
    private static void validateCipherTable(Map<String, String> cipherTable) {
        for (Map.Entry<String, String> entry : cipherTable.entrySet()) {
            String plaintext = entry.getKey();
            String ciphertext = entry.getValue();

            // 验证明文不为空
            if (plaintext == null || plaintext.isEmpty()) {
                throw new RuntimeException("密码表中存在空的明文键");
            }

            // 验证密文格式：必须是 4 位
            if (ciphertext == null || ciphertext.length() != CIPHER_LENGTH) {
                throw new RuntimeException(
                        String.format("密文格式错误，明文 '%s' 的密文 '%s' 长度不为 %d",
                                plaintext, ciphertext, CIPHER_LENGTH));
            }

            // 验证密文格式：第一位必须是小写字母
            char firstChar = ciphertext.charAt(0);
            if (firstChar < 'a' || firstChar > 'z') {
                throw new RuntimeException(
                        String.format("密文格式错误，明文 '%s' 的密文 '%s' 第一位不是小写字母",
                                plaintext, ciphertext));
            }

            // 验证密文格式：后三位必须是字母或数字
            for (int i = 1; i < CIPHER_LENGTH; i++) {
                char c = ciphertext.charAt(i);
                if (!Character.isLetterOrDigit(c)) {
                    throw new RuntimeException(
                            String.format("密文格式错误，明文 '%s' 的密文 '%s' 第 %d 位不是字母或数字",
                                    plaintext, ciphertext, i + 1));
                }
            }
        }
    }

    /**
     * 检查是否已初始化
     *
     * @throws IllegalStateException 未初始化时抛出
     */
    private static void checkInitialized() {
        if (!initialized) {
            throw new IllegalStateException("密码表未初始化，请先调用 init() 方法");
        }
    }

    /**
     * 加密字符串
     * <p>
     * 遍历输入字符串的每个字符：
     * - 若字符在密码表中存在映射，替换为对应的密文（4位）
     * - 否则保持原字符不变
     * </p>
     *
     * @param plaintext 明文字符串
     * @return 密文字符串
     * @throws IllegalArgumentException 输入为 null 时抛出
     * @throws IllegalStateException    密码表未初始化时抛出
     */
    public static String encrypt(String plaintext) {
        if (plaintext == null) {
            log.warn("明文不能为 null");
            return null;
        }

        checkInitialized();

        if (plaintext.isEmpty()) {
            return plaintext;
        }

        StringBuilder result = new StringBuilder(plaintext.length() * CIPHER_LENGTH);

        // 遍历每个字符
        for (int i = 0; i < plaintext.length(); i++) {
            String character = String.valueOf(plaintext.charAt(i));

            // 查找映射
            String cipher = encryptMap.get(character);

            // 存在映射，使用密文
            // 不存在映射，保持原样
            result.append(Objects.requireNonNullElse(cipher, character));
        }

        return result.toString();
    }

    /**
     * 解密字符串
     * <p>
     * 遍历密文字符串，每 4 位作为一个密文单元：
     * - 若该单元在解密表中存在，替换为对应的明文字符
     * - 否则按单个字符逐个保留（处理未加密的字符）
     * </p>
     *
     * @param ciphertext 密文字符串
     * @return 明文字符串
     * @throws IllegalArgumentException 输入为 null 时抛出
     * @throws IllegalStateException    密码表未初始化时抛出
     */
    public static String decrypt(String ciphertext) {
        if (ciphertext == null) {
            throw new IllegalArgumentException("密文不能为 null");
        }

        checkInitialized();

        if (ciphertext.isEmpty()) {
            return ciphertext;
        }

        StringBuilder result = new StringBuilder(ciphertext.length());
        int i = 0;

        while (i < ciphertext.length()) {
            // 尝试读取 4 位作为一个密文单元
            if (i + CIPHER_LENGTH <= ciphertext.length()) {
                String cipherUnit = ciphertext.substring(i, i + CIPHER_LENGTH);
                String plainChar = decryptMap.get(cipherUnit);

                if (plainChar != null) {
                    // 找到映射，解密成功
                    result.append(plainChar);
                    i += CIPHER_LENGTH;
                    continue;
                }
            }

            // 未找到映射，保留当前字符
            result.append(ciphertext.charAt(i));
            i++;
        }

        return result.toString();
    }

    /**
     * 比较辅助：判断明文对应的密文是否与已有密文相等
     */
    public static boolean equalsCipher(String newPlain, String existingCipher) {
        if (newPlain == null && (existingCipher == null || existingCipher.isEmpty())) return true;
        if (newPlain == null) return false;
        String newCipher = encrypt(newPlain);
        return Objects.equals(newCipher, existingCipher);
    }

    /**
     * 比较辅助：判断明文对应的密文是否发生变化
     */
    public static boolean hasCipherChanged(String newPlain, String existingCipher) {
        return !equalsCipher(newPlain, existingCipher);
    }

    /**
     * 获取加密映射表（只读）
     * 主要用于调试和验证
     *
     * @return 加密映射表的只读副本
     * @throws IllegalStateException 密码表未初始化时抛出
     */
    public static Map<String, String> getEncryptMap() {
        checkInitialized();
        return encryptMap; // 已经是不可变的
    }

    /**
     * 获取解密映射表（只读）
     * 主要用于调试和验证
     *
     * @return 解密映射表的只读副本
     * @throws IllegalStateException 密码表未初始化时抛出
     */
    public static Map<String, String> getDecryptMap() {
        checkInitialized();
        return decryptMap; // 已经是不可变的
    }

    /**
     * 检查是否已初始化
     *
     * @return true 如果已初始化，否则 false
     */
    public static boolean isInitialized() {
        return initialized;
    }

    /**
     * 获取密文固定长度
     *
     * @return 密文长度（固定为 4）
     */
    public static int getCipherLength() {
        return CIPHER_LENGTH;
    }
}


