package com.af.v4.system.common.payment.utils;



import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import javax.crypto.Cipher;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * RSA 加密和解密工具类。
 * 支持 SHA1 和 SHA256 签名算法。
 *
 * @author llzh
 * @since 2024/12/25
 */
public class RSAUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(RSAUtils.class);
    /**
     * 签名算法套件枚举
     */
    public enum SignatureSuite {
        SHA1("SHA1WithRSA"),
        SHA256("SHA256WithRSA");

        private final String suite;

        SignatureSuite(String suite) {
            this.suite = suite;
        }

        public String getValue() {
            return suite;
        }
    }

    private static KeyFactory getKeyFactory() {
        try {
            return KeyFactory.getInstance("RSA");
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("初始化 RSA KeyFactory 失败", e);
        }
    }

    /**
     * 使用私钥对消息签名
     *
     * @param suite        签名算法套件
     * @param message      待签名消息
     * @param privateKeyStr 私钥字符串（Base64 编码）
     * @return 签名结果（字节数组）
     */
    public static byte[] sign(SignatureSuite suite, byte[] message, String privateKeyStr) {
        try {
            // 获取签名实例
            Signature signature = Signature.getInstance(suite.getValue());
            byte[] keyBytes = Base64.decodeBase64(privateKeyStr);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
            PrivateKey privateKey = getKeyFactory().generatePrivate(keySpec);

            signature.initSign(privateKey);
            signature.update(message);

            return signature.sign();
        } catch (Exception e) {
            LOGGER.error("签名生成失败: {}", e.getMessage(), e);
        }
        return null;
    }

    /**
     * 验证签名
     *
     * @param suite        签名算法套件
     * @param message      原始消息
     * @param signatureBase64 签名结果（字节数组）
     * @param publicKeyStr 公钥字符串（Base64 编码）
     * @return 验签结果，true 表示验签成功
     */
    public static boolean verifySign(SignatureSuite suite, String message, String signatureBase64, String publicKeyStr) {
        try {
            // 获取签名实例
            Signature signature = Signature.getInstance(suite.getValue());
            // 解析公钥
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKeyStr));
            PublicKey publicKey = getKeyFactory().generatePublic(keySpec);
            // 初始化验签
            signature.initVerify(publicKey);
            signature.update(message.getBytes(StandardCharsets.UTF_8));

            // 返回验签结果
            return signature.verify(Base64.decodeBase64(signatureBase64));
        } catch (Exception e) {
            LOGGER.error("签名验证失败: {}", e.getMessage(), e);
        }
        return false;
    }

    /**
     * 使用公钥加密数据
     * 加密模式：RSA/ECB/PKCS1Padding
     *
     * @param data          待加密的数据
     * @param publicKeyStr  公钥字符串（Base64 编码）
     * @return 加密后的数据（Base64 编码），加密失败返回 null
     */
    public static String encryptByPublicKey(String data, String publicKeyStr) {
        try {
            // 解析公钥
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKeyStr));
            PublicKey publicKey = getKeyFactory().generatePublic(keySpec);

            // 初始化加密器
            Cipher cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding");
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);

            // 加密数据
            byte[] encryptedData = cipher.doFinal(data.getBytes(StandardCharsets.UTF_8));

            // 返回 Base64 编码的加密数据
            return Base64.encodeBase64String(encryptedData);
        } catch (Exception e) {
            LOGGER.error("RSA公钥加密失败: {}", e.getMessage(), e);
        }
        return null;
    }

    /**
     * 使用私钥解密数据
     * 加密模式：RSA/ECB/PKCS1Padding
     *
     * @param encryptedData 加密的数据（Base64 编码）
     * @param privateKeyStr 私钥字符串（Base64 编码）
     * @return 解密后的数据，解密失败返回 null
     */
    public static String decryptByPrivateKey(String encryptedData, String privateKeyStr) {
        try {
            // 解析私钥
            byte[] keyBytes = Base64.decodeBase64(privateKeyStr);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
            PrivateKey privateKey = getKeyFactory().generatePrivate(keySpec);

            // 初始化解密器
            Cipher cipher = Cipher.getInstance("RSA/ECB/PKCS1Padding");
            cipher.init(Cipher.DECRYPT_MODE, privateKey);

            // Base64 解码并解密
            byte[] decryptedData = cipher.doFinal(Base64.decodeBase64(encryptedData));

            // 返回解密后的字符串
            return new String(decryptedData, StandardCharsets.UTF_8);
        } catch (Exception e) {
            LOGGER.error("RSA私钥解密失败: {}", e.getMessage(), e);
        }
        return null;
    }
}
