package com.af.v4.system.common.payment.factory;

import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.PaymentHandler;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * PaymentHandlerFactory
 * 根据银行名称和对接方式获取具体的 PaymentHandler 实现
 */
@Component
public class PaymentHandlerFactory {
    // 存储银行名称 + 对接方式的处理器映射
    private static final Map<String, PaymentHandler> HANDLER_MAP = new ConcurrentHashMap<>();
    private final ApplicationContext applicationContext;
    private static final Logger LOGGER = LoggerFactory.getLogger(PaymentHandlerFactory.class);

    public PaymentHandlerFactory(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    /**
     * 获取处理器
     *
     * @param bankName        银行名称
     * @param integrationType 对接方式
     * @return 对应的 PaymentHandler 实现
     * @throws RuntimeException 如果找不到处理器
     */
    public static PaymentHandler getHandler(BankName bankName, IntegrationType integrationType) {
        String key = generateKey(bankName, integrationType);
        PaymentHandler handler = HANDLER_MAP.get(key);
        if (handler == null) {
            throw new PaymentException(PaymentStatus.INVALID_CONNECTION_METHOD_OR_BANK_NAME.getMessage(), "{}", PaymentStatus.INVALID_CONNECTION_METHOD_OR_BANK_NAME.getMessage());
        }
        return handler;
    }

    /**
     * 根据银行名称和对接方式生成唯一键
     *
     * @param bankName        银行名称
     * @param integrationType 对接方式
     * @return 唯一键
     */
    private static String generateKey(BankName bankName, IntegrationType integrationType) {
        return bankName + "-" + integrationType.name();
    }

    /**
     * 注册处理器
     */
    @PostConstruct
    private void registerHandlers() {
        Map<String, Object> beansWithAnnotation = applicationContext.getBeansWithAnnotation(PaymentHandlerConfig.class);
        for (Map.Entry<String, Object> entry : beansWithAnnotation.entrySet()) {
            PaymentHandler handler = (PaymentHandler) entry.getValue();
            PaymentHandlerConfig config = handler.getClass().getAnnotation(PaymentHandlerConfig.class);
            String key = generateKey(config.bankName(), config.integrationType());
            if (HANDLER_MAP.containsKey(key)) {
                LOGGER.warn("支付业务处理器重复注册: {}", key);
            }
            LOGGER.debug("注册支付业务处理器[key: {}，handler: {}]", key, handler);
            HANDLER_MAP.put(key, handler);
        }
        LOGGER.debug("所有支付业务处理器注册完成：共{}个", HANDLER_MAP.size());
    }
}
