package com.af.v4.system.common.payment.handler.impl.rcb;


import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.impl.AbstractPaymentHandler;
import com.af.v4.system.common.payment.utils.PaymentUtils;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import com.af.v4.system.common.plugins.other.Base64Tools;
import com.af.v4.system.common.redis.RedisService;
import org.apache.commons.codec.digest.DigestUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;


/**
 * @Author: 小六
 * @Description: 安徽农商银行
 * @Date: 2025-06-28 16:00
 */
@Component
@PaymentHandlerConfig(bankName = BankName.AN_RCB, integrationTypes = {IntegrationType.AGGREGATE, IntegrationType.SCAN_DEVICE})
public class AHRcbPaymentHandlerImpl extends AbstractPaymentHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(AHRcbPaymentHandlerImpl.class);

    private final RedisService redisService;

    private final String SUCCESS_CODE = "00";

    public AHRcbPaymentHandlerImpl(RedisService redisService) {
        this.redisService = redisService;
    }

    /**
     * 构建公共请求参数
     *
     * @param paymentConfig
     * @return
     */
    private JSONObject buildCommonParams(JSONObject paymentConfig) {
        LOGGER.info(">>> 构建公共请求参数");
        try {
            String batchNo = getBatchNo(paymentConfig);
            return new JSONObject()
                    // 商户号
                    .put("merchantNo", paymentConfig.optString("mchId"))
                    // 终端号
                    .put("terminalNo", paymentConfig.optString("posId"))
                    // 批次号
                    .put("batchNo", batchNo)
                    // 系统跟踪号
                    .put("traceNo", redisService.incr(generateTrackingNoKey(paymentConfig), -1, 999999, secondsUntilTomorrow()))
                    // 商户订单号
                    .put("outTradeNo", PaymentUtils.generateSerialNumber())
                    // 随机字符串
                    .put("nonceStr", PaymentUtils.generateOrderId());
        } catch (Exception e) {
            LOGGER.error("构建公共请求参数异常", e);
            throw new PaymentException(PaymentStatus.BUILD_PARAMETER_FAILED);
        }
    }

    /**
     * 构建支付订单请求参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建扫码收款请求参数");
        try {
            JSONObject params = buildCommonParams(paymentConfig)
                    .put("transAmount", request.getAmount())
                    .put("remark", request.getDescription());

            // 有授权码就是被扫，没有就是主扫
            String authCode = request.getAuthCode();
            if (!Objects.isNull(authCode)) {
                params.put("url", "/microPay");
                params.put("payCode", authCode);
            } else {
                params.put("url", "/nativePay");
            }

            return params;
        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR, request.toString());
        }
    }

    /**
     * 处理获取二维码响应结果
     *
     * @param response      响应结果
     * @param request       支付请求
     * @param resultJson    响应结果
     * @param paymentConfig 支付配置
     * @return 响应结果
     */
    @Override
    protected PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request, JSONObject resultJson, JSONObject paymentConfig) {
        LOGGER.info(">>> 处理获取二维码响应结果");
        try {
            if (SUCCESS_CODE.equals(resultJson.optString("resultCode"))) {
                response.setState(PaymentStatus.PAY_SUCCESS);
                response.setMchId(resultJson.optString("merchantNo"));
                String qrUrlBase64 = resultJson.optString("qrUrl");
                String qrUrl = Base64Tools.decode(qrUrlBase64);
                response.setPayUrl(qrUrl);
                response.setCustomParam(new JSONObject().put("payImgUrl", resultJson.optString("qrUrl")).toString());
                response.setNeedQuery(true);
                response.setOrderNo(request.getOrderNo());
                response.setPayResult("交易成功");
            } else {
                LOGGER.info(">>> 获取二维码银行返回失败！");
                response.setState(PaymentStatus.COMMUNICATION_FAIL);
            }
        } catch (Exception e) {
            LOGGER.error("构建支付订单返回参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建订单状态查询请求参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return
     */
    @Override
    protected JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建订单状态查询请求参数");
        try {
            return buildCommonParams(paymentConfig)
                    .put("outTradeNo", request.getOrderNo())
                    .put("url", "/orderQuery");
        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_REQUEST_PARAM_ERROR, request.toString());
        }
    }

    @Override
    protected QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response, QueryPaymentStatusRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理订单查询响应结果");
        try {
            if (SUCCESS_CODE.equals(resultJson.optString("resultCode"))) {
                response.setState(PaymentStatus.PAY_SUCCESS);
                response.setMchId(resultJson.optString("merchantNo"));
                response.setOrderNo(request.getOrderNo());
                response.setTransactionId(resultJson.optString("chnOrderId", resultJson.optString("cposOrderId")));
                response.setAmount(resultJson.optInt("transAmount"));
                String tradeState = resultJson.optString("orderStatus");
                PaymentStatus status = switch (tradeState) {
                    case "3" -> PaymentStatus.SUCCESS_FOR_PAYMENT;
                    case "6", "7" -> PaymentStatus.REFUND_FOR_PAYMENT;
                    case "1", "2", "8" -> PaymentStatus.NOT_FOR_PAYMENT;
                    case "5" -> PaymentStatus.ORDER_BEEN_CLOSED;
                    default -> PaymentStatus.FAIL_FOR_PAYMENT;
                };
                response.setPaymentStatus(status.getMessage());
                response.setState(status);
                response.setPaySuccessDate(
                        DateTools.formatDateTime(
                                resultJson.optString(
                                        "paidTime",
                                        resultJson.optString("paidTime", DateTools.getNow("yyyyMMddHHmmss"))
                                )
                        )
                );
            } else {
                LOGGER.info(">>> 订单状态查询银行返回失败！");
                response.setState(PaymentStatus.PAY_FAIL);
            }
        } catch (Exception e) {
            LOGGER.error(">>> 处理订单状态查询响应结果异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_RESPONSE_PARAM_ERROR, request.toString());
        }
        return response;
    }

    @Override
    protected JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建撤销订单请求参数");
        try {
            return buildCommonParams(paymentConfig)
                    .put("outTradeNo", PaymentUtils.generateOrderId())
                    .put("originalOutTradeNo", request.getOrderNo())
                    .put("url", "/reverse");
        } catch (Exception e) {
            LOGGER.error("构建撤销订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.CANCEL_ORDER_BUILD_REQUEST_PARAM_ERROR, request.toString());
        }
    }

    @Override
    protected CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理撤销订单响应结果");
        try {
            if (SUCCESS_CODE.equals(resultJson.optString("resultCode"))) {
                response.setMchId(resultJson.optString("merchantNo"));
                response.setOrderNo(resultJson.optString("chnOrderId", resultJson.optString("chnOrderId")));
                String refundStatus = resultJson.optString("refundStatus");
                PaymentStatus status = switch (refundStatus) {
                    case "1" -> PaymentStatus.REFUND_PROCESSING;
                    case "2" -> PaymentStatus.CANCEL_ORDER_SUCCESS;
                    default -> PaymentStatus.REFUND_FAIL;
                };
                response.setCancelResult(status.getMessage());
            } else {
                LOGGER.info(">>> 撤销订单银行返回失败！");
                response.setCancelResult(PaymentStatus.CANCEL_ORDER_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error(">>> 处理撤销订单响应结果异常", e);
            throw new PaymentException(PaymentStatus.CANCEL_ORDER_BUILD_RESPONSE_PARAM_ERROR, request.toString());
        }
        return response;
    }

    @Override
    protected JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建订单退款请求参数");
        try {
            return buildCommonParams(paymentConfig)
                    .put("mchtRefundNo", request.getRefundOrderNo())
                    .put("outTradeNo", request.getOrderNo())
                    .put("refundAmount", request.getAmount())
                    .put("url", "/refund");
        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR, request.toString());
        }
    }

    @Override
    protected RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理订单退款响应结果");
        try {
            if (SUCCESS_CODE.equals(resultJson.optString("resultCode"))) {
                response.setMchId(resultJson.optString("merchantNo"));
                response.setOrderNo(request.getOrderNo());
                response.setTransactionId(resultJson.optString("cposOrderId"));
                response.setRefundOrderNo(request.getRefundOrderNo());
                response.setRefundId(resultJson.optString("cposOrderId"));
                response.setRefundAmount(resultJson.optInt("transAmount"));
                response.setRefundResult(PaymentStatus.REFUND_REQUEST_SUCCESS.getMessage());
                response.setState(PaymentStatus.REFUND_REQUEST_SUCCESS);
            } else {
                LOGGER.info(">>> 银行返回订单退款交易失败！");
                response.setState(PaymentStatus.REFUND_REQUEST_FAIL);
                response.setMessage(resultJson.optString("resultMessage", PaymentStatus.REFUND_REQUEST_FAIL.getMessage()));
            }
        } catch (Exception e) {
            LOGGER.error(" 构建订单退款返回参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    @Override
    protected JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建退款订单状态查询请求参数");
        try {
            return buildCommonParams(paymentConfig)
                    .put("outTradeNo", request.getOrderNo())
                    .put("url", "/refundQuery");
        } catch (Exception e) {
            LOGGER.error("构建退款订单状态查村请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_REQUEST_PARAM_ERROR, request.toString());
        }
    }

    @Override
    protected QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response, QueryRefundStatusRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理退款订单状态查询响应结果");
        try {
            if (SUCCESS_CODE.equals(resultJson.optString("resultCode"))) {
                response.setMchId(resultJson.optString("merchantNo"));
                response.setState(PaymentStatus.REFUND_SUCCESS);
                response.setOrderNo(request.getOrderNo());
                response.setTransactionId(resultJson.optString("cposOrderId"));
                response.setRefundAmount(resultJson.optInt("transAmount"));
                response.setRefundDateTime(
                        DateTools.formatDateTime(
                                resultJson.optString(
                                        "paidTime",
                                        resultJson.optString("paidTime", DateTools.getNow("yyyyMMddHHmmss"))
                                )
                        )
                );
            } else {
                LOGGER.info(">>> 退款订单状态查询银行返回失败！");
                response.setState(PaymentStatus.PAY_FAIL);
            }
        } catch (Exception e) {
            LOGGER.error(">>> 处理退款订单状态查询响应结果异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_RESPONSE_PARAM_ERROR, request.toString());
        }
        return response;
    }

    /**
     * 签名请求
     * 该方法会将签名直接放入参数中
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     */
    @Override
    protected void signRequest(JSONObject reqParams, JSONObject paymentConfig) {
        // 过滤后的请求参数
        JSONObject filteredParams = new JSONObject();
        // 需要过滤掉空的参数
        for (String key : reqParams.keySet()) {
            Object value = reqParams.get(key);
            if (value != null && !"".equals(value.toString()) && !"url".equals(key)) {
                filteredParams.put(key, value);
            }
        }

        // 将所有参数的key放进一个list中并进行排序
        List<String> keys = new ArrayList<>(filteredParams.keySet());
        Collections.sort(keys);

        StringBuilder beforeSigning = new StringBuilder();
        // 再遍历key进行参数组装
        for (String key : keys) {
            if (beforeSigning.length() > 0) {
                beforeSigning.append("&");
            }
            beforeSigning.append(key).append("=").append(filteredParams.get(key));
        }
        LOGGER.info(">>> 签名字符串：{}", beforeSigning);

        // 加上密钥
        beforeSigning.append("&key=" + paymentConfig.getString("secretKey"));
        LOGGER.info(">>> 待验签字符串：{}", beforeSigning);

        // 进行md5加密
        String sign = DigestUtils.md5Hex(String.valueOf(beforeSigning)).toUpperCase();
        LOGGER.info(">>> 最终签名串: {}", sign);

        reqParams.put("sign", sign);
    }

    /**
     * 执行支付业务请求
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     * @return 业务执行请求结果
     */
    @Override
    protected JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            JSONObject headers = new JSONObject();
            LOGGER.info("请求支付业务, 请求参数：{}", reqParams);
            String baseUrl = paymentConfig.getString("transactionurl");
            String bizUrl = reqParams.getString("url");
            String url = baseUrl + bizUrl;
            LOGGER.info(">>> 请求地址：{}", url);
            reqParams.remove("url");
            headers.put("Content-Type", "application/xml");
            String response = RestTools.post(
                    url,
                    String.valueOf(reqParams),
                    headers.toString()
            );
            JSONObject jsonObject = JsonTools.convertToJson(response);
            LOGGER.info(">>> 银行响应原始结果：" + jsonObject);
            if ("无效商户".equals(jsonObject.optString("resultMessage"))) {
                LOGGER.info(">>> 无效商户，重新签到");
                signInAndSaveBatchNo(paymentConfig);
            }
            return jsonObject;
        } catch (Exception e) {
            LOGGER.error("请求支付业务接口异常", e);
            throw new PaymentException(
                    PaymentStatus.PAY_ORDER_ERROR.getCode(),
                    reqParams.toString(),
                    PaymentStatus.PAY_ORDER_ERROR.getMessage()
            );
        }
    }

    @Override
    protected boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig) {
        try {
            String sign = response.optString("sign");
            if (sign == null) {
                return false;
            }

            // 取出来后将sign字段去除
            response.remove("sign");

            // 调用签名方法进行签名
            signRequest(response, paymentConfig);
            // 签名后取出新的签名串
            String calculatedSign = response.optString("sign");
            LOGGER.info(">>> 原签名串：{}，新签名串：{}", sign, calculatedSign);
            return sign.equals(calculatedSign);
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    protected JSONObject buildDownloadReconciliationFileRequestParams(DownloadReconciliationFileRequest request, JSONObject paymentConfig) {
        return null;
    }

    @Override
    protected DownloadReconciliationFileResponse processDownloadReconciliationFileResponse(DownloadReconciliationFileResponse response, DownloadReconciliationFileRequest request, JSONObject resultJson) {
        return null;
    }

    /**
     * 各交易终端，每⽇第⼀笔交易时，需要通过签到，获取批次号等信息
     * 每次请求之前都检查 redis 中是否有批次号
     * 如果没有则调用该方法进行签到并将获取到的批次号保存到 redis 中
     *
     * @return
     */
    private String signInAndSaveBatchNo(JSONObject paymentConfig) {
        JSONObject signParams = new JSONObject()
                .put("merchantNo", paymentConfig.optString("mchId"))
                .put("terminalNo", paymentConfig.optString("posId"));
        signRequest(signParams, paymentConfig);
        String url = paymentConfig.getString("transactionurl") + paymentConfig.getString("signUrl");
        LOGGER.info(">>> 签到请求地址：{}，请求参数：{}", url, signParams);
        try {
            String response = RestTools.post(url, signParams);
            LOGGER.info(">>> 签到响应结果：{}", response);

            JSONObject result = JsonTools.convertToJson(response);
            if ("00".equals(result.getString("resultCode"))) {
                String batchNo = result.getString("batchNo");

                // 保存到Redis，0点重置
                String redisKey = generateBatchNoKey(paymentConfig);
                redisService.set(redisKey, batchNo, secondsUntilTomorrow());
                return batchNo;
            } else {
                throw new RuntimeException(">>> 签到失败: " + result.getString("resultMessage"));
            }

        } catch (Exception e) {
            throw new RuntimeException(">>> 签到异常: {}" + e.getMessage());
        }
    }

    /**
     * 获取批次号（优先从Redis获取，没有则签到）
     */
    public String getBatchNo(JSONObject paymentConfig) {
        String redisKey = generateBatchNoKey(paymentConfig);
        try {
            LOGGER.info(">>> 获取 [{}] 批次号:{}", redisKey, redisService.get(redisKey));
            String batchNo = redisService.get(redisKey);
            // 判断Redis中是否有批次号，如果有则直接返回
            if (!Objects.equals(batchNo, "null") || !batchNo.isEmpty()) {
                LOGGER.info(">>> 从Redis获取key为 [{}] 的批次号: {}", redisKey, batchNo);
                return batchNo;
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        // Redis中没有，重新签到
        return signInAndSaveBatchNo(paymentConfig);
    }

    /**
     * 生成 Rediskey
     * 格式：payment_商户号_终端号_日期（payment_商户号_日期）
     *
     * @param paymentConfig
     * @param hasTerminal   是否包含终端号
     * @return
     */
    private String generateKey(JSONObject paymentConfig, boolean hasTerminal) {
        String today = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
        return hasTerminal ?
                String.format("payment_%s_%s_%s", paymentConfig.getString("mchId"), paymentConfig.getString("posId"), today) :
                String.format("payment_%s_%s", paymentConfig.getString("mchId"), today);
    }

    /**
     * 生成获取批次号的Rediskey
     *
     * @param paymentConfig
     * @return
     */
    private String generateBatchNoKey(JSONObject paymentConfig) {
        return generateKey(paymentConfig, true);
    }

    /**
     * 生成获取追踪号的Rediskey
     *
     * @param paymentConfig
     * @return
     */
    private String generateTrackingNoKey(JSONObject paymentConfig) {
        return generateKey(paymentConfig, false);
    }

    /**
     * 返回当前时间到明天 0 点整的秒数（跨时区安全）
     * 例如现在 2025-11-20 18:30:45，则返回 5*3600 + 29*60 + 15 = 19755
     */
    public static long secondsUntilTomorrow() {
        // 当前瞬时时间
        Instant now = Instant.now();
        // 系统默认时区
        ZoneId zone = ZoneId.systemDefault();
        // 当前日期
        LocalDate today = LocalDate.ofInstant(now, zone);
        // 明天 0 点
        LocalDate tomorrow = today.plusDays(1);
        ZonedDateTime tomorrowMidnight = tomorrow.atStartOfDay(zone);
        // 计算秒差
        return tomorrowMidnight.toEpochSecond() - now.getEpochSecond();
    }

}
