package com.af.v4.system.common.payment.handler.impl.psbc;


import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.enums.PsbcStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.impl.AbstractPaymentHandler;
import com.af.v4.system.common.payment.utils.FileUtils;
import com.af.v4.system.common.payment.utils.SignUtils;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import dm.jdbc.util.DateUtil;
import org.apache.hc.client5.http.utils.DateUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.text.SimpleDateFormat;
import java.util.Date;


/**
 * @Author: 张巧敏
 * @Description: 中国邮储扫码设备处理器
 * @Date: 2025-10-20 18:25
 */
@Component
@PaymentHandlerConfig(bankName = BankName.PSBC, integrationTypes = {IntegrationType.AGGREGATE})
public class PsbcAbstractPaymentHandlerImpl extends AbstractPaymentHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(PsbcAbstractPaymentHandlerImpl.class);

    /**
     * 构建支付下单请求参数（创建二维码）
     *
     * @param request       支付下单请求数据传输对象
     * @param paymentConfig 支付配置信息
     * @return 构建后的请求参数 JSONObject
     */
    @Override
    protected JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>>>>>>>开始构建支付下单请求参数（创建二维码）");
        try {
            JSONObject head = getRequestHeard(paymentConfig);
            // 报文id，使用17位时间戳格式
            head.put("msgId", new SimpleDateFormat("yyyyMMddHHmmssSSS").format(new Date()));
            //交易码
            head.put("trancode", "qrcodePayOrder");
            JSONObject body = new JSONObject();
            //App标识
            body.put("appId", paymentConfig.getString("appId"));
//            //缴费项目编号 0-H5支付，1-小程序支付，2-APP支付，3-微信小程序单产品支付，4-支付宝小程序单产品支付，5-微信H5单产品支付。如果该字段为空，则默认为H5支付。
//            body.put("payType", "0");
            //缴费模式，默认为“D”
            body.put("payMode", "D");
            // 缴费项目编号
            body.put("payItemNo", paymentConfig.getString("payItemNo"));
            // 收费单位订单号
            body.put("orderNo", request.getOrderNo());
            // 订单描述
            body.put("orderDesc", request.getDescription());
            //用户id
            body.put("openId", request.getUserInfoCode());
            //用户号
            body.put("userNo", request.getUserInfoCode());

            // 交易金额，单位分
            body.put("oweAmt", request.getAmount());

            String custom = request.getCustomParam();
            JSONObject customParam = custom != null ? new JSONObject(custom) : null;
            //后台通知地址customParam
            body.put("notifyUrl", request.getNotifyUrl() != null ? request.getNotifyUrl() : paymentConfig.optString("notifyUrl"));
            //返回商户地址
            body.put("backUrl", paymentConfig.optString("backUrl"));

            //商户订单生成时间 yyyyMMddHHmmss
            body.put("tradeTime", DateUtils.formatDate(new Date(), "yyyyMMddHHmmss"));
            //接入方式 0-专线接入，1-互联网接入
            body.put("accessMethod", "1");
            //商户是否开启通知接口（非必填）
            body.put("notifyFlag", "");
            //二维码有效期（非必填）
            body.put("payExpire", "");
            //缴费记录链接是否显示 1-显示，0-不显示（非必填）
            body.put("historyFlag", "1");

            //商户号
//            body.put("merchantNo", paymentConfig.optString("merchantNo"));

            //扩展字段1
            body.put("input1", "");
            //扩展字段2
            body.put("input2", "");
            //扩展字段3
            body.put("input3", "");
            //扩展字段4
            body.put("input4", "");
            //扩展字段5
            body.put("input5", "");

            JSONObject reqParams = new JSONObject();
            reqParams.put("head", head);
            reqParams.put("body", body);
            return reqParams;

        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理支付下单响应结果（创建二维码）
     *
     * @param response      原始支付下单响应对象
     * @param request       支付下单请求数据传输对象
     * @param resultJson    接口返回的原始响应 JSON 数据
     * @param paymentConfig 支付配置信息
     * @return 处理后的支付下单响应对象
     */
    @Override
    protected PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request, JSONObject resultJson, JSONObject paymentConfig) {
        LOGGER.info(">>> 处理支付订单响应结果 （创建二维码）> ");
        try {
            // 获取消息头信息
            JSONObject head = resultJson.optJSONObject("head");
            String respCode = head != null ? head.optString("respCode") : resultJson.optString("code");
            String respMsg = head != null ? head.optString("respMsg") : resultJson.optString("message");

            // 记录响应头信息
            if (head != null) {
                String version = head.optString("version");
                String trancode = head.optString("trancode");
                String msgId = head.optString("msgId");
                String channelId = head.optString("channelId");
                LOGGER.info("响应头信息 - 版本号: {}, 交易码: {}, 报文ID: {}, 收费单位代号: {}", version, trancode, msgId, channelId);
            }

            PsbcStatus psbcStatus = PsbcStatus.fromCode(respCode);

            if (psbcStatus != null && psbcStatus.isSuccess()) {
                // 成功响应 - 获取消息体信息
                JSONObject body = resultJson.optJSONObject("body");
                if (body != null) {
                    // 获取二维码地址
                    String qrcode = body.optString("qrcode");
                    if (!qrcode.isEmpty()) {
                        // 设置二维码支付URL
                        response.setPayUrl(qrcode);
                        response.setState(PaymentStatus.PAY_SUCCESS);
                        response.setNeedQuery(false);
                        response.setPayResult("二维码支付订单创建成功");
                        LOGGER.info("二维码支付订单创建成功，二维码地址: {}", qrcode);
                    } else {
                        LOGGER.warn("响应成功但二维码地址为空");
                        response.setState(PaymentStatus.QUERY_ORDER_FAIL);
                        response.setNeedQuery(true);
                        response.setPayResult("二维码地址获取失败");
                    }
                } else {
                    LOGGER.warn("响应消息体为空");
                    response.setState(PaymentStatus.QUERY_ORDER_FAIL);
                    response.setNeedQuery(true);
                    response.setPayResult("响应数据异常");
                }
            } else {
                // 失败响应
                String errorMessage = "支付请求失败";
                if (psbcStatus != null) {
                    errorMessage = psbcStatus.getMessage();
                    LOGGER.warn("支付订单失败，错误码: {}, 错误信息: {}", psbcStatus.getCode(), psbcStatus.getMessage());
                } else {
                    LOGGER.warn("支付订单失败，未知错误码: {}, 响应信息: {}", respCode, respMsg);
                    errorMessage = respMsg.isEmpty() ? "支付请求失败" : respMsg;
                }

                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(errorMessage);
                response.setState(PaymentStatus.PAY_FAIL);
            }
        } catch (Exception e) {
            LOGGER.error("构建支付订单返回参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建查询支付状态请求参数
     *
     * @param request       查询支付状态请求数据传输对象
     * @param paymentConfig 支付配置信息
     * @return 构建后的请求参数 JSONObject
     */
    @Override
    protected JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>>>>>>>开始构建查询支付下单请求参数");
        try {
            JSONObject head = getRequestHeard(paymentConfig);
            // 报文id，和订单号一样
            //交易码
            head.put("trancode", "confirm");
            // 报文id，使用17位时间戳格式
            head.put("msgId", new SimpleDateFormat("yyyyMMddHHmmssSSS").format(new Date()));
            JSONObject body = new JSONObject();

            // 缴费项目编号
            body.put("payItemNo", paymentConfig.getString("payItemNo"));
            // 收费单位订单号
            body.put("orderNo", request.getOrderNo());
            //缴费平台h5支付模式标识 固定填D，标识为D模式
            body.put("h5PayMode", "D");
            //扩展字段1
            body.put("extends1", "");
            //扩展字段2
            body.put("extends2", "");
            //扩展字段3
            body.put("extends3", "");
            //扩展字段4
            body.put("extends4", "");
            // 交易日期（customParams 可能为空，做好空安全）
            JSONObject custom = request.getCustomParams();
            String txDate = DateUtils.formatDate(new Date(), "yyyyMMdd");
            if (custom != null) {
                txDate = custom.optString("txDate", txDate);
            }
            body.put("txDate", txDate);

            JSONObject reqParams = new JSONObject();
            reqParams.put("head", head);
            reqParams.put("body", body);
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建查询订单支付状态请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理查询支付状态响应结果
     *
     * @param response   原始查询支付状态响应对象
     * @param request    查询支付状态请求数据传输对象
     * @param resultJson 接口返回的原始响应 JSON 数据
     * @return 处理后的查询支付状态响应对象
     */
    @Override
    protected QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response, QueryPaymentStatusRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理查询支付状态响应结果 > ");
        try {
            JSONObject head = resultJson.optJSONObject("head");
            String respCode = head != null ? head.optString("respCode") : resultJson.optString("code");
            String respMsg = head != null ? head.optString("respMsg") : resultJson.optString("message");

            PsbcStatus psbcStatus = PsbcStatus.fromCode(respCode);

            if (psbcStatus != null && psbcStatus.isSuccess()) {
                JSONObject body = resultJson.optJSONObject("body");
                if (body != null) {
                    // 基础字段映射
                    String orderNo = body.optString("orderNo");
                    String billNo = body.optString("billNo");
                    int txAmt = body.optInt("txAmt");
                    String txTime = body.optString("txTime");
                    String orderStatus = body.optString("orderStatus");

                    response.setOrderNo(orderNo);
                    response.setTransactionId(billNo);
                    response.setAmount(txAmt);

                    // 状态映射：01-支付中 02-支付成功 03-支付失败 04-退款中 05-退款成功 06-退款失败 07-订单已关闭
                    PaymentStatus status;
                    switch (orderStatus) {
                        case "02":
                            status = PaymentStatus.SUCCESS_FOR_PAYMENT;
                            response.setPaySuccessDate(DateTools.formatDateTime(txTime));
                            break;
                        case "01":
                            status = PaymentStatus.PAYMENT_IN_PROGRESS;
                            break;
                        case "03":
                            status = PaymentStatus.FAIL_FOR_PAYMENT;
                            break;
                        case "04":
                            status = PaymentStatus.REFUND_PROCESSING;
                            break;
                        case "05":
                            status = PaymentStatus.REFUND_SUCCESS;
                            break;
                        case "06":
                            status = PaymentStatus.REFUND_FAIL;
                            break;
                        case "07":
                            status = PaymentStatus.ORDER_BEEN_CLOSED;
                            break;
                        default:
                            status = PaymentStatus.QUERY_ORDER_FAIL;
                            break;
                    }

                    response.setPaymentStatus(status.getMessage());
                    response.setState(status);
                } else {
                    LOGGER.warn("响应消息体为空");
                    response.setState(PaymentStatus.QUERY_ORDER_FAIL);
                    response.setPaymentStatus(PaymentStatus.QUERY_ORDER_FAIL.getMessage());
                }
            } else {
                String errorMessage = respMsg != null && !respMsg.isEmpty() ? respMsg : "查询支付状态失败";
                if (psbcStatus != null) {
                    // 统一使用枚举的描述
                    errorMessage = psbcStatus.getMessage();
                    LOGGER.warn("查询支付状态失败，错误码: {}, 错误信息: {}", psbcStatus.getCode(), psbcStatus.getMessage());
                } else {
                    LOGGER.warn("查询支付状态失败，未知错误码: {}", respCode);
                }

                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(errorMessage);
                response.setState(PaymentStatus.PAY_FAIL);
            }
        } catch (Exception e) {
            LOGGER.error("构建查询支付状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建取消订单请求参数(不支持)
     *
     * @param request       取消支付请求数据传输对象
     * @param paymentConfig 支付配置信息
     * @return 构建后的请求参数 JSONObject
     */
    @Override
    protected JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig) {
        return null;
    }

    /**
     * 处理取消订单响应结果(不支持)
     *
     * @param response   原始取消支付响应对象
     * @param request    取消支付请求数据传输对象
     * @param resultJson 接口返回的原始响应 JSON 数据
     * @return 处理后的取消支付响应对象
     */
    @Override
    protected CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request, JSONObject resultJson) {
        return null;
    }

    /**
     * 构建退款请求参数
     *
     * @param request       退款请求数据传输对象
     * @param paymentConfig 支付配置信息
     * @return 构建后的请求参数 JSONObject
     */
    @Override
    protected JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>>>>>>>开始构建退款请求参数");
        try {
            JSONObject head = getRequestHeard(paymentConfig);
            // 报文id，和订单号一样
            //交易码
            head.put("trancode", "repeatRefund");
            // 报文id，使用17位时间戳格式
            head.put("msgId", new SimpleDateFormat("yyyyMMddHHmmssSSS").format(new Date()));
            JSONObject body = new JSONObject();

            // 缴费项目编号
            body.put("payItemNo", paymentConfig.getString("payItemNo"));
            // 收费单位订单号
            body.put("orderNo", request.getOrderNo());
            // 收费单位退款订单号
            body.put("outerRefundNo", request.getRefundOrderNo());
            // 退款原因
            body.put("refundReason", "需要退款");
            // 退款金额
            body.put("refundAmt", request.getRefundAmount());
            //扩展字段1
            body.put("extends1", "");
            //扩展字段2
            body.put("extends2", "");
            //扩展字段3
            body.put("extends3", "");
            //扩展字段4
            body.put("extends4", "");

            JSONObject reqParams = new JSONObject();
            reqParams.put("head", head);
            reqParams.put("body", body);
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建订单撤销请求参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理退款响应结果
     *
     * @param response   原始退款响应对象
     * @param request    退款请求数据传输对象
     * @param resultJson 接口返回的原始响应 JSON 数据
     * @return 处理后的退款响应对象
     */
    @Override
    protected RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理退款响应结果 > ");
        try {
            // 获取消息头信息
            JSONObject head = resultJson.optJSONObject("head");
            String respCode = head != null ? head.optString("respCode") : resultJson.optString("code");
            String respMsg = head != null ? head.optString("respMsg") : resultJson.optString("message");

            // 记录响应头信息
            if (head != null) {
                String version = head.optString("version");
                String trancode = head.optString("trancode");
                String msgId = head.optString("msgId");
                String channelId = head.optString("channelId");
                LOGGER.info("退款响应头信息 - 版本号: {}, 交易码: {}, 报文ID: {}, 收费单位代号: {}", version, trancode, msgId, channelId);
            }

            PsbcStatus psbcStatus = PsbcStatus.fromCode(respCode);

            if (psbcStatus != null && psbcStatus.isSuccess()) {
                // 成功响应 - 退款响应只有消息头，没有消息体
                // 使用请求中的信息填充响应
                response.setOrderNo(request.getOrderNo());
                response.setRefundOrderNo(request.getRefundOrderNo());
                response.setRefundAmount(request.getRefundAmount());

                // 邮储银行退款成功响应
                response.setState(PaymentStatus.REFUND_SUCCESS);
                response.setRefundResult("退款成功");
                response.setCode(PaymentStatus.REFUND_SUCCESS.getCode());
                response.setMessage(PaymentStatus.REFUND_SUCCESS.getMessage());

                LOGGER.info("退款处理成功，订单号: {}, 退款单号: {}, 退款金额: {}", request.getOrderNo(), request.getRefundOrderNo(), request.getRefundAmount());
            } else {
                // 失败响应
                String errorMessage = "退款请求失败";
                if (psbcStatus != null) {
                    errorMessage = psbcStatus.getMessage();
                    LOGGER.warn("退款失败，错误码: {}, 错误信息: {}", psbcStatus.getCode(), psbcStatus.getMessage());
                } else {
                    LOGGER.warn("退款失败，未知错误码: {}, 响应信息: {}", respCode, respMsg);
                    errorMessage = respMsg != null && !respMsg.isEmpty() ? respMsg : "退款请求失败";
                }

                response.setState(PaymentStatus.REFUND_FAIL);
                response.setRefundResult(errorMessage);
                response.setCode(PaymentStatus.REFUND_FAIL.getCode());
                response.setMessage(errorMessage);
            }
        } catch (Exception e) {
            LOGGER.error("处理退款响应异常", e);
            response.setState(PaymentStatus.REFUND_FAIL);
            response.setRefundResult("退款处理异常");
            response.setCode(PaymentStatus.REFUND_FAIL.getCode());
            response.setMessage("退款处理异常");
        }
        return response;
    }

    /**
     * 构建查询退款状态请求参数
     *
     * @param request       查询退款状态请求数据传输对象
     * @param paymentConfig 支付配置信息
     * @return 构建后的请求参数 JSONObject
     */
    @Override
    protected JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>>>>>>>开始构建查询退款状态请求参数");
        try {
            JSONObject head = getRequestHeard(paymentConfig);
            // 报文id，和订单号一样
            //交易码
            head.put("trancode", "refundQuery");
            // 报文id，使用17位时间戳格式
            head.put("msgId", new SimpleDateFormat("yyyyMMddHHmmssSSS").format(new Date()));
            JSONObject body = new JSONObject();

            // 缴费项目编号
            body.put("payItemNo", paymentConfig.getString("payItemNo"));
            // 收费单位订单号
            body.put("orderNo", request.getOrderNo());
            // 退款单号
            body.put("outerRefundNo", request.getRefundId());
            //扩展字段1
            body.put("extends1", "");
            //扩展字段2
            body.put("extends2", "");
            //扩展字段3
            body.put("extends3", "");
            //扩展字段4
            body.put("extends4", "");
            //交易日期
            JSONObject customParams = request.getCustomParams();
            String txDate = DateUtils.formatDate(new Date(), "yyyyMMdd");
            if (customParams != null) {
                txDate = customParams.optString("txDate", txDate);
            }
            body.put("txDate", txDate);

            JSONObject reqParams = new JSONObject();
            reqParams.put("head", head);
            reqParams.put("body", body);
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建查询退款请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理查询退款状态响应结果
     *
     * @param response   原始查询退款状态响应对象
     * @param request    查询退款状态请求数据传输对象
     * @param resultJson 接口返回的原始响应 JSON 数据
     * @return 处理后的查询退款状态响应对象
     */
    @Override
    protected QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response, QueryRefundStatusRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理查询退款状态响应结果 > ");
        try {
            // 获取消息头信息
            JSONObject head = resultJson.optJSONObject("head");
            String respCode = head != null ? head.optString("respCode") : resultJson.optString("code");
            String respMsg = head != null ? head.optString("respMsg") : resultJson.optString("message");

            PsbcStatus psbcStatus = PsbcStatus.fromCode(respCode);

            if (psbcStatus != null && psbcStatus.isSuccess()) {
                // 成功响应 - 获取消息体信息
                JSONObject body = resultJson.optJSONObject("body");
                if (body != null) {
                    // 基础字段映射
                    String orderNo = body.optString("orderNo");
                    String outerRefundNo = body.optString("outerRefundNo");
                    String refundNo = body.optString("refundNo");
                    Integer refundAmt = body.optInt("refundAmt");
                    String refundTime = body.optString("refundTime");
                    String refundStatus = body.optString("refundStatus");

                    response.setOrderNo(orderNo);
                    response.setRefundOrderNo(outerRefundNo);
                    response.setRefundId(refundNo);
                    response.setRefundAmount(refundAmt);
                    response.setRefundDateTime(refundTime);
                    response.setTransactionId(refundTime);

                    // 根据邮储银行退款状态码设置响应状态
                    PaymentStatus status;
                    switch (refundStatus) {
                        case "040": // 退款成功
                            status = PaymentStatus.REFUND_SUCCESS;
                            response.setRefundResult("退款成功");
                            break;
                        case "041": // 退款中
                            status = PaymentStatus.REFUND_PROCESSING;
                            response.setRefundResult("退款处理中");
                            break;
                        case "042": // 退款失败
                            status = PaymentStatus.REFUND_FAIL;
                            response.setRefundResult("退款失败");
                            break;
                        case "043": // 退款未明
                            status = PaymentStatus.QUERY_REFUND_FAIL;
                            response.setRefundResult("退款状态未明");
                            break;
                        default:
                            status = PaymentStatus.QUERY_REFUND_FAIL;
                            response.setRefundResult("退款状态未知");
                            break;
                    }

                    response.setState(status);
                    response.setCode(status.getCode());
                    response.setMessage(status.getMessage());

                    LOGGER.info("查询退款状态成功，订单号: {}, 退款单号: {}, 状态: {}, 金额: {}", orderNo, outerRefundNo, refundStatus, refundAmt);
                } else {
                    LOGGER.warn("查询退款状态响应消息体为空");
                    response.setState(PaymentStatus.QUERY_REFUND_FAIL);
                    response.setRefundResult("退款状态未知");
                }
            } else {
                // 失败响应
                String errorMessage = "查询退款状态失败";
                if (psbcStatus != null) {
                    errorMessage = psbcStatus.getMessage();
                    LOGGER.warn("查询退款状态失败，错误码: {}, 错误信息: {}", psbcStatus.getCode(), psbcStatus.getMessage());
                } else {
                    LOGGER.warn("查询退款状态失败，未知错误码: {}, 响应信息: {}", respCode, respMsg);
                    errorMessage = respMsg != null && !respMsg.isEmpty() ? respMsg : "查询退款状态失败";
                }

                response.setState(PaymentStatus.QUERY_REFUND_FAIL);
                response.setRefundResult(errorMessage);
                response.setCode(PaymentStatus.QUERY_REFUND_FAIL.getCode());
                response.setMessage(errorMessage);
            }
        } catch (Exception e) {
            LOGGER.error("处理查询退款状态响应异常", e);
            response.setState(PaymentStatus.QUERY_REFUND_FAIL);
            response.setRefundResult("查询异常");
            response.setCode(PaymentStatus.QUERY_REFUND_FAIL.getCode());
            response.setMessage("查询异常");
        }
        return response;
    }


    /**
     * 构建下载对账文件请求参数
     *
     * @param  paymentConfig 支付配置
     * @param request         下载对账文件请求数据传输对象
     * @return 处理后的查询退款状态响应对象
     */
    @Override
    protected JSONObject buildDownloadReconciliationFileRequestParams(DownloadReconciliationFileRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>>>>>>>开始构建下载对账文件请求参数");
        try {
            JSONObject head = getRequestHeard(paymentConfig);
            // 报文id，和订单号一样
            //交易码
            head.put("trancode", "reconcile");
            // 报文id，使用17位时间戳格式
            head.put("msgId", new SimpleDateFormat("yyyyMMddHHmmssSSS").format(new Date()));
            JSONObject body = new JSONObject();

            // 缴费项目编号
            body.put("payItemNo", paymentConfig.getString("payItemNo"));

            //扩展字段1
            body.put("extends1", "");
            //扩展字段2
            body.put("extends2", "");
            //扩展字段3
            body.put("extends3", "");
            //扩展字段4
            body.put("extends4", "");
            //交易日期
            body.put("txnDate",request.getTradeTime());


            JSONObject reqParams = new JSONObject();
            reqParams.put("head", head);
            reqParams.put("body", body);
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建下载对账文件请求参数异常", e);
            throw new PaymentException(PaymentStatus.BUILD_RECONCILE_FILE_DOWNLOAD_REQUEST_FAIL.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理下载对账文件响应
     *
     * @param response       下载对账文件响应数据传输对象
     * @param request         下载对账文件请求数据传输对象
     * @param resultJson      响应结果
     * @return 处理后的下载对账文件响应对象
     */
    @Override
    protected DownloadReconciliationFileResponse processDownloadReconciliationFileResponse(DownloadReconciliationFileResponse response, DownloadReconciliationFileRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理下载对账文件响应结果 > ");
        try {
            // 获取消息头信息
            JSONObject head = resultJson.optJSONObject("head");
            String respCode = head != null ? head.optString("respCode") : resultJson.optString("code");
            String respMsg = head != null ? head.optString("respMsg") : resultJson.optString("message");

            // 记录响应头信息
            if (head != null) {
                String version = head.optString("version");
                String trancode = head.optString("trancode");
                String msgId = head.optString("msgId");
                String channelId = head.optString("channelId");
                LOGGER.info("下载对账文件响应头信息 - 版本号: {}, 交易码: {}, 报文ID: {}, 收费单位代号: {}", version, trancode, msgId, channelId);
            }

            PsbcStatus psbcStatus = PsbcStatus.fromCode(respCode);

            if (psbcStatus != null && psbcStatus.isSuccess()) {
                // 成功响应 - 获取消息体信息
                JSONObject body = resultJson.optJSONObject("body");
                if (body != null) {
                    // 文件名称
                    String fileName = body.optString("fileName");
                    // 文件内容
                    String fileContent = body.optString("fileContent");
                    // 文件下载地址
                    String fileDownloadUrl = request.getDownloadPath();

                    final String zipFilePath = fileDownloadUrl + "\\" + fileName + ".zip";
//                    final String unzipDir = fileDownloadUrl + "\\" + fileName ;
                    FileUtils.decodeFileContent(zipFilePath, fileContent);
//                    FileUtils.unzip(zipFilePath, unzipDir);
                    if (!fileDownloadUrl.isEmpty()) {
                        // 设置文件下载地址
                        response.setFileDownloadUrl(fileDownloadUrl);
                        response.setCode(PaymentStatus.RECONCILE_FILE_SUCCESS.getCode());
                        response.setMessage(PaymentStatus.RECONCILE_FILE_SUCCESS.getMessage());
                        LOGGER.info("对账文件下载成功，文件下载地址: {}，文件名称：{}", fileDownloadUrl, fileName);
                    } else {
                        LOGGER.warn("响应成功但文件地址为空");
                        response.setCode(PaymentStatus.RECONCILE_FILE_DOWNLOAD_URL_ERROR.getCode());
                        response.setMessage(PaymentStatus.RECONCILE_FILE_DOWNLOAD_URL_ERROR.getMessage());
                    }
                } else {
                    LOGGER.warn("响应消息体为空");
                    response.setCode(PaymentStatus.RECONCILE_FILE_FAIL.getCode());
                    response.setMessage("响应数据异常");
                }
            } else {
                // 失败响应
                String errorMessage;
                if (psbcStatus != null) {
                    errorMessage = psbcStatus.getMessage();
                    LOGGER.warn("下载对账文件失败，错误码: {}, 错误信息: {}", psbcStatus.getCode(), psbcStatus.getMessage());
                } else {
                    LOGGER.warn("下载对账文件失败，未知错误码: {}, 响应信息: {}", respCode, respMsg);
                    errorMessage = respMsg != null && !respMsg.isEmpty() ? respMsg : "下载对账文件失败";
                }

                response.setCode(PaymentStatus.RECONCILE_FILE_FAIL.getCode());
                response.setMessage(errorMessage);
            }
        } catch (Exception e) {
            LOGGER.error("处理下载对账文件响应异常", e);
            response.setCode(PaymentStatus.RECONCILE_FILE_FAIL.getCode());
            response.setMessage("处理下载对账文件响应异常");
        }
        return response;
    }


    /**
     * 对请求参数进行签名处理
     *
     * @param reqParams     待签名的请求参数
     * @param paymentConfig 支付配置信息
     */
    @Override
    protected void signRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            // 使用统一的SignUtils进行SM2签名
            JSONObject signedParams = SignUtils.signRequestParamsWithSM2(reqParams, paymentConfig);
            // 将签名后的参数复制回原始参数
            reqParams.clear();
            for (String key : signedParams.keySet()) {
                reqParams.put(key, signedParams.get(key));
            }
        } catch (Exception e) {
            LOGGER.error("签名失败", e);
            throw new PaymentException(
                    PaymentStatus.SIGN_FAIL.getCode(),
                    reqParams.toString(),
                    PaymentStatus.SIGN_FAIL.getMessage()
            );
        }
    }

    /**
     * 执行支付请求并获取响应结果
     *
     * @param reqParams     已签名的请求参数
     * @param paymentConfig 支付配置信息
     * @return 接口返回的原始响应 JSON 数据
     */
    @Override
    protected JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            // 根据不同操作选择不同网关地址，未配置则回退到 transactionurl
            String action = reqParams.optString("trancode", "PAY");
            String urlKey = switch (action) {
                case "QUERY" -> "queryUrl";
                case "REFUND" -> "refundUrl";
                case "REFUND_QUERY" -> "refundQueryUrl";
                default -> "payUrl"; // 下单/支付
            };
            String url = buildRequestUrl(paymentConfig, urlKey);

            // 邮储银行接口要求请求体为cipherInfo字符串，不是JSON格式
            String cipherInfo = reqParams.getString("cipherInfo");
            String requestBody = cipherInfo;

            LOGGER.info("=== 发送请求参数详情 ===");
            LOGGER.info("操作类型: {}", action);
            LOGGER.info("请求URL: {}", url);
            LOGGER.info("发送给网关的参数: {}", reqParams);
            LOGGER.info("发送给网关的JSON字符串: {}", requestBody);

            // 使用自定义HTTP请求方法，完全模拟官方代码的行为
            String response = sendPostLikeOfficial(url, requestBody);

            // 接口返回
            JSONObject jsonObject;
            try {
                // 先尝试解析为JSON
                jsonObject = new JSONObject(response);
            } catch (Exception first) {
                // 如果是PSBC明文“sign|+|cipher”格式，则走解密流程
                if (response.contains("|+|")) {
                    try {
                        jsonObject = com.af.v4.system.common.payment.utils.SignUtils.parsePsbcPlainResponse(response, paymentConfig);
                    } catch (Exception second) {
                        LOGGER.error("PSBC明文响应解析失败: {}", second.getMessage(), second);
                        throw second;
                    }
                } else if (response.contains("invalid IPv6 address")) {
                    LOGGER.error("邮储银行返回错误响应: {}", response);
                    jsonObject = new JSONObject();
                    jsonObject.put("code", "9999");
                    jsonObject.put("message", "邮储银行网关错误: " + response);
                    jsonObject.put("error", response);
                } else {
                    // 否则尝试按XML处理
                    LOGGER.warn("响应不是JSON格式，尝试XML转JSON: {}", first.getMessage());
                    jsonObject = JsonTools.xmlConvertToJson(response);
                }
            }

            LOGGER.info(">>> [{}] 响应结果：{}", action, jsonObject);
            return jsonObject;
        } catch (Exception e) {
            LOGGER.error("请求支付业务接口异常", e);
            throw new PaymentException(
                    PaymentStatus.PAY_ORDER_ERROR.getCode(),
                    reqParams.toString(),
                    PaymentStatus.PAY_ORDER_ERROR.getMessage()
            );
        }
    }

    /**
     * 构建请求URL：支持 base( transactionurl ) + 相对路径 的组合；
     * 若具体URL为绝对地址(http/https)，则直接使用。
     */
    private String buildRequestUrl(JSONObject paymentConfig, String urlKey) {
        // 按文档固定使用 /payment/{channelId} 路径，忽略具体业务相对路径
        String channelId = paymentConfig.getString("channelId");
        String base = paymentConfig.optString("transactionurl", "");
        if (base == null) {
            base = "";
        }
        if (base.endsWith("/")) {
            return base + "payment/" + channelId;
        } else {
            return base + "/payment/" + channelId;
        }
    }


    /**
     * 验证响应数据的签名是否合法
     *
     * @param response      接口返回的原始响应 JSON 数据
     * @param paymentConfig 支付配置信息
     * @return 签名验证是否通过
     */
    @Override
    protected boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig) {
        try {
            // 使用统一的SignUtils进行SM2验签
            return SignUtils.verifySignWithSM2(response, paymentConfig);
        } catch (Exception e) {
            LOGGER.error("验签失败！", e);
            throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), "{}", PaymentStatus.VERIFY_SIGN_FAIL.getMessage());
        }
    }

    /**
     * 获取请求头
     *
     * @param paymentConfig 支付配置
     * @return 请求头
     */
    private JSONObject getRequestHeard(JSONObject paymentConfig) {
        JSONObject reqParams = new JSONObject();
        //收费单位代号
        reqParams.put("channelId", paymentConfig.getString("channelId"));
        reqParams.put("version", "1.0.0");
        return reqParams;
    }

    /**
     * 发送POST请求，完全模拟官方代码的行为
     * 不设置任何Content-Type，让HTTP客户端自动处理
     *
     * @param uri 请求URL
     * @param param 请求参数
     * @return 响应结果
     */
    private String sendPostLikeOfficial(String uri, String param) {
        try {
            // 使用项目中的HttpConnectionPoolUtil，不设置Content-Type
            return com.af.v4.system.common.plugins.http.core.HttpConnectionPoolUtil.request(
                    param,
                    null, // 不设置headers
                    null, // 不设置config
                    false, // 不是标准响应
                    new org.apache.hc.client5.http.classic.methods.HttpPost(uri)
            );
        } catch (Exception e) {
            LOGGER.error("发送HTTP请求失败", e);
            throw new PaymentException("发送HTTP请求失败", e.getMessage());
        }
    }


}
