package com.af.v4.system.common.payment.utils;



import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * RSA 加密和解密工具类。
 * 支持 SHA1 和 SHA256 签名算法。
 *
 * @author llzh
 * @since 2024/12/25
 */
public class RSAUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(RSAUtils.class);
    /**
     * 签名算法套件枚举
     */
    public enum SignatureSuite {
        SHA1("SHA1WithRSA"),
        SHA256("SHA256WithRSA");

        private final String suite;

        SignatureSuite(String suite) {
            this.suite = suite;
        }

        public String getValue() {
            return suite;
        }
    }

    private static KeyFactory getKeyFactory() {
        try {
            return KeyFactory.getInstance("RSA");
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("初始化 RSA KeyFactory 失败", e);
        }
    }

    /**
     * 使用私钥对消息签名
     *
     * @param suite        签名算法套件
     * @param message      待签名消息
     * @param privateKeyStr 私钥字符串（Base64 编码）
     * @return 签名结果（字节数组）
     */
    public static byte[] sign(SignatureSuite suite, byte[] message, String privateKeyStr) {
        try {
            // 获取签名实例
            Signature signature = Signature.getInstance(suite.getValue());
            byte[] keyBytes = Base64.decodeBase64(privateKeyStr);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
            PrivateKey privateKey = getKeyFactory().generatePrivate(keySpec);

            signature.initSign(privateKey);
            signature.update(message);

            return signature.sign();
        } catch (Exception e) {
            LOGGER.error(STR."签名生成失败: \{e.getMessage()}", e);
        }
        return null;
    }

    /**
     * 验证签名
     *
     * @param suite        签名算法套件
     * @param message      原始消息
     * @param signatureBase64 签名结果（Base64 编码）
     * @param publicKeyStr 公钥字符串（Base64 编码）
     * @return 验签结果，true 表示验签成功
     */
    public static boolean verifySign(SignatureSuite suite, String message, String signatureBase64, String publicKeyStr) {
        try {
            // 获取签名实例
            Signature signature = Signature.getInstance(suite.getValue());
            // 解析公钥
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKeyStr));
            PublicKey publicKey = getKeyFactory().generatePublic(keySpec);
            // 初始化验签
            signature.initVerify(publicKey);
            signature.update(message.getBytes(StandardCharsets.UTF_8));

            // 返回验签结果
            return signature.verify(Base64.decodeBase64(signatureBase64));
        } catch (Exception e) {
            LOGGER.error("签名验证失败: {}", e.getMessage(), e);
        }
        return false;
    }
}
