package com.af.v4.system.common.log.service;

import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.proxy.liuli.ILiuLiMonitorServiceProxy;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.core.utils.ip.IpUtils;
import com.af.v4.system.common.log.enums.BusinessStatus;
import com.af.v4.system.common.security.utils.SecurityUtils;
import jakarta.servlet.http.HttpServletRequest;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import java.util.function.Consumer;

@Component
public class LogResolveService {

    private final ApplicationService applicationService;

    private final AsyncLogService asyncLogService;

    private final ILiuLiMonitorServiceProxy liuLiMonitorService;



    public LogResolveService(ApplicationService applicationService, AsyncLogService logService, ILiuLiMonitorServiceProxy liuLiMonitorService) {
        this.applicationService = applicationService;
        this.asyncLogService = logService;
        this.liuLiMonitorService = liuLiMonitorService;
    }

    public void resolveLog(String className, String methodName, String businessType, String title,
                           String operatorType, String operParams, long costTime, Throwable e) {
        resolveLog(className, methodName, businessType, title, operatorType, operParams, costTime, e, null);
    }
    public void resolveLog(String className, String methodName, String businessType, String title,
                           String operatorType, String operParams, long costTime, Throwable e, Consumer<JSONObject> appendOperLogConsumer) {
        // 本地环境不存储操作日志
        if (applicationService.getEnvType() == EnvType.LOCAL) {
            return;
        }
        LoginUser loginUser = SecurityUtils.getLoginUser();
        if (loginUser == null && e == null) {
            // 如果没有登录，且没有发生异常时，不存储操作日志
            return;
        }
        JSONObject operLog = new JSONObject();
        operLog.put("f_oper_ip", IpUtils.getIpAddr());
        if (e == null) {
            operLog.put("f_status", BusinessStatus.SUCCESS.ordinal());
        } else {
            operLog.put("f_status", BusinessStatus.FAIL.ordinal());
            if (e instanceof LogicException logicException) {
                operLog.put("f_method", logicException.getLogicPath());
                operLog.put("f_title", "Logic异常:" + logicException.getTitle());
                operLog.put("f_oper_param", logicException.getLogicParams());
                operLog.put("f_error_msg", logicException.getStack());
            } else {
                operLog.put("f_error_msg", e.getMessage());
            }
        }
        if (loginUser != null) {
            String username = loginUser.getUsername();
            if (StringUtils.isNotBlank(username)) {
                operLog.put("f_oper_name", username);
            }
        }
        if (!operLog.has("f_method")) {
            // 设置方法名称
            operLog.put("f_method", className + "." + methodName + "()");
        }
        HttpServletRequest request = ServletUtils.getRequest();
        if (request != null) {
            // 请求地址
            operLog.put("f_oper_url", StringUtils.substring(request.getRequestURI(), 0, 255));
            // 请求方式
            operLog.put("f_request_method", request.getMethod());
        }
        // 设置action动作
        if (!operLog.has("f_business_type")) {
            operLog.put("f_business_type", businessType);
        }
        // 设置标题
        if (!operLog.has("f_title")) {
            operLog.put("f_title", title);
        }
        // 设置操作人类别
        if (!operLog.has("f_operator_type")) {
            operLog.put("f_operator_type", operatorType);
        }
        // 设置操作参数
        if (!operLog.has("f_oper_param") && operParams != null) {
            operLog.put("f_oper_param", operParams);
        }
        // 设置消耗时间
        operLog.put("f_cost_time", costTime);
        // 追加信息
        if (appendOperLogConsumer != null) {
            appendOperLogConsumer.accept(operLog);
        }
        // 处理关键字
        if (operLog.has("f_error_msg")) {
            operLog.put("f_error_msg", StringUtils.substring(operLog.getString("f_error_msg").replace("'", "''"), 0, 8000));
        }
        if (operLog.has("f_oper_param")) {
            operLog.put("f_oper_param", StringUtils.substring(operLog.getString("f_oper_param").replace("'", "''"), 0, 8000));
        }
        if (operLog.has("f_title")) {
            operLog.put("f_title", StringUtils.substring(operLog.getString("f_title").replace("'", "''"), 0, 300));
        }
        if (operLog.has("f_json_result")) {
            operLog.put("f_json_result", StringUtils.substring(operLog.getString("f_json_result").replace("'", "''"), 0, 2000));
        }
        // 保存数据库
        asyncLogService.saveSysLog(operLog.toString());
        // 避免uploadErrorLog报错导致循环调用
        if (e instanceof LogicException ex && ex.getLogicName().equals("uploadErrorLog")) {
            return;
        }
        if (e != null) {
            liuLiMonitorService.uploadLogicErrorLog(operLog);
        }
    }
}
