package com.af.v4.system.common.jpa.utils;

import cn.hutool.core.util.StrUtil;
import com.af.v4.system.common.core.proxy.jpa.IConditionProxy;
import com.af.v4.system.common.core.proxy.jpa.IQueryParamsProxy;
import com.af.v4.system.common.plugins.core.CommonTools;
import org.json.JSONArray;

import java.util.stream.Stream;

/**
 * 查询表达式生成器，用于以链式、可读的方式构建 JPQL/SQL 条件片段。
 *
 * <p>职责</p>
 * - 累积 where/order/group 等表达式。
 * - 维护命名参数，避免直接拼接字面量导致注入风险。
 * - 同一字段重复参与条件时自动避免参数名冲突（追加序号后缀）。
 *
 * <p>特性</p>
 * - 支持：=、!=、>、>=、<、<=、IS NULL、IS NOT NULL、BETWEEN、LIKE、IN、NOT IN、AND/OR 组合、ORDER BY、GROUP BY、DISTINCT。
 * - eq/neq 在值为 null 或字符串 "NULL" 时自动转换为 IS NULL/IS NOT NULL。
 * - 可通过 {@link #clone()} 派生副本，互不影响。
 *
 * <p>用法示例</p>
 * <pre>
 * Condition c = Condition.build()
 *     .eq("u.status", "enabled").and()
 *     .like("u.name", "张").or(
 *         Condition.build().between("u.age", 18, 30)
 *     )
 *     .orderBy("u.createdAt");
 * String where = c.getValue();
 * QueryParams.Builder params = c.getQueryParamsBuilder();
 * </pre>
 */
public class Condition implements IConditionProxy, Cloneable {
    /**
     * 条件字串累积器，仅拼接安全的关键字/占位符。
     */
    private StringBuilder conditionValue = new StringBuilder();
    /**
     * 命名参数构建器：键为占位符名称，值为实际绑定对象。
     */
    private QueryParams.Builder queryParamsBuilder = new QueryParams.Builder();

    private Condition() {

    }

    public static Condition build() {
        return new Condition();
    }

    /**
     * 统一追加条件：根据操作符决定是否生成参数并向字串累加。
     * - 需要参数的类型：生成唯一参数名并加入 {@link #queryParamsBuilder}，表达式内使用 :param。
     * - 无需参数的类型（IS NULL/IS NOT NULL）：直接追加关键字。
     *
     * @param queryType 操作符
     * @param column 列名（建议传入已带别名的安全标识）
     * @param value 参数值
     * @return this（链式）
     */
    private Condition appendCondition(QueryType queryType, String column, Object value) {
        String paramName = column;
        // IS_NULL 和 IS_NOT_NULL 不需要绑定参数
        if (queryType != QueryType.IS_NULL && queryType != QueryType.IS_NOT_NULL) {
            // 检查是否已有相同字段的参数，添加下标后缀
            int index = 1;
            while (queryParamsBuilder.hasParameter(paramName)) {
                paramName = column + " " + index;
                index++;
            }
            queryParamsBuilder.addParameter(paramName, value);
            conditionValue.append(column)
                    .append(" ")
                    .append(queryType.value)
                    .append(" :")
                    .append(paramName);
        } else {
            conditionValue.append(column)
                    .append(" ")
                    .append(queryType.value);
        }
        return this;
    }

    public Condition and() {
        conditionValue.append(" AND ");
        return this;
    }

    public Condition and(IConditionProxy condition) {
        conditionValue.append(" AND (").append(condition.getValue()).append(")");
        return this;
    }

    public Condition or() {
        conditionValue.append(" OR ");
        return this;
    }

    public Condition or(IConditionProxy condition) {
        conditionValue.append(" OR (").append(condition.getValue()).append(")");
        return this;
    }

    public Condition eq(String column, Object value) {
        QueryType queryType;
        if (StrUtil.equalsIgnoreCase(value == null ? "NULL" : value.toString(), "NULL")) {
            queryType = QueryType.IS_NULL;
        } else {
            queryType = QueryType.EQUALS;
        }
        return appendCondition(queryType, column, value);
    }

    public Condition neq(String column, Object value) {
        QueryType queryType;
        if (StrUtil.equalsIgnoreCase(value == null ? "NULL" : value.toString(), "NULL")) {
            queryType = QueryType.IS_NOT_NULL;
        } else {
            queryType = QueryType.NOT_EQUALS;
        }
        return appendCondition(queryType, column, value);
    }

    public Condition gt(String column, Object value) {
        return appendCondition(QueryType.GREATER_THEN, column, value);
    }

    public Condition gtEquals(String column, Object value) {
        return appendCondition(QueryType.GREATER_EQUALS_THEN, column, value);
    }

    public Condition lt(String column, Object value) {
        return appendCondition(QueryType.LESS_THEN, column, value);
    }

    public Condition ltEquals(String column, Object value) {
        return appendCondition(QueryType.LESS_EQUALS_THEN, column, value);
    }

    public Condition between(String column, Object beginValue, Object endValue) {
        String betweenBeginKey = column + "Begin";
        String betweenEndKey = column + "End";
        queryParamsBuilder.addParameter(betweenBeginKey, beginValue);
        queryParamsBuilder.addParameter(betweenEndKey, endValue);
        conditionValue.append(column)
                .append(" BETWEEN :")
                .append(betweenBeginKey)
                .append(" AND :")
                .append(betweenEndKey);
        return this;
    }

    public Condition between(String column, JSONArray values) {
        return between(column, values.get(0), values.get(1));
    }

    public Condition like(String column, Object value) {
        queryParamsBuilder.addParameter(column, "%" + value + "%");
        conditionValue.append(column)
                .append(" LIKE :")
                .append(column);
        return this;
    }

    public Condition rightLike(String column, Object value) {
        queryParamsBuilder.addParameter(column, value + "%");
        conditionValue.append(column)
                .append(" LIKE :")
                .append(column);
        return this;
    }

    public Condition leftLike(String column, Object value) {
        queryParamsBuilder.addParameter(column, "%" + value);
        conditionValue.append(column)
                .append(" LIKE :")
                .append(column);
        return this;
    }

    public Condition notIn(String column, String... values) {
        return notIn(column, new JSONArray(values));
    }

    public Condition notIn(String column, JSONArray values) {
        conditionValue.append(column).append(" NOT IN (");
        for (Object value : values) {
            conditionValue.append("'").append(value).append("',");
        }
        conditionValue.deleteCharAt(conditionValue.length() - 1).append(")");
        return this;
    }

    public Condition in(String column, String... values) {
        return in(column, new JSONArray(values));
    }

    public Condition in(String column, JSONArray values) {
        conditionValue.append(column).append(" IN (");
        String conditionSplit = CommonTools.union(values);
        conditionValue.append(conditionSplit).append(")");
        return this;
    }

    public Condition orderBy(String column) {
        conditionValue.append(" ORDER BY ").append(column);
        return this;
    }

    /**
     * 添加GROUP BY子句
     *
     * @param column 要分组的列名
     * @return 当前条件对象
     */
    public Condition groupBy(String column) {
        conditionValue.append(" GROUP BY ").append(column);
        return this;
    }

    /**
     * 添加多个列的GROUP BY子句
     *
     * @param columns 要分组的列名数组
     * @return 当前条件对象
     */
    public Condition groupBy(String... columns) {
        conditionValue.append(" GROUP BY ");
        for (int i = 0; i < columns.length; i++) {
            conditionValue.append(columns[i]);
            if (i < columns.length - 1) {
                conditionValue.append(", ");
            }
        }
        return this;
    }

    /**
     * 添加DISTINCT关键字
     *
     * @return 当前条件对象
     */
    public Condition distinct() {
        // 假设条件的开头位置添加DISTINCT
        // 如果已经有SELECT语句，则在SELECT后添加DISTINCT
        if (conditionValue.indexOf("SELECT") == 0) {
            conditionValue.insert(6, " DISTINCT");
        } else {
            // 否则添加DISTINCT前缀，实际使用时可能需要根据具体SQL构建逻辑调整
            conditionValue.insert(0, "DISTINCT ");
        }
        return this;
    }

    public String getValue() {
        return conditionValue.toString();
    }

    public QueryParams.Builder getQueryParamsBuilder() {
        return queryParamsBuilder;
    }

    @Override
    public Condition clone() {
        try {
            Condition cloned = (Condition) super.clone();
            // 创建一个新的 StringBuilder 并复制原始内容
            cloned.conditionValue = new StringBuilder(this.conditionValue.toString());
            // 创建一个新的 QueryParamsBuilder 并复制原始内容
            IQueryParamsProxy queryParams = this.queryParamsBuilder.build();
            cloned.queryParamsBuilder = new QueryParams.Builder().parameterList(queryParams.getParameterList());
            return cloned;
        } catch (CloneNotSupportedException e) {
            // 这不应该发生，因为我们实现了 Cloneable
            throw new RuntimeException("克隆 Condition 失败", e);
        }
    }

    public enum QueryType {
        EQUALS("="),
        NOT_EQUALS("!="),
        IS_NULL("IS NULL"),
        IS_NOT_NULL("IS NOT NULL"),
        GREATER_THEN(">"),
        GREATER_EQUALS_THEN(">="),
        LESS_THEN("<"),
        LESS_EQUALS_THEN("<=");

        private final String value;

        QueryType(String value) {
            this.value = value;
        }

        public static QueryType toType(String value) {
            return Stream.of(QueryType.values())
                    .filter(p -> p.value.equals(value))
                    .findAny()
                    .orElse(null);
        }

        public static boolean is(String value) {
            return toType(value) != null;
        }

        public String getValue() {
            return value;
        }
    }

    public enum WhereType {
        AND("AND"),
        OR("OR");

        private final String value;

        WhereType(String value) {
            this.value = value;
        }

        public static WhereType toType(String value) {
            return Stream.of(WhereType.values())
                    .filter(p -> p.value.equals(value))
                    .findAny()
                    .orElse(null);
        }

        public static boolean is(String value) {
            return toType(value) != null;
        }

        public String getValue() {
            return value;
        }
    }
}
