package com.af.v4.system.common.jpa.sensitive;

import com.af.v4.system.common.plugins.security.SubstitutionCipherUtil;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.Set;

/**
 * @description: 加密字段对应目标jsonobject 加密
 * @author: 唐梓烨
 * @time: 2025/11/14 16:18
 */
@Component
@RefreshScope
@ConfigurationProperties(prefix = "security.sensitive")
public class ConditionalEncryption {
    private static final Logger log = LoggerFactory.getLogger(ConditionalEncryption.class);

    /**
     * 是否启用敏感数据处理功能（来自配置 security.sensitive.enabled）
     */
    private boolean enabled = true;

    public boolean isEnabled() {
        return enabled;
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }


    /**
     * 处理单个字段错误
     */
    private void handleFieldError(String fieldName, Exception e) {
        log.warn("{} 字段处理出错: {} - {}", "字段加密回写", fieldName, e.getMessage());
        if (log.isDebugEnabled()) {
            log.debug("{} 字段处理详细错误信息:", "字段加密回写", e);
        }
    }
    /**
     * 基于条件配置进行加密处理
     * @param conditionConfig 条件配置数组
     * @param data 需要处理的数据对象
     * @return 处理后的数据对象（包含加密数据）
     */
    public JSONObject processConditionalEncryption(JSONArray conditionConfig, JSONObject data) {
        if (!this.enabled || conditionConfig == null || data == null) {
            return data;
        }

        // 获取需要加密的字段集合
        Set<String> encryptionFields = extractEncryptionFields(conditionConfig);

        if (encryptionFields.isEmpty()) {
            log.debug("没有找到需要加密的字段配置");
            return data;
        }

        log.debug("基于条件配置进行加密，字段数: {}", encryptionFields.size());

        // 创建数据副本进行处理
        JSONObject result = new JSONObject(data.toString());

        // 直接复用现有的processSingleField加密逻辑
        encryptionFields.stream()
                .filter(fieldName -> result.opt(fieldName) != null)
                .forEach(fieldName -> processSingleField(fieldName, result));

        return result;
    }

    /**
     * 从条件配置中提取需要加密的字段
     */
    private Set<String> extractEncryptionFields(JSONArray conditionConfig) {
        Set<String> encryptionFields = new HashSet<>();

        for (int i = 0; i < conditionConfig.length(); i++) {
            JSONObject configItem = conditionConfig.getJSONObject(i);

            // 检查是否包含conditionDESENS且为true
            if (configItem.has("conditionDESENS") &&
                    configItem.getBoolean("conditionDESENS") &&
                    configItem.has("model")) {

                String model = configItem.getString("model");
                encryptionFields.add(model);
                log.debug("找到条件加密字段: {} -> {}",
                        configItem.optString("name", "未知"), model);
            }
        }

        return encryptionFields;
    }


    /**
     * 处理单个敏感字段
     *
     * @param fieldName 字段名
     * @param row       数据行
     */
    private void processSingleField(String fieldName, JSONObject row) {
        Object fieldValue = row.opt(fieldName);

        if (fieldValue == null || row.isNull(fieldName)) {
            log.debug("字段 {} 值为 null，跳过处理", fieldName);
            return;
        }

        // 处理不同类型
        if (fieldValue instanceof String) {
            processStringField(fieldName, (String) fieldValue, row);
        } else if (fieldValue instanceof JSONArray) {
            processArrayField(fieldName, (JSONArray) fieldValue, row);
        } else if (fieldValue instanceof Boolean) {
            log.debug("字段 {} 为布尔类型 [{}]，跳过加密", fieldName, fieldValue);
        } else if (fieldValue instanceof Number) {
            log.debug("字段 {} 为数值类型 [{}]，跳过加密", fieldName, fieldValue);
        } else if (fieldValue instanceof JSONObject) {
            log.debug("字段 {} 为对象类型，跳过加密", fieldName);
        } else {
            log.debug("字段 {} 为未知类型 [{}]，跳过加密",
                    fieldName, fieldValue.getClass().getSimpleName());
        }
    }

    private void processStringField(String fieldName, String value, JSONObject row) {
        if (value.isBlank()) {
            log.debug("字段 {} 值为空字符串，跳过处理", fieldName);
            return;
        }

        try {
            String cipher = SubstitutionCipherUtil.encrypt(value);
            if (cipher != null && !cipher.equals(value)) {
                row.put(fieldName, cipher);
                log.debug("敏感字段已加密写回: {} (版本: {}, 原文长度: {}, 密文长度: {})",
                        fieldName,
                        SubstitutionCipherUtil.getCurrentVersion(),
                        value.length(),
                        cipher.length());
            } else {
                log.warn("字段 {} 加密失败或加密前后无变化", fieldName);
            }
        } catch (Exception ex) {
            handleFieldError(fieldName, ex);
        }
    }

    private void processArrayField(String fieldName, JSONArray array, JSONObject row) {
        boolean modified = false;

        for (int i = 0; i < array.length(); i++) {
            Object element = array.opt(i);
            if (element instanceof String strElement) {
                if (!strElement.isBlank()) {
                    try {
                        String cipher = SubstitutionCipherUtil.encrypt(strElement);
                        if (cipher != null && !cipher.equals(strElement)) {
                            array.put(i, cipher);
                            modified = true;
                            log.debug("数组字段 {}[{}] 已加密 (原文长度: {}, 密文长度: {})",
                                    fieldName, i, strElement.length(), cipher.length());
                        }
                    } catch (Exception ex) {
                        log.error("加密数组字段 {}[{}] 时出错: {}", fieldName, i, ex.getMessage());
                    }
                }
            }
        }

        if (modified) {
            row.put(fieldName, array);
            log.info("数组字段 {} 加密完成，共处理 {} 个元素", fieldName, array.length());
        }
    }
}
