package com.af.v4.system.common.jpa.utils;

import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.security.utils.SecurityUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Component
public class DesensitizationTool {
    private static final Logger LOGGER = LoggerFactory.getLogger(DesensitizationTool.class);

    private final LiuLiConfigService liuLiConfigService;

    public DesensitizationTool(LiuLiConfigService liuLiConfigService) {
        this.liuLiConfigService = liuLiConfigService;
    }

    /**
     * 对查询结果集进行脱敏处理
     *
     * @param jsonArray 需要脱敏的结果集
     * @return 脱敏后的结果集
     */
    public JSONArray processEncryption(JSONArray jsonArray) {
        JSONArray result = new JSONArray();
        JSONObject desensitizationConfig = liuLiConfigService.get("desensitizationConfig");
        // 从琉璃配置获取是否需要脱敏
        boolean isDesensitization = desensitizationConfig.optBoolean("isDesensitization", false);
        // 从琉璃配置获取要脱敏的查询列
        JSONArray field = desensitizationConfig.optJSONArray("field", new JSONArray());
        // 从琉璃配置获取不用脱敏信息账号id集合
        JSONArray userId = desensitizationConfig.optJSONArray("userId", new JSONArray());
        boolean isDesensitized = containsString(userId, String.valueOf(SecurityUtils.getLoginUser().getUserid()));
        LOGGER.info("SecurityUtils.getLoginUser()" + SecurityUtils.getLoginUser().getUserid());
        LOGGER.info("SecurityUtils.getLoginUser()" + SecurityUtils.getLoginUser().getUserid());
        LOGGER.info("isDesensitized" + isDesensitized + "===========>" + desensitizationConfig);
        if (isDesensitization) {
            for (int z = 0; z < jsonArray.length(); z++) {
                JSONObject json = jsonArray.getJSONObject(z);
                LOGGER.info("json1------->" + json);
                for (String key : json.keySet()) {
                    if (!isDesensitized) {
                        if (!field.isEmpty()) {
                            for (int i = 0; i < field.length(); i++) {
                                String rowKey = field.getString(i);
                                LOGGER.info("rowKey------->" + rowKey);
                                if (key.equals(rowKey) && !key.equals("f_user_phone")) {
                                    String row = json.optString(rowKey);
                                    json.put(rowKey, maskUserName(row));
                                } else if (key.equals(rowKey)) {
                                    String row = json.optString(rowKey);
                                    json.put(rowKey, maskPhoneNumber(row));
                                }
                            }
                        }
                    }
                }
                LOGGER.info("json2------->" + json);
                result.put(json);
            }
            LOGGER.info("result------->" + result);
            return result;
        } else {
            LOGGER.info("result=======>" + result);
            return jsonArray;
        }
    }


    /**
     * 对用户姓名进行脱敏处理
     *
     * @param name 原始姓名
     * @return 脱敏后的姓名
     */
    private String maskUserName(String name) {
        if (name == null || name.isEmpty()) {
            return name;
        }

        int length = name.length();
        StringBuilder masked = new StringBuilder();

        switch (length) {
            case 1:
                // 单字符直接返回
                return name;
            case 2:
                // 两位：把姓改成*
                masked.append("*").append(name.charAt(1));
                break;
            case 3:
                // 三位：把中间改成*
                masked.append(name.charAt(0)).append("*").append(name.charAt(2));
                break;
            case 4:
                // 四位：把中间两位改成*
                masked.append(name.charAt(0)).append("**").append(name.charAt(3));
                break;
            case 5:
                // 五位：把中间三位改成*
                masked.append(name.charAt(0)).append("***").append(name.charAt(4));
                break;
            case 6:
                // 六位：把中间四位改成*
                masked.append(name.charAt(0)).append("****").append(name.charAt(5));
                break;
            case 7:
                // 七位：把中间五位改成*
                masked.append(name.charAt(0)).append("*****").append(name.charAt(6));
                break;
            case 8:
                // 八位：把中间六位改成*
                masked.append(name.charAt(0)).append("******").append(name.charAt(7));
                break;
            default:
                // 超过8位的情况，保留首尾，中间用*代替
                masked.append(name.charAt(0));
                for (int i = 1; i < length - 1; i++) {
                    masked.append("*");
                }
                masked.append(name.charAt(length - 1));
                break;
        }

        return masked.toString();
    }

    /**
     * 验证电话号码（支持手机号和座机号）
     * @param phone 待验证的电话号码
     * @return 符合规则返回 true，否则返回 false
     */
    private boolean validatePhoneNumber(String phone) {
        if (phone == null || phone.trim().isEmpty()) {
            return false;
        }

        // 移除所有非数字字符（用于快速判断手机号）
        String pureNum = phone.replaceAll("[^0-9]", "");

        // 验证手机号：11位数字，1开头，第二位3-9
        if (pureNum.matches("^1[3-9]\\d{9}$")) {
            return true;
        }

        // 验证座机号（支持带分隔符和纯数字）
        // 格式：0 + 3-4位区号 + (可选分隔符) + 7-8位本地号
        Pattern landlinePattern = Pattern.compile("^0\\d{2,3}([\\-\\s]?)\\d{7,8}$");
        Matcher landlineMatcher = landlinePattern.matcher(phone);
        if (landlineMatcher.matches()) {
            return true;
        }

        // 特殊情况：纯数字座机号（无分隔符）
        if (pureNum.length() >= 10 && pureNum.length() <= 11
                && pureNum.startsWith("0")
                && pureNum.substring(1).matches("\\d{9,10}")) {
            return true;
        }

        return false;
    }

    /**
     * 电话号码脱敏处理（兼容手机号和座机号）
     * @param phone 原始电话号码
     * @return 脱敏后的电话号码，若无效则返回原字符串
     */
    private String maskPhoneNumber(String phone) {
        if (phone == null || phone.trim().isEmpty() || !validatePhoneNumber(phone)) {
            return phone; // 无效号码不脱敏
        }

        String pureNum = phone.replaceAll("[^0-9]", "");

        // 手机号脱敏：前3位 + **** + 后4位
        if (pureNum.matches("^1[3-9]\\d{9}$")) {
            return pureNum.substring(0, 3) + "****" + pureNum.substring(7);
        }

        // 座机号脱敏：保留完整区号 + 分隔符 + **** + 后4位
        Pattern landlinePattern = Pattern.compile("(0\\d{2,3})([\\-\\s]?)(\\d{7,8})");
        Matcher matcher = landlinePattern.matcher(phone);
        if (matcher.matches()) {
            String areaCode = matcher.group(1);
            String separator = matcher.group(2);
            String localNum = matcher.group(3);

            // 本地号码保留后4位，前面用*填充
            String maskedLocal = "****" + localNum.substring(localNum.length() - 4);
            return areaCode + separator + maskedLocal;
        }

        // 处理纯数字座机号（无分隔符）
        if (pureNum.startsWith("0")) {
            // 提取区号（3-4位）
            int areaCodeLen = (pureNum.length() == 10) ? 3 : 4; // 10位：3位区号+7位本地号；11位：4位区号+7位本地号
            String areaCode = pureNum.substring(0, areaCodeLen);
            String localNum = pureNum.substring(areaCodeLen);

            int hideLength = localNum.length() - 4; // 计算需要隐藏的位数
            String hideStr = hideLength > 0 ? "*".repeat(hideLength) : ""; // 生成对应数量的*
            String maskedLocal = hideStr + localNum.substring(localNum.length() - 4);
            return areaCode + maskedLocal;
        }

        return phone;
    }
    /**
     * 检查JSONArray中是否包含指定元素
     * @param jsonArray 待检查的JSONArray
     * @param target 目标元素（支持String、Number、Boolean、JSONObject等）
     * @return 包含返回true，否则返回false
     */
    private boolean containsString(JSONArray jsonArray, String target) {
        if (jsonArray == null || target == null) {
            return false;
        }

        for (int i = 0; i < jsonArray.length(); i++) {
            String element;
            try {
                // 直接获取字符串元素（若元素不是字符串会抛出JSONException）
                element = jsonArray.getString(i);
            } catch (JSONException e) {
                // 若元素不是字符串类型，直接跳过（符合"都是字符串"的前提）
                continue;
            }

            // 字符串精确匹配（区分大小写）
            if (target.equals(element)) {
                return true;
            }
        }
        return false;
    }
}
