package com.af.v4.system.common.jpa.sensitive;

import java.util.Arrays;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;

/**
 * 智能识别字段值的数据类型，判断是否需要脱敏加密处理
 * - 自动识别原始/已脱敏/已加密数据状态
 * - 防止重复处理，提升系统性能
 * - 支持多种数据格式的智能判断
 * - 使用 Pattern Matching、Stream API 和 预编译正则表达式
 */
public class SensitiveDataClassifier {

    // 预编译正则表达式以提高性能
    private static final Pattern MASKED_PATTERN = Pattern.compile(".*\\*.*");
    private static final Pattern ENCRYPTED_PATTERN = Pattern.compile(".*\\|-.+-\\|.*");
    private static final Pattern WHITESPACE_PATTERN = Pattern.compile("^\\s*$");

    // 字段类型匹配策略
    private static final Set<String> NAME_FIELD_KEYWORDS = Set.of("name", "username", "realname", "nickname");
    private static final Set<String> PHONE_FIELD_KEYWORDS = Set.of("phone", "mobile", "tel", "telephone");
    private static final Set<String> ADDRESS_FIELD_KEYWORDS = Set.of("address", "addr", "location");
    private static final Set<String> ID_FIELD_KEYWORDS = Set.of("idnumber", "cardnumber", "idcard");

    // 字段匹配器
    private enum FieldType {
        NAME(field -> containsAnyKeyword(field, NAME_FIELD_KEYWORDS)),
        PHONE(field -> containsAnyKeyword(field, PHONE_FIELD_KEYWORDS)),
        ADDRESS(field -> containsAnyKeyword(field, ADDRESS_FIELD_KEYWORDS)),
        ID_NUMBER(SensitiveDataClassifier::containsIdKeywords),
        GENERIC(field -> true); // 默认匹配所有

        private final Predicate<String> matcher;

        FieldType(Predicate<String> matcher) {
            this.matcher = matcher;
        }

        boolean matches(String fieldName) {
            return matcher.test(fieldName.toLowerCase());
        }

        static FieldType classify(String fieldName) {
            return Arrays.stream(values())
                    .filter(type -> type.matches(fieldName))
                    .findFirst()
                    .orElse(GENERIC);
        }
    }

    /**
     * 检查字段名是否包含任何关键词
     */
    private static boolean containsAnyKeyword(String fieldName, Set<String> keywords) {
        String lowerField = fieldName.toLowerCase();
        return keywords.stream().anyMatch(lowerField::contains);
    }

    /**
     * 检查身份证类型字段的特殊逻辑
     */
    private static boolean containsIdKeywords(String fieldName) {
        String lowerField = fieldName.toLowerCase();
        return (lowerField.contains("id") &&
                (lowerField.contains("number") || lowerField.contains("card"))) ||
                ID_FIELD_KEYWORDS.stream().anyMatch(lowerField::contains);
    }

    /**
     * 数据分类结果
     * 提供类型安全的分类结果，支持模式匹配
     */
    public sealed interface DataClassificationResult
            permits DataClassificationResult.RawData,
            DataClassificationResult.AlreadyMasked,
            DataClassificationResult.AlreadyEncrypted,
            DataClassificationResult.NullValue,
            DataClassificationResult.Uncertain {

        /**
         * 判断是否需要处理
         */
        boolean needsProcessing();

        /**
         * 获取分类描述
         */
        String getDescription();

        /**
         * 原始数据 - 需要脱敏和加密
         */
        record RawData(String reason) implements DataClassificationResult {
            public RawData() {
                this("检测到原始敏感数据");
            }

            @Override
            public boolean needsProcessing() {
                return true;
            }

            @Override
            public String getDescription() {
                return reason;
            }
        }

        /**
         * 已脱敏数据 - 跳过处理
         */
        record AlreadyMasked(String pattern) implements DataClassificationResult {
            public AlreadyMasked() {
                this("包含脱敏标识符(*)");
            }

            @Override
            public boolean needsProcessing() {
                return false;
            }

            @Override
            public String getDescription() {
                return STR."已脱敏: \{pattern}";
            }
        }

        /**
         * 已加密数据 - 跳过处理
         */
        record AlreadyEncrypted(String format) implements DataClassificationResult {
            public AlreadyEncrypted() {
                this("格式: |-...-|");
            }

            @Override
            public boolean needsProcessing() {
                return false;
            }

            @Override
            public String getDescription() {
                return STR."已加密: \{format}";
            }
        }

        /**
         * 空值 - 跳过处理
         */
        record NullValue() implements DataClassificationResult {
            @Override
            public boolean needsProcessing() {
                return false;
            }

            @Override
            public String getDescription() {
                return "空值或空字符串";
            }
        }

        /**
         * 不确定 - 需要进一步判断
         */
        record Uncertain(String reason) implements DataClassificationResult {
            public Uncertain() {
                this("无法确定数据类型");
            }

            @Override
            public boolean needsProcessing() {
                return false;
            }

            @Override
            public String getDescription() {
                return STR."不确定: \{reason}";
            }
        }

        // 便捷工厂方法
        static RawData rawData() {
            return new RawData();
        }

        static RawData rawData(String reason) {
            return new RawData(reason);
        }

        static AlreadyMasked alreadyMasked() {
            return new AlreadyMasked();
        }

        static AlreadyMasked alreadyMasked(String pattern) {
            return new AlreadyMasked(pattern);
        }

        static AlreadyEncrypted alreadyEncrypted() {
            return new AlreadyEncrypted();
        }

        static AlreadyEncrypted alreadyEncrypted(String format) {
            return new AlreadyEncrypted(format);
        }

        static NullValue nullValue() {
            return new NullValue();
        }

        static Uncertain uncertain() {
            return new Uncertain();
        }

        static Uncertain uncertain(String reason) {
            return new Uncertain(reason);
        }
    }

    /**
     * Classify sensitive field data
     *
     * @param fieldName field name
     * @param value     field value
     * @return data classification result
     */
    public DataClassificationResult classify(String fieldName, Object value) {
        if (value == null) {
            return DataClassificationResult.nullValue();
        }

        String strValue = value.toString().trim();
        if (strValue.isEmpty()) {
            return DataClassificationResult.nullValue();
        }

        try {
            // 使用 FieldType enum 进行字段类型识别
            FieldType fieldType = FieldType.classify(fieldName);

            // 使用 switch 表达式 (Java 21) 进行类型分类
            return switch (fieldType) {
                case NAME -> classifyValueByType(strValue, "用户名");
                case PHONE -> classifyValueByType(strValue, "手机号");
                case ADDRESS -> classifyValueByType(strValue, "地址");
                case ID_NUMBER -> classifyValueByType(strValue, "身份证号");
                case GENERIC -> classifyValueByType(strValue, "通用字段");
            };
        } catch (Exception e) {
            return DataClassificationResult.uncertain(STR."分类过程中发生异常: \{e.getMessage()}");
        }
    }

    /**
     * 通用的值分类方法，使用预编译的正则表达式
     */
    private DataClassificationResult classifyValueByType(String value, String fieldType) {
        // 使用预编译正则表达式检查是否已脱敏
        if (MASKED_PATTERN.matcher(value).matches()) {
            return DataClassificationResult.alreadyMasked(STR."\{fieldType}包含*符号");
        }

        // 使用预编译正则表达式检查是否已加密
        if (ENCRYPTED_PATTERN.matcher(value).matches()) {
            return DataClassificationResult.alreadyEncrypted(STR."\{fieldType}加密格式");
        }

        // 检查是否为有效数据
        if (value.length() >= 2 && !WHITESPACE_PATTERN.matcher(value).matches()) {
            return DataClassificationResult.rawData(STR."\{fieldType}原始数据");
        }

        return DataClassificationResult.uncertain(STR."\{fieldType}长度过短或全为空格");
    }


    /**
     * 检查值是否已被处理（已脱敏或已加密）
     * 使用统一的分类方法和预编译正则表达式
     *
     * @param value 要检查的值
     * @return true 表示已被处理
     */
    public boolean isAlreadyProcessed(String value) {
        if (value == null || value.trim().isEmpty()) {
            return true; // 空值不需要处理
        }

        DataClassificationResult result = classifyValueByType(value, "通用字段");
        return !result.needsProcessing();
    }
}