package com.af.v4.system.common.job.processer;

import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.log.enums.OperatorType;
import com.af.v4.system.common.log.service.LogResolveService;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import tech.powerjob.worker.core.processor.ProcessResult;
import tech.powerjob.worker.core.processor.TaskContext;
import tech.powerjob.worker.core.processor.sdk.BasicProcessor;

/**
 * 简单定时调度任务封装
 *
 * @author Mr.river
 */
@Component
public abstract class GlazeBasicTimer implements BasicProcessor {

    private final Logger LOGGER = LoggerFactory.getLogger(GlazeBasicTimer.class);

    private final LogResolveService logResolveService;

    protected GlazeBasicTimer(LogResolveService logResolveService) {
        this.logResolveService = logResolveService;
    }

    @Override
    public ProcessResult process(TaskContext taskContext) {
        LOGGER.info("开始执行定时任务：{}", taskContext.getTaskName());
        long beginTime = System.currentTimeMillis();
        Exception sourceEx = null;
        JSONObject jobParams = StringUtils.isNotEmpty(taskContext.getJobParams())
                ? new JSONObject(taskContext.getJobParams())
                : new JSONObject();
        try {
            return exec(jobParams);
        } catch (Exception e) {
            if (e instanceof InterruptedException _) {
                String msg = "定时业务手动中断执行";
                LOGGER.warn(msg);
                return new ProcessResult(false, msg);
            } else {
                sourceEx = e;
                if (e instanceof LogicException ex) {
                    LOGGER.error("定时业务执行失败：{}", ex.getStack());
                } else {
                    LOGGER.error("定时业务执行失败", e);
                }
                return new ProcessResult(false, "定时业务执行失败:" + e.getMessage());
            }
        } finally {
            long endTime = System.currentTimeMillis();
            long costTime = endTime - beginTime;
            handleLog(taskContext, sourceEx, costTime);
            LOGGER.info("定时任务：{} 执行完成，耗时：{} ms", taskContext.getTaskName(), costTime);
        }
    }

    protected void handleLog(TaskContext taskContext, final Throwable e, long costTime) {
        logResolveService.resolveLog(
                this.getClass().getName(),
                "process",
                BusinessType.LOGIC.name(),
                "执行定时调度任务：" + taskContext.getTaskName(),
                OperatorType.SYSTEM.name(),
                taskContext.getJobParams(),
                costTime,
                e
        );
    }

    public abstract ProcessResult exec(JSONObject jobParams) throws InterruptedException;
}
