package com.af.v4.system.common.elasticsearch.common;


import co.elastic.clients.elasticsearch._types.SortOrder;
import co.elastic.clients.elasticsearch.core.SearchRequest;
import co.elastic.clients.elasticsearch.core.search.HighlightField;

import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

/**
 * 构造ES SearchSourceBuilder 对象
 * 可以多次，手动指定条件
 * 支持链式调用
 */
public class ESConditionFactory {

    // ES请求条件构造器
    private final SearchRequest.Builder builder = new SearchRequest.Builder();

    // 高亮定义Map，用于构建高亮条件
    private final Map<String, HighlightField> highlightFieldMap = new HashMap<>();

    // 查询条件
    private final StringBuffer queryCondition = new StringBuffer();
    // index
    private final String index;
    // should、must、mustNot三个条件构建构造字符串
    private StringBuffer shouldCondition = null;
    private StringBuffer mustCondition = null;
    private StringBuffer mustNotCondition = null;
    // should必须有多少个满足条件
    private int minimumShouldMatch = 0;

    /**
     * 构造方法，需要传入需要查询的index名
     *
     * @param index es中index名
     */
    public ESConditionFactory(String index) {
        queryCondition.append("{");
        queryCondition.append("\"bool\": {");
        this.index = index;
    }

    /**
     * 获取should条件构造字符串
     */
    private StringBuffer getShouldCondition() {
        if (shouldCondition == null) {
            shouldCondition = new StringBuffer();
            shouldCondition.append("\"should\": [");
        } else {
            shouldCondition.append(",");
        }
        return shouldCondition;
    }

    /**
     * 获取must条件构造字符串
     */
    private StringBuffer getMustCondition() {
        if (mustCondition == null) {
            mustCondition = new StringBuffer();
            mustCondition.append("\"must\": [");
        } else {
            mustCondition.append(",");
        }
        return mustCondition;
    }

    /**
     * 获取mustNot条件构造字符串
     */
    private StringBuffer getMustNotCondition() {
        if (mustNotCondition == null) {
            mustNotCondition = new StringBuffer();
            mustNotCondition.append("\"must_not\": [");
        } else {
            mustNotCondition.append(",");
        }
        return mustNotCondition;
    }

    /**
     * 获取should条件结果
     */
    private String getShouldResult() {
        if (shouldCondition == null) {
            return "";
        } else {
            shouldCondition.append("]");
            return shouldCondition.toString();
        }
    }

    /**
     * 获取must条件结果
     */
    private String getMustResult() {
        if (mustCondition == null) {
            return "";
        } else {
            mustCondition.append("]");
            return mustCondition.toString();
        }
    }

    /**
     * 获取mustNot条件结果
     */
    private String getMustNotResult() {
        if (mustNotCondition == null) {
            return "";
        } else {
            mustNotCondition.append("]");
            return mustNotCondition.toString();
        }
    }

    /**
     * 高亮查询
     * ！！！注意！！！
     * 该方法必须明确指定查询哪一个字段，哪一个值
     * 否则不会生效，findAll一起使用会失效
     * allField查询，其实就是在查all_field_for_search字段，如果需要allField查询高亮
     * 请指定高亮字段为all_field_for_search
     *
     * @param preTag          高亮前缀
     * @param postTag         高亮后缀
     * @param highlightFields 需要高亮的字段
     * @return 当前实例对象
     */
    public ESConditionFactory highlight(String preTag, String postTag, String[] highlightFields) {
        HighlightField.Builder builder = new HighlightField.Builder();
        builder.preTags(preTag);
        builder.postTags(postTag);
        HighlightField highlightOption = builder.build();
        for (String highlightField : highlightFields) {
            this.highlightFieldMap.put(highlightField, highlightOption);
        }
        return this;
    }

    /**
     * 全文检索，搜索所有字段
     *
     * @param target 目标值
     * @return 当前
     */
    public ESConditionFactory allField(String target) {
        StringBuffer shouldSB = getShouldCondition();
        shouldSB.append("{\"match\": {\"all_field_for_search\": \"");
        shouldSB.append(target);
        shouldSB.append("\"\n}}");
        return this;
    }

    /**
     * 按范围查询
     *
     * @param fieldName 字段名
     * @param rangeMode 范围条件
     * @param target    目标值
     * @return 当前实例对象
     */
    public ESConditionFactory range(String fieldName, String rangeMode, String target) {
        StringBuffer mustSB = getMustCondition();
        switch (rangeMode) {
            case "gt" ->
                    mustSB.append("{\"range\": {\"").append(fieldName).append("\": {\"gt\": ").append(target).append("}}}");
            case "gte" ->
                    mustSB.append("{\"range\": {\"").append(fieldName).append("\": {\"gte\": ").append(target).append("}}}");
            case "lt" ->
                    mustSB.append("{\"range\": {\"").append(fieldName).append("\": {\"lt\": ").append(target).append("}}}");
            case "lte" ->
                    mustSB.append("{\"range\": {\"").append(fieldName).append("\": {\"lte\": ").append(target).append("}}}");
        }
        return this;
    }

    /**
     * 范围选择
     *
     * @param fieldName 字段名
     * @param min       最小值
     * @param minMode   大于 or 大于等于
     * @param max       最大值
     * @param maxMode   小于 or 小于等于
     * @return 当前实例对象
     */
    public ESConditionFactory between(String fieldName, String min, String minMode, String max, String maxMode) {
        StringBuffer mustSB = getMustCondition();
        String minModeStr = null;
        String maxModeStr = null;
        switch (minMode) {
            case "gt" -> minModeStr = "gt";
            case "gte" -> minModeStr = "gte";
            case "lt" -> minModeStr = "lt";
            case "lte" -> minModeStr = "lte";
        }
        switch (maxMode) {
            case "gt" -> maxModeStr = "gt";
            case "gte" -> maxModeStr = "gte";
            case "lt" -> maxModeStr = "lt";
            case "lte" -> maxModeStr = "lte";
        }
        mustSB.append("{\"range\": {\"");
        mustSB.append(fieldName);
        mustSB.append("\": {\"");
        mustSB.append(minModeStr);
        mustSB.append("\": ");
        mustSB.append(min);
        mustSB.append(",\"");
        mustSB.append(maxModeStr);
        mustSB.append("\": ");
        mustSB.append(max);
        mustSB.append("}}}");
        return this;
    }

    /**
     * 应该是xxx，与or的作用相同
     *
     * @param fieldName 字段
     * @param target    目标值
     * @return 当前实例对象
     */
    public ESConditionFactory should(String fieldName, String target) {
        StringBuffer shouldSB = getShouldCondition();
        shouldSB.append("{\"match\": {\"");
        shouldSB.append(fieldName);
        shouldSB.append("\": \"");
        shouldSB.append(target);
        shouldSB.append("\"}}");
        return this;
    }

    /**
     * 应该是xxx，与or的作用相同
     * 本方法禁止对目标词再次分词
     *
     * @param fieldName 字段
     * @param target    目标值
     * @return 当前实例对象
     */
    public ESConditionFactory shouldWithoutSplit(String fieldName, String target) {
        StringBuffer shouldSB = getShouldCondition();
        shouldSB.append("{\"term\": {\"");
        shouldSB.append(fieldName);
        shouldSB.append("\": \"");
        shouldSB.append(target);
        shouldSB.append("\"}}");
        return this;
    }

    /**
     * 必须是
     *
     * @param fieldName 字段名
     * @param target    目标值
     * @return 当前实例对象
     */
    public ESConditionFactory must(String fieldName, String target) {
        StringBuffer mustSB = getMustCondition();
        mustSB.append("{\"term\": {\"");
        mustSB.append(fieldName);
        mustSB.append("\": \"");
        mustSB.append(target);
        mustSB.append("\"}}");
        return this;
    }

    /**
     * 必须不是
     *
     * @param fieldName 字段名
     * @param target    目标值
     * @return 当前实例对象
     */
    public ESConditionFactory mustNot(String fieldName, String target) {
        StringBuffer notSB = getMustNotCondition();
        notSB.append("{\"term\": {\"");
        notSB.append(fieldName);
        notSB.append("\": \"");
        notSB.append(target);
        notSB.append("\"}}");
        return this;
    }


    /**
     * 查找所有
     *
     * @return 当前实例对象
     */
    public ESConditionFactory findAll() {
        StringBuffer shouldSB = getShouldCondition();
        shouldSB.append("{\"match_all\": {}}");
        return this;
    }

    /**
     * 对搜索内容进行排序
     *
     * @param fieldName 排序字段
     * @param order     排序类型，该值为 org.elasticsearch.search.sort.SortOrder 枚举类
     * @return 当前实例对象
     */
    public ESConditionFactory order(String fieldName, SortOrder order) {
        builder.sort(s0 -> s0.field(f1 -> f1.field(fieldName).order(SortOrder.Desc)));
        return this;
    }

    /**
     * 分页查询
     *
     * @param from 从第几条开始
     * @param size 每页大小
     * @return 当前实例对象
     */
    public ESConditionFactory limit(int from, int size) {
        builder.from(from);
        builder.size(size);
        return this;
    }

    public ESConditionFactory minimumShouldMatch(int minimumShouldMatch) {
        this.minimumShouldMatch = minimumShouldMatch;
        return this;
    }

    /**
     * 获取最终builder对象
     *
     * @return builder对象
     */
    public SearchRequest result() {
        String shouldResult = getShouldResult();
        String mustResult = getMustResult();
        String mustNotResult = getMustNotResult();
        boolean hasShould = false;
        if (!shouldResult.isEmpty()) {
            hasShould = true;
            queryCondition.append(shouldResult);
        }
        boolean hasMust = false;
        if (!mustResult.isEmpty()) {
            hasMust = true;
            if (hasShould) {
                queryCondition.append(",");
                queryCondition.append(mustResult);
            } else {
                queryCondition.append(mustResult);
            }
        }
        if (!mustNotResult.isEmpty()) {
            if (hasShould || hasMust) {
                queryCondition.append(",");
                queryCondition.append(mustNotResult);
            } else {
                queryCondition.append(mustNotResult);
            }
        }
        if (minimumShouldMatch > 0) {
            queryCondition.append(",");
            queryCondition.append("\"minimum_should_match\": ");
            queryCondition.append(minimumShouldMatch);
        }
        queryCondition.append("}");
        queryCondition.append("}");
        builder.index(index);
        builder.highlight(h -> h.fields(highlightFieldMap));
        String encodedJSON = Base64.getEncoder().encodeToString(queryCondition.toString().getBytes());
        builder.query(b0 -> b0.wrapper(b1 -> b1.query(encodedJSON)));
        return builder.build();
    }
}
