package com.af.v4.system.common.datasource.provider;

import com.af.v4.system.common.datasource.config.AfDataSourceConfig;
import com.af.v4.system.common.datasource.enums.DbType;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;

import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 数据源创建器
 */
public class DataSourceProvider {

    private final static Map<String, HikariDataSource> INNER_DATA_SOURCES = new ConcurrentHashMap<>();

    private final static int DEFAULT_MAX_ACTIVE_SIZE = Runtime.getRuntime().availableProcessors() * 2 + 1;

    public static HikariDataSource create(AfDataSourceConfig config, Integer minimumIdle, Integer maximumPoolSize,
                                          Integer connectionTimeout, Integer idleTimeout, Integer maxLifetime) {
        String dataSourceName = config.getName();
        return INNER_DATA_SOURCES.computeIfAbsent(dataSourceName, k -> {
            HikariConfig hikariConfig = new HikariConfig();
            hikariConfig.setPoolName(k);
            hikariConfig.setDriverClassName(config.getDriverClassName());
            hikariConfig.setJdbcUrl(config.getUrl());
            hikariConfig.setUsername(config.getUsername());
            hikariConfig.setPassword(config.getPassword());
            hikariConfig.setMinimumIdle(Objects.requireNonNullElse(minimumIdle, DEFAULT_MAX_ACTIVE_SIZE));
            hikariConfig.setMaximumPoolSize(Objects.requireNonNullElse(maximumPoolSize, DEFAULT_MAX_ACTIVE_SIZE));
            hikariConfig.setConnectionTimeout(Objects.requireNonNullElse(connectionTimeout, 20000));
            hikariConfig.setIdleTimeout(Objects.requireNonNullElse(idleTimeout, 20000));
            hikariConfig.setMaxLifetime(Objects.requireNonNullElse(maxLifetime, 180000));
            Properties props = hikariConfig.getDataSourceProperties(); // 拿到已有的
            if (config.getType() == DbType.sqlserver){
                props.setProperty("encrypt","false");
                props.setProperty("trustServerCertificate","true");
                if (config.getSocketTimeout() != null) {
                    props.setProperty("socketTimeout", config.getSocketTimeout().toString());
                }
                hikariConfig.setDataSourceProperties(props);
            }
            return new HikariDataSource(hikariConfig);
        });
    }
}
