package com.af.v4.system.common.ai.service;

import com.af.v4.system.common.ai.entity.AiProvider;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.proxy.liuli.ILiuLiConfigServiceProxy;
import com.alibaba.cloud.ai.dashscope.api.DashScopeApi;
import com.alibaba.cloud.ai.dashscope.chat.DashScopeChatModel;
import com.alibaba.cloud.ai.dashscope.chat.DashScopeChatOptions;
import com.alibaba.cloud.ai.dashscope.embedding.DashScopeEmbeddingModel;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.client.advisor.PromptChatMemoryAdvisor;
import org.springframework.ai.chat.client.advisor.QuestionAnswerAdvisor;
import org.springframework.ai.chat.memory.ChatMemory;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.embedding.EmbeddingModel;
import org.springframework.ai.vectorstore.SimpleVectorStore;
import org.springframework.ai.vectorstore.VectorStore;
import org.springframework.stereotype.Service;
import reactor.core.publisher.Flux;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import static org.springframework.ai.chat.client.advisor.AbstractChatMemoryAdvisor.CHAT_MEMORY_CONVERSATION_ID_KEY;
import static org.springframework.ai.chat.client.advisor.AbstractChatMemoryAdvisor.CHAT_MEMORY_RETRIEVE_SIZE_KEY;

@Service
public class AiService {
    private static final Logger LOGGER = LoggerFactory.getLogger(AiService.class);
    private static final String AI_CONFIG_NAME = "ai_config";
    private static final List<AiProvider> AI_PROVIDERS = new ArrayList<>();
    private static String DEFAULT_PROMPT;
    private final ChatMemory chatMemory;

    public AiService(ILiuLiConfigServiceProxy liuLiConfigServiceProxy, ChatMemory chatMemory) {
        this.chatMemory = chatMemory;
        // 构造模型
        JSONObject aiOptions = liuLiConfigServiceProxy.get(AI_CONFIG_NAME);
        if (aiOptions == null) {
            LOGGER.error(STR."加载Ai服务失败，使用Ai相关接口服务请先创建[\{AI_CONFIG_NAME}]配置");
            return;
        }
        JSONArray modelProviders = aiOptions.getJSONArray("modelProviders");
        if (modelProviders == null || modelProviders.isEmpty()) {
            throw new ServiceException(STR."[\{AI_CONFIG_NAME}]配置里没有定义任何模型供应商");
        }
        modelProviders.forEach(item -> {
            JSONObject modelProvider = (JSONObject) item;
            // 模型供应商名称
            String modelProviderName = modelProvider.getString("name");
            // 注册模型集合
            JSONObject models = modelProvider.getJSONObject("models");
            // ApiKey集合
            JSONArray apiKeys = modelProvider.getJSONArray("apiKeys");
            // 构造模型供应商
            AiProvider aiProvider = new AiProvider(modelProviderName, models, apiKeys);
            AI_PROVIDERS.add(aiProvider);
        });
        // 设置默认Prompt
        String prompt = aiOptions.optString("defaultPrompt", null);
        if (prompt != null && !prompt.isEmpty()) {
            DEFAULT_PROMPT = prompt;
        } else {
            DEFAULT_PROMPT = "{value}";
        }
    }

    /**
     * 构造Ai模型调用客户端
     *
     * @param modelName       模型名
     * @param customizePrompt 自定义prompt
     * @param isUseWebSearch  是否联网搜索，部分模型不支持，会自动关闭
     * @return 客户端
     */
    public ChatClient buildChatClient(String modelName, String customizePrompt, boolean isUseWebSearch) {
        AiProvider aiProvider = AI_PROVIDERS.stream()
                .filter((item) -> item.models().has(modelName))
                .findFirst()
                .orElseThrow(() -> new ServiceException(STR."[\{AI_CONFIG_NAME}]配置里没有提供该模型：\{modelName}"));
        JSONArray apiKeys = aiProvider.apiKeys();
        if (apiKeys == null || apiKeys.isEmpty()) {
            throw new ServiceException(STR."Ai模型[\{modelName}]中没有设置ApiKey");
        }
        // 随机选择一个 API key
        String apiKey = apiKeys.get(ThreadLocalRandom.current().nextInt(apiKeys.length())).toString();
        DashScopeApi dashScopeApi = new DashScopeApi(apiKey);
        // 初始化客户端
        ChatModel chatModel = new DashScopeChatModel(dashScopeApi);
        ChatClient.Builder builder = ChatClient.builder(chatModel);
        // 设置客户端属性
        DashScopeChatOptions chatOptions = new DashScopeChatOptions();
        chatOptions.setModel(modelName);
        boolean isSupportWebSearch = ((JSONObject) aiProvider.models().get(modelName)).optBoolean("isSupportWebSearch", true);
        if (isSupportWebSearch) {
            chatOptions.setEnableSearch(isUseWebSearch);
        }
        // 初始化向量存储
        EmbeddingModel embeddingModel = new DashScopeEmbeddingModel(dashScopeApi);
        VectorStore vectorStore = SimpleVectorStore.builder(embeddingModel).build();
        // 定义Prompt
        String prompt = DEFAULT_PROMPT;
        return builder
                .defaultOptions(chatOptions)
                .defaultSystem(promptSystemSpec -> {
                    promptSystemSpec.text(prompt);
                    promptSystemSpec.param("value", customizePrompt == null ? "" : customizePrompt);
                })
                .defaultAdvisors(
                        new PromptChatMemoryAdvisor(chatMemory),
                        new QuestionAnswerAdvisor(vectorStore)
                ).build();
    }

    public ChatClient buildChatClient(String modelName, String customizePrompt) {
        return buildChatClient(modelName, customizePrompt, false);
    }

    public ChatClient buildChatClient(String modelName) {
        return buildChatClient(modelName, null);
    }

    /**
     * 流式对话
     *
     * @param client      Ai模型调用客户端
     * @param chatId      对话标识
     * @param userMessage 发起内容
     * @return 回复
     */
    public Flux<String> chat(ChatClient client, String chatId, String userMessage) {
        return client.prompt()
                .user(formatText(userMessage))
                .advisors(a -> a.param(CHAT_MEMORY_CONVERSATION_ID_KEY, chatId)
                        .param(CHAT_MEMORY_RETRIEVE_SIZE_KEY, 100))
                .stream()
                .content();
    }

    /**
     * 同步对话
     *
     * @param client      Ai模型调用客户端
     * @param chatId      对话标识
     * @param userMessage 发起内容
     * @return 回复
     */
    public String syncChat(ChatClient client, String chatId, String userMessage) {
        return client.prompt()
                .user(formatText(userMessage))
                .advisors(a -> a.param(CHAT_MEMORY_CONVERSATION_ID_KEY, chatId)
                        .param(CHAT_MEMORY_RETRIEVE_SIZE_KEY, 100))
                .call()
                .content();
    }

    /**
     * 格式化文本
     *
     * @param userMessage 用户输入的文本
     * @return 格式化后的文本
     */
    private String formatText(String userMessage) {
        return userMessage.replace("{", "\\{").replace("}", "\\}");
    }
}
