package com.af.v4.system.runtime.dto;

import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.dialects.Dialect;
import com.af.v4.system.common.datasource.enums.DbType;
import com.af.v4.system.runtime.utils.DataTypeUtils;

import java.util.List;

/**
 * UpgradeTable
 */
public record UpgradeTable(String tableName, List<UpgradeField> fields) {

    /**
     * 获取检查表是否存在的SQL
     *
     * @return 检查表是否存在的SQL
     */
    public String getSearchSql() {
        DbType dbType = DynamicDataSource.getDbType();
        Dialect dialect = dbType.dialect;

        if (dialect.isOracleFamily()) {
            return STR."select 1 from user_tables where table_name = '\{tableName.toUpperCase()}'";
        } else if (dialect.isPostgresFamily()) {
            return STR."select 1 from information_schema.tables where table_name = '\{tableName.toLowerCase()}'";
        } else if (dialect.isMySqlFamily()) {
            return STR."select 1 from information_schema.tables where table_name = '\{tableName}'";
        } else if (dialect.isSQLServerFamily()) {
            return STR."select 1 from sysobjects where name = '\{tableName}' and type = 'U'";
        } else {
            throw new UnsupportedOperationException(STR."不支持的数据库类型: \{dbType}");
        }
    }

    /**
     * 获取创建表的SQL
     *
     * @return 创建表的SQL
     */
    public String getCreateSql() {
        DbType dbType = DynamicDataSource.getDbType();
        Dialect dialect = dbType.dialect;

        StringBuilder sql = new StringBuilder();
        sql.append("CREATE TABLE ").append(DataTypeUtils.getTableNameWithCase(tableName, dialect)).append(" (");

        // 添加字段定义
        for (int i = 0; i < fields.size(); i++) {
            UpgradeField field = fields.get(i);
            String fieldName = field.field();
            String fieldType = field.type();
            String defaultValue = field.defaultValue();

            fieldName = DataTypeUtils.getFieldNameWithCase(fieldName, dialect);
            fieldType = DataTypeUtils.convertDataType(fieldType, dbType);

            sql.append(fieldName).append(" ").append(fieldType);

            // 添加默认值（如果有）
            if (defaultValue != null && !defaultValue.isEmpty()) {
                sql.append(" DEFAULT ").append(defaultValue);
            }

            // 如果不是最后一个字段，添加逗号
            if (i < fields.size() - 1) {
                sql.append(", ");
            }
        }

        sql.append(")");

        // 添加数据库特定的表尾部
        if (dialect.isMySqlFamily()) {
            sql.append(" ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
        }

        return sql.toString();
    }
}
