package com.af.v4.v3.workflow.plugin;

import com.af.v4.system.common.jpa.service.SqlService;
import com.af.v4.system.common.jpa.session.SessionPool;
import com.af.v4.v3.workflow.controller.Server;
import com.af.v4.v3.workflow.perform.ActivityDef;
import com.af.v4.v3.workflow.perform.DiversionDef;
import com.af.v4.v3.workflow.perform.ProcessDef;
import com.af.v4.v3.workflow.perform.ProcessInstance;
import org.hibernate.Session;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

@Component
public class WorkFlowLogicServer {

    private final SqlService sqlService;

    private final SessionPool sessionPool;

    private final Server server;

    public WorkFlowLogicServer(SqlService sqlService, SessionPool sessionPool, Server server) {
        this.sqlService = sqlService;
        this.sessionPool = sessionPool;
        this.server = server;
    }

    public static boolean isRepeat(JSONArray jsonArray, JSONObject object) {

        for (int i = 0; i < jsonArray.length(); i++) {
            if (object.get("id").toString().equals(jsonArray.getJSONObject(i).get("id").toString())) {
                return false;
            }
        }
        return true;
    }

    public String startWorkFlow(String name, String username,
                                String userid,
                                JSONObject personExpression) {
        JSONArray arr = new JSONArray();
        String personExpressionArr = arr.put(personExpression).toString();
        return server.txStartProcess(name, username, userid, personExpressionArr);
    }

    /**
     * 开启一个流程
     *
     * @param name 流程名
     * @return 流程实例编号
     */
    public String startWorkflow(String name) {
        return server.txstartProcess(name);
    }

    /**
     * 开始一个活动
     *
     * @param processid 流程实例id
     * @param name      活动名称
     * @return
     */
    public String startActivity(String processid, String name) {
        return server.txstartActivity(processid, name);
    }

    /**
     * 开始一个活动
     *
     * @param processid 流程实例id
     * @param name      活动名称
     * @param username  操作用户名
     * @param userid    操作用户id
     */
    public int startActivity(String processid, String name, String username, String userid) {
        return server.txstartActivity(processid, name, username, userid);
    }

    /**
     * 开始一个活动
     *
     * @param processid 流程实例id
     * @param name      活动名称
     * @param vars      变量值
     * @param username  操作用户名
     * @param userid    操作用户id
     */
    public int startActivity(String processid, String name, JSONObject vars, String username, String userid) {
        return server.txstartActivity(processid, name, vars.toString(), username, userid);
    }

    /**
     * 流程中变量变化
     *
     * @param processid 流程id
     * @param vars:     JSON串 形式的变量值，形式：{varname: value}
     */
    public void setVars(String processid, String vars) {
        server.ttxsetVars(processid, vars);
    }

    /**
     * 流程中变量变化
     *
     * @param processid 流程id
     * @param vars:     JSON串 形式的变量值，形式：{varname: value}
     */
    public void setVars(String processid, String vars, String username, String userid) {
        server.txsetVars(processid, vars, username, userid);
    }

    /**
     * 以json串方式给流程变量赋值
     *
     * @param processid 流程id
     * @param json      json对象
     */
    public void setVars(String processid, JSONObject json) {
        server.ttxsetVars(processid, json.toString());
    }

    /**
     * 以json串方式给流程变量赋值
     *
     * @param processid 流程id
     * @param json      json对象
     */
    public JSONObject setVars(String processid, JSONObject json, String username, String userid) {
        return server.ttxsetVars(processid, json.toString(), username, userid);
    }

    /**
     * 以json串方式给流程变量赋值
     *
     * @param processid 流程id
     * @param json      json对象
     */
    public JSONObject setVars(String defid, String processid, JSONObject json, String username, String userid) {
        return server.ttxsetVars(defid, processid, json.toString(), username, userid);
    }

    /**
     * 以json串方式给流程变量赋值
     *
     * @param processid 流程id
     * @param json      json对象
     */
    public JSONObject setVars(int actid, String processid, JSONObject json, String username, String userid) {
        return server.ttxsetVars(actid, processid, json.toString(), username, userid);
    }

    /**
     * 取得活动定义中action属性内容
     *
     * @param activityId: 活动实例ID
     */
    public String getAction(String activityId) {
        return server.txgetAction(activityId);
    }

    /**
     * 获取当前待办
     *
     * @param processid
     */
    public String getWait(String processid) {
        return server.txgetWait(processid);
    }

    /**
     * 获取工作流程图
     * 得到一批点和一批有方向线
     */
    public String getWFGraph(String processid) {
        return server.txgetWFGraph(processid);
    }

    /**
     * 获取流程图数据
     *
     * @param processid   流程id
     * @param width       画布宽度
     * @param levelHeight 等级高度
     * @param nodeType    节点类型
     * @param edgeType    线类型
     */
    public JSONObject getWFGraph(String processid, Integer width, Integer levelHeight, String nodeType, String edgeType) throws Exception {
        Session session = sessionPool.getSession();
        ProcessInstance procIns = session.get(ProcessInstance.class, processid);

        ProcessDef pdef = procIns.getDefine();

        // 开始活动
        ActivityDef act = pdef.getStartActivity();

        Map<Integer, ArrayList<String>> levelCount = new HashMap<>();
        getLevelCount(act, 1, levelCount);


        JSONObject res;
        JSONArray nodes = new JSONArray();
        JSONArray edges = new JSONArray();
        res = getNodes(nodes, edges, processid, act, width, levelHeight, nodeType, edgeType, 1, levelCount);

        return res;
    }

    /**
     * 组织节点、线
     *
     * @param nodes       节点集合
     * @param edges       线集合
     * @param processid   流程id
     * @param act         当前节点
     * @param width       画布宽度
     * @param levelHeight 级别高度
     * @param nodeType    节点类型
     * @param edgeType    线类型
     * @param level       当前级别
     * @param map         级别数量集合
     */
    public JSONObject getNodes(JSONArray nodes, JSONArray edges, String processid, ActivityDef act, int width, Integer levelHeight, String nodeType, String edgeType, Integer level, Map<Integer, ArrayList<String>> map) throws Exception {

        ArrayList<String> list = (ArrayList<String>) map.get(level);
        int levelCount = list.size();

        int index = list.indexOf(act.getID()) + 1;

        JSONObject node = new JSONObject();
        node.put("id", act.getID());
        node.put("type", nodeType);
        node.put("text", act.getName());
        node.put("x", (width * ((2 * index) - 1)) / (2 * levelCount));
        node.put("y", levelHeight * level);
        JSONArray jsonArray = sqlService.querySQL("workflowSelectActivityins",
                "select * from activityins where processid = '" + processid + "' and defid = '" + act.getID() + "' order by id desc");
        node.put("properties", jsonArray);

        if (isRepeat(nodes, node)) {
            nodes.put(node);
        }

        for (int i = 0; i < act.getSplits().size(); i++) {
            JSONObject edge = new JSONObject();
            DiversionDef divDef = act.getSplits().get(i);
            ActivityDef tailDef = divDef.getTail();
            edge.put("id", divDef.getID());
            edge.put("type", edgeType);
            edge.put("sourceNodeId", act.getID());
            edge.put("targetNodeId", tailDef.getID());

            if (isRepeat(edges, edge)) {
                edges.put(edge);
            }

            getNodes(nodes, edges, processid, tailDef, width, levelHeight, nodeType, edgeType, level + 1, map);
        }
        JSONObject obj = new JSONObject();

        obj.put("edges", edges);
        obj.put("nodes", nodes);
        return obj;
    }

    /**
     * 获取当前流程各级别数量
     *
     * @param act   当前节点
     * @param level 级别
     * @param map   级别数量集合
     */
    public Map<Integer, ArrayList<String>> getLevelCount(ActivityDef act, Integer level, Map<Integer, ArrayList<String>> map) {

        if (map.containsKey(level)) {
            ArrayList<String> list = map.get(level);
            if (!list.contains(act.getID())) {
                list.add(act.getID());
                map.put(level, list);
            }
        } else {
            ArrayList<String> list = new ArrayList<>();
            list.add(act.getID());
            map.put(level, list);
        }

        for (int i = 0; i < act.getSplits().size(); i++) {
            DiversionDef divDef = act.getSplits().get(i);
            ActivityDef tailDef = divDef.getTail();
            getLevelCount(tailDef, level + 1, map);
        }

        return map;
    }

    public String excuteWorkFlow(String acitityid,
                                 String username,
                                 String userid,
                                 JSONObject personExpression) {
        JSONArray arr = new JSONArray();
        String personExpressionArr = arr.put(personExpression).toString();
        return server.txperformActivity(acitityid, username, userid, personExpressionArr);
    }

    public String stopWorkFlow(String acitityid) {
        return server.txStopProcess(acitityid);
    }

    /*
     * 从执行者表达式R/P(执行者) 获取 执行者
     * 参数：执行者表达式 actorexpression
     * @return:执行者
     */
    public String getOperator(String actorexpression) {
        if (actorexpression.contains("userid")) {
            String sub = actorexpression.substring(actorexpression.indexOf("userid"));
            return sub.substring(sub.indexOf("'") + 1, sub.lastIndexOf("'"));
        } else if (actorexpression.contains("(") && actorexpression.length() > 3) {
            return actorexpression.substring(actorexpression.indexOf("(") + 1, actorexpression.indexOf(")"));
        } else {
            return "";
        }
    }
}
