package com.af.v4.v3.workflow.perform;

import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.liuli.application.service.GlazeApplicationService;
import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.io.*;
import java.net.JarURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * 流程图管理器
 */
@Component
public class DiagramManager implements Serializable {
    @Serial
    private static final long serialVersionUID = 373843474314241569L;
    private static final Logger log = LoggerFactory.getLogger(DiagramManager.class);
    /**
     * 流程图名字与内容列表
     */
    private final Map<String, Diagram> nameAndDiagrams = new ConcurrentHashMap<>();

    private final GlazeApplicationService applicationService;

    private final LiuLiConfigService liuliConfigService;

    /**
     * 构造管理器
     */
    public DiagramManager(GlazeApplicationService applicationService, LiuLiConfigService liuliConfigService) {
        this.applicationService = applicationService;
        this.liuliConfigService = liuliConfigService;
        load();
    }

    public static DiagramManager getInstance() {
        return SpringUtils.getBean(DiagramManager.class);
    }

    private Diagram getDiagram(String name) {
        // 根据文件路径得到流
        JSONObject result = liuliConfigService.get(name);
        if (result == null) {
            return null;
        } else {
            Diagram diagram = new Diagram();
            diagram.setName(name);
            // 设置流程图内容
            diagram.setContext(result.get("source").toString());
            return diagram;
        }
    }

    /**
     * 装载所有流程图, 在processes下
     */
    public void load() {
        URL url = this.getClass().getClassLoader().getResource("processes/");
        // 流程定义目录不存在，跳过
        if (url == null) {
            return;
        }
        log.info("========>路径>{}", url);
        String path = url.getPath();
        File[] files = new File(path).listFiles();
        if (files != null) {
            for (File f : files) {
                // 得到文件名
                String fileName = f.getName();
                try {
                    // 生成流程图
                    String digName = fileName.substring(0, fileName.indexOf("."));
                    Diagram diagram = new Diagram();
                    diagram.setName(digName);
                    nameAndDiagrams.put(digName, diagram);
                    // 根据文件路径得到流
                    StringBuilder result = new StringBuilder();
                    String encoding = "UTF-8";
                    InputStreamReader read = new InputStreamReader(
                            new FileInputStream(f), encoding);// 考虑到编码格式
                    BufferedReader bufferedReader = new BufferedReader(read);
                    String lineTxt;
                    while ((lineTxt = bufferedReader.readLine()) != null) {
                        result.append(lineTxt);
                    }
                    int index = result.indexOf("<");
                    if (index != -1) {
                        result = new StringBuilder(result.substring(index));
                    }
                    bufferedReader.close();
                    // 设置流程图内容
                    diagram.setContext(result.toString());
                } catch (Exception e) {
                    log.error("流程{}: {}", fileName, e.getMessage());
                }
            }
        }
    }

    /**
     * 根据路径去加载流程文件
     *
     * @param loadPath 给定路径
     */
    public void load(String loadPath) {
        if (loadPath == null || loadPath.isEmpty()) {
            return;
        }

        File files = new File(loadPath);
        if (!files.isFile()) {
            return;
        }

        String fileName = files.getName();
        try {
            // 生成流程图
            String digName = fileName.substring(0, fileName.indexOf("."));
            Diagram diagram = new Diagram();
            diagram.setName(digName);
            nameAndDiagrams.put(digName, diagram);
            // 根据文件路径得到流
            StringBuilder result = new StringBuilder();
            String encoding = "UTF-8";
            try (InputStreamReader read = new InputStreamReader(
                    new FileInputStream(files), encoding);
                 BufferedReader bufferedReader = new BufferedReader(read)
            ) {
                String lineTxt;
                while ((lineTxt = bufferedReader.readLine()) != null) {
                    result.append(lineTxt);
                }
                // 设置流程图内容
                diagram.setContext(result.toString());
            }
        } catch (Exception e) {
            log.error("流程{}: {}", fileName, e.getMessage());
        }
    }

    /**
     * 装载登录人目录下流程图, 在/(登录人目录)/processes下
     */
    public Diagram userLoad(String name) {
        try {
            // 生成流程图
            return getDiagram(name);
        } catch (Exception e) {
            throw new RuntimeException("流程" + name + ": " + e.getMessage(), e);
        }
    }

    /**
     * 该方法提供将 jar包的对应资源文件夹中所有的资源文件获取并读成字符串集合
     *
     * @param url  url路径，通过this.getClass().getClassLoader().getResource(path)获取jar包文件路径
     * @param path 需要遍历的文件夹名称--后面记得加上/右斜杠
     * @return 读取到的文件内容---一个字符串是一个文件的内容
     */
    private List<String> loadResources(URL url, String path) {
        List<String> strs = new ArrayList<>();
        int statu = 0;          //状态分0：未读 1：正在读 2：已读
        try {
            URL jarURL = new URL(url.toString().substring(0, url.toString().indexOf("!/") + 2));
            JarURLConnection jarCon = (JarURLConnection) jarURL.openConnection();
            JarFile jarFile = jarCon.getJarFile();
            Enumeration<JarEntry> jarEntrys = jarFile.entries();
            while (jarEntrys.hasMoreElements()) {
                if (statu == 2)    //当状态为已读时，则关掉循环，避免报错
                    break;
                JarEntry entry;
                try {
                    entry = jarEntrys.nextElement();
                } catch (Exception e) {
                    break;
                }
                String name = entry.getName();
                if (name.startsWith(path) && !entry.isDirectory()) {
                    if (statu == 0)
                        statu = 1;      //将状态更改为正在读
                    try (BufferedInputStream biput = new BufferedInputStream(Objects.requireNonNull(this.getClass().getClassLoader().getResourceAsStream(name)))) {
                        byte[] buffer = new byte[(int) entry.getSize()];
                        biput.read(buffer, 0, buffer.length);
                        String str = new String(buffer, StandardCharsets.UTF_8);
                        strs.add(str);
                    }
                } else if (statu == 1) {
                    statu = 2;
                }
            }

        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return strs;
    }

    /**
     * 得到所有流程图
     *
     * @return 流程图列表
     */
    public Collection<Diagram> getDiagrams() {
        return nameAndDiagrams.values();
    }

    /**
     * 取的所有流程图的名字
     *
     * @return 所有流程图的名字
     */
    public List<String> getDiagramNames() {
        return new ArrayList<>(nameAndDiagrams.keySet());
    }

    /**
     * 根据流程图名字取的流程图
     *
     * @param name 流程图名字
     * @return 流程图
     */
    public Diagram getDiagramByName(String name) {
        return nameAndDiagrams.get(name);
    }

    /**
     * 取登录用户的特殊流程图
     *
     * @param name 流程文件名
     * @return diagram 返回流程图
     */
    public Diagram getUserDiagramByName(String name) {
        return this.userLoad(name);
    }
}
