package com.af.v4.system.common.workflow;

import com.af.v4.system.common.core.proxy.jpa.IConditionProxy;
import com.af.v4.system.common.jpa.service.EntityService;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * FlowPathFinder
 * ----------------------------------------------------------
 * 从流程图（stepMap）和已执行节点（extraData）推导完整通路
 */
@Component
public class FlowPathFinder {


    @Autowired
    private EntityService entityService;

    /**
     * 获取完整路径
     *
     * @param stepMap   流程定义图，每个 key 为节点 id
     * @param extraData 已执行节点
     * @return JSONArray 完整可通行路径
     */
    public JSONArray findPath(JSONObject stepMap, JSONArray extraData, String workFlowId, String curUserId) {
        JSONArray path = new JSONArray();
        Set<String> visited = new HashSet<>();

        // 1. 清理 extraData 中的回退
        LinkedHashMap<String, JSONObject> filteredSteps = new LinkedHashMap<>();
        for (int i = 0; i < extraData.length(); i++) {
            JSONObject step = extraData.getJSONObject(i);
            String id = String.valueOf(step.optInt("id"));

            if (filteredSteps.containsKey(id)) {
                // 删除 id 之后的节点（回退）
                List<String> toRemove = new ArrayList<>();
                boolean startRemove = false;
                for (String key : filteredSteps.keySet()) {
                    if (startRemove) toRemove.add(key);
                    if (key.equals(id)) startRemove = true;
                }
                for (String r : toRemove) filteredSteps.remove(r);
            }
            filteredSteps.put(id, step);
        }

        // 2. 从第一个节点开始，沿流程顺序生成完整路径
        if (filteredSteps.isEmpty()) return path;

        // 获取第一个节点 id
        String curId = String.valueOf(filteredSteps.values().iterator().next().optInt("id"));

        int i = 1;
        while (curId != null && !visited.contains(curId)) {
            JSONObject node = stepMap.optJSONObject(curId);
            if (node == null) break;

            // 已执行节点信息补充
            JSONObject stepInfo = filteredSteps.getOrDefault(curId, new JSONObject());
            node.put("handler", stepInfo.optString("name"));
            node.put("f_handler_id", stepInfo.optString("f_handler_id"));
            node.put("date", stepInfo.optString("date"));
            node.put("note", stepInfo.optString("note"));
            node.put("deadline", stepInfo.optString("deadline"));
            node.put("status", stepInfo.opt("status"));
            node.put("f_data", stepInfo.opt("f_data"));

            node.put("sequence", i++);

            // 判断 如果是当前正在执行的节点 添加 是否当前处理人的判断, 兼容未更新vue2-client, 当满足 参数 curUserId 不为 null 的时候
            if(curUserId != null && "1".equals(stepInfo.optString("status"))) {
                node.put("isCurrently", false);
                // 判断逻辑: t_workflow_step_users 中 是否存在当前用户
                int id = stepInfo.getInt("t_workflow_step_id"); // 对应 t_workflow_step_users 表 f_step_id
                IConditionProxy condition = EntityService.getCondition();
                condition = condition.eq("f_step_id", id).and().eq("f_handler_id", curUserId).and().eq("f_workflow_id", workFlowId);
                JSONArray byCondition = entityService.getByCondition("t_workflow_step_users", condition);

                if (!byCondition.isEmpty()) {
                    node.put("isCurrently", true);
                }
            }

            path.put(node);
            visited.add(curId);

            // 如果此节点是条件节点，且未执行，则直接展开剩余的所有节点
            String branchType = JsonTools.convertToJson(node.optString("properties", "{}")).optString("branchType");
            if ("condition".equalsIgnoreCase(node.optString("type")) && "conditionalBranch".equalsIgnoreCase(branchType) && !hasConditionResult(stepInfo)) {
                appendRemainingSteps(path, stepMap, visited, curId, i);
                break;
            }

            // 如果此节点是并行分支节点，需要处理所有并行分支
            if ("parallelBranch".equalsIgnoreCase(branchType)) {
                List<String> nextIds = getNextParallelSteps(stepMap, curId);
                if (!nextIds.isEmpty()) {
                    // 将所有并行分支的下一个节点添加到路径中
                    for (String nextId : nextIds) {
                        if (!visited.contains(nextId)) {
                            JSONObject nextNode = stepMap.optJSONObject(nextId);
                            // 已经执行过的节点 填充数据
                            if (nextNode != null) {
                                JSONObject nextStepInfo = filteredSteps.getOrDefault(nextId, new JSONObject());
                                nextNode.put("handler", nextStepInfo.optString("name"));
                                nextNode.put("f_handler_id", nextStepInfo.optString("f_handler_id"));
                                nextNode.put("date", nextStepInfo.optString("date"));
                                nextNode.put("note", nextStepInfo.optString("note"));
                                nextNode.put("deadline", nextStepInfo.optString("deadline"));
                                nextNode.put("status", nextStepInfo.opt("status"));
                                nextNode.put("f_data", nextStepInfo.opt("f_data"));
                                nextNode.put("sequence", i++);
                                path.put(nextNode);
                                visited.add(nextId);
                            }
                        }
                    }
                    // 找到所有并行分支节点都指向的汇聚节点
                    String mergeNodeId = findMergeNode(stepMap, nextIds);
                    if (mergeNodeId != null) {
                        if (!visited.contains(mergeNodeId)) {
                            // 继续处理汇聚节点
                            curId = mergeNodeId;
                        } else {
                            // 如果汇聚节点已访问，继续处理汇聚节点的下一个节点
                            JSONObject mergeStepInfo = filteredSteps.getOrDefault(mergeNodeId, new JSONObject());
                            curId = getNextStep(stepMap, mergeNodeId, mergeStepInfo);
                        }
                    } else {
                        // 如果没有找到汇聚节点，继续处理第一个并行分支的下一个节点
                        curId = nextIds.isEmpty() ? null : getNextStep(stepMap, nextIds.get(0), filteredSteps.getOrDefault(nextIds.get(0), new JSONObject()));
                    }
                } else {
                    curId = null;
                }
            } else {
                // 获取下一个节点
                curId = getNextStep(stepMap, curId, stepInfo);
            }
        }

        return path;
    }

    /**
     * 找到所有并行分支节点都指向的汇聚节点
     * 并行节点都会聚集到一点，只需要取第一个并行分支节点的下一个节点
     *
     * @param stepMap 流程定义图
     * @param parallelNodeIds 并行分支节点 id 列表
     * @return 汇聚节点 id，如果找不到则返回 null
     */
    private static String findMergeNode(JSONObject stepMap, List<String> parallelNodeIds) {
        if (parallelNodeIds == null || parallelNodeIds.isEmpty()) return null;

        // 取第一个并行分支节点
        String firstNodeId = parallelNodeIds.get(0);
        JSONObject firstNode = stepMap.optJSONObject(firstNodeId);
        if (firstNode == null) return null;

        JSONObject props = firstNode.optJSONObject("properties");
        if (props == null) return null;

        JSONArray actions = props.optJSONArray("actions");
        if (actions == null || actions.isEmpty()) return null;

        // 获取第一个并行分支节点的下一个节点（汇聚节点）
        // 优先选择 type=submit 的动作，否则选择第一个
        for (int i = 0; i < actions.length(); i++) {
            JSONObject action = actions.getJSONObject(i);
            if ("submit".equalsIgnoreCase(action.optString("type"))) {
                String toId = String.valueOf(action.optInt("to"));
                if (!toId.equals("0") && !toId.equals("null")) {
                    return toId;
                }
            }
        }

        // 如果没有 submit，选择第一个
        String toId = String.valueOf(actions.getJSONObject(0).optInt("to"));
        if (!toId.equals("0") && !toId.equals("null")) {
            return toId;
        }

        return null;
    }

    /**
     * 获取并行分支节点的所有下一个节点
     *
     * @param stepMap 流程定义图
     * @param curId 当前节点 id
     * @return 所有并行分支的下一个节点 id 列表
     */
    private static List<String> getNextParallelSteps(JSONObject stepMap, String curId) {
        JSONObject curStep = stepMap.optJSONObject(curId);
        if (curStep == null) return new ArrayList<>();

        JSONObject props = curStep.optJSONObject("properties");
        if (props == null) return new ArrayList<>();

        JSONArray actions = props.optJSONArray("actions");
        if (actions == null || actions.isEmpty()) return new ArrayList<>();

        List<String> nextIds = new ArrayList<>();
        for (int i = 0; i < actions.length(); i++) {
            JSONObject action = actions.getJSONObject(i);
            String toId = String.valueOf(action.optInt("to"));
            if (!toId.equals("0") && !toId.equals("null")) {
                nextIds.add(toId);
            }
        }
        return nextIds;
    }

    /**
     * 获取当前节点的下一个节点
     * 优先选择 type=submit 的，否则选择第一个
     */
    private static String getNextStep(JSONObject stepMap, String curId, JSONObject stepInfo) {
        JSONObject curStep = stepMap.optJSONObject(curId);
        if (curStep == null) return null;

        JSONObject props = curStep.optJSONObject("properties");
        if (props == null) return null;

        JSONArray actions = props.optJSONArray("actions");
        if (actions == null || actions.isEmpty()) return null;

        String type = curStep.optString("type");

        // 条件节点优先使用 f_data 中的 __expressionRs
        if ("condition".equalsIgnoreCase(type) && stepInfo != null) {
            String fData = stepInfo.optString("f_data");
            if (fData != null && !fData.isEmpty()) {
                JSONObject fDataJson = new JSONObject(fData);
                if (fDataJson.has("__expressionRs")) {
                    return String.valueOf(fDataJson.optInt("__expressionRs"));
                }
            }
        }

        // 普通节点，优先 submit
        for (int i = 0; i < actions.length(); i++) {
            JSONObject action = actions.getJSONObject(i);
            if ("submit".equalsIgnoreCase(action.optString("type"))) {
                return String.valueOf(action.optInt("to"));
            }
        }

        // 如果没有 submit，选择第一个动作
        return String.valueOf(actions.getJSONObject(0).optInt("to"));
    }

    /**
     * 判断条件节点是否有明确的下一环节 (__expressionRs)
     */
    private static boolean hasConditionResult(JSONObject stepInfo) {
        if (stepInfo == null) return false;
        String fData = stepInfo.optString("f_data");
        if (fData == null || fData.isEmpty()) return false;
        try {
            JSONObject fDataJson = new JSONObject(fData);
            return fDataJson.has("__expressionRs");
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 从当前节点起，直接展开 stepMap 剩余的所有节点
     * 用于条件节点未确定分支的情况
     */
    private static void appendRemainingSteps(JSONArray path, JSONObject stepMap, Set<String> visited, String startId, int sequence) {
        // 保持 stepMap 遍历顺序
        for (String key : stepMap.keySet()) {
            if (!visited.contains(key)) {
                JSONObject node = stepMap.optJSONObject(key);
                if (node != null) {
                    node.put("sequence", sequence++);
                    path.put(node);
                    visited.add(key);
                }
            }
        }
    }
}
