package com.af.v4.system.common.socket.core.ssl;

import com.af.v4.system.common.socket.config.SocketSslConfig;
import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * SSL 上下文工厂，负责复用 SslContext 并加载 keystore/truststore
 */
public final class SocketSslContextFactory {
    private static final Logger LOGGER = LoggerFactory.getLogger(SocketSslContextFactory.class);
    private static final Map<String, SslContext> SERVER_CACHE = new ConcurrentHashMap<>();
    private static final Map<String, SslContext> CLIENT_CACHE = new ConcurrentHashMap<>();

    private SocketSslContextFactory() {
    }

    public static SslContext forServer(String identifier, SocketSslConfig config) {
        validateConfig(config);
        if (!config.hasKeyStore()) {
            throw new IllegalStateException("Server SSL 需要配置 keyStorePath");
        }
        String cacheKey = "server:" + identifier + ":" + config.identityKey();
        return SERVER_CACHE.computeIfAbsent(cacheKey, key -> buildServerContext(config));
    }

    public static SslContext forClient(String identifier, SocketSslConfig config) {
        validateConfig(config);
        String cacheKey = "client:" + identifier + ":" + config.identityKey();
        return CLIENT_CACHE.computeIfAbsent(cacheKey, key -> buildClientContext(config));
    }

    private static void validateConfig(SocketSslConfig config) {
        if (config == null || !config.isEnabled()) {
            throw new IllegalStateException("未开启 SSL");
        }
    }

    private static SslContext buildServerContext(SocketSslConfig config) {
        try {
            KeyManagerFactory kmf = buildKeyManagerFactory(config);
            SslContextBuilder builder = SslContextBuilder.forServer(kmf);
            if (config.isRequireClientAuth() && config.hasTrustStore()) {
                builder.trustManager(buildTrustManagerFactory(config));
                builder.clientAuth(ClientAuth.REQUIRE);
            } else {
                builder.clientAuth(ClientAuth.NONE);
            }
            return builder.build();
        } catch (Exception e) {
            LOGGER.error("构建服务端 SSL 上下文失败: {}", e.getMessage(), e);
            throw new IllegalStateException("构建服务端 SSL 上下文失败", e);
        }
    }

    private static SslContext buildClientContext(SocketSslConfig config) {
        try {
            SslContextBuilder builder = SslContextBuilder.forClient();
            if (config.isVerifyPeerCertificate()) {
                if (!config.hasTrustStore()) {
                    throw new IllegalStateException("需要校验证书时必须配置 trustStorePath");
                }
                builder.trustManager(buildTrustManagerFactory(config));
            } else {
                builder.trustManager(InsecureTrustManagerFactory.INSTANCE);
            }
            if (config.hasKeyStore()) {
                builder.keyManager(buildKeyManagerFactory(config));
            }
            return builder.build();
        } catch (Exception e) {
            LOGGER.error("构建客户端 SSL 上下文失败: {}", e.getMessage(), e);
            throw new IllegalStateException("构建客户端 SSL 上下文失败", e);
        }
    }

    private static KeyManagerFactory buildKeyManagerFactory(SocketSslConfig config) throws Exception {
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        KeyStore keyStore = loadKeyStore(
                config.getKeyStorePath(),
                config.getKeyStoreType(),
                config.keyStorePasswordChars()
        );
        kmf.init(keyStore, passwordOrNull(config.keyPasswordChars()));
        return kmf;
    }

    private static TrustManagerFactory buildTrustManagerFactory(SocketSslConfig config) throws Exception {
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        KeyStore trustStore = loadKeyStore(
                config.getTrustStorePath(),
                config.getTrustStoreType(),
                config.trustStorePasswordChars()
        );
        tmf.init(trustStore);
        return tmf;
    }

    private static KeyStore loadKeyStore(String location, String type, char[] password) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException {
        Objects.requireNonNull(location, "证书路径不能为空");
        KeyStore keyStore = KeyStore.getInstance(type == null || type.isBlank() ? KeyStore.getDefaultType() : type);
        try (InputStream inputStream = openInputStream(location)) {
            keyStore.load(inputStream, passwordOrNull(password));
        }
        return keyStore;
    }

    private static InputStream openInputStream(String location) throws IOException {
        if (location.startsWith("classpath:")) {
            String resource = location.substring("classpath:".length());
            InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream(resource);
            if (stream == null) {
                throw new IllegalStateException("classpath 下未找到证书: " + resource);
            }
            return stream;
        }
        return Files.newInputStream(Path.of(location));
    }

    private static char[] passwordOrNull(char[] original) {
        if (original == null || original.length == 0) {
            return null;
        }
        return original;
    }

    /**
     * 获取证书库中的所有别名
     */
    public static List<String> getCertificateAliases(SocketSslConfig config, boolean isKeyStore) {
        List<String> aliases = new ArrayList<>();
        try {
            String storePath = isKeyStore ? config.getKeyStorePath() : config.getTrustStorePath();
            String storeType = isKeyStore ? config.getKeyStoreType() : config.getTrustStoreType();
            char[] password = isKeyStore ? config.keyStorePasswordChars() : config.trustStorePasswordChars();
            
            if (storePath == null || storePath.isBlank()) {
                return aliases;
            }
            
            KeyStore keyStore = loadKeyStore(storePath, storeType, password);
            Enumeration<String> enumeration = keyStore.aliases();
            while (enumeration.hasMoreElements()) {
                aliases.add(enumeration.nextElement());
            }
        } catch (Exception e) {
            LOGGER.warn("获取证书别名失败: {}", e.getMessage());
        }
        return aliases;
    }
}

