package com.af.v4.system.common.socket;

import com.af.v4.system.common.socket.config.SocketClientConfigItem;
import com.af.v4.system.common.socket.config.SocketConfig;
import com.af.v4.system.common.socket.config.SocketSslConfig;
import com.af.v4.system.common.socket.core.client.modbus.ModBusClient;
import com.af.v4.system.common.socket.core.client.tcp.TcpClient;
import com.af.v4.system.common.socket.core.client.SocketClient;
import com.af.v4.system.common.socket.core.ssl.SocketSslContextFactory;
import com.af.v4.system.common.socket.enums.SocketTypeEnum;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Component
public class SocketClientManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(SocketClientManager.class);

    private static final Map<String, SocketClientConfigItem> socketClientConfigMap = new HashMap<>();
    private final SocketConfig socketConfig;
    private final List<SocketClient> socketServerList = new ArrayList<>();

    public SocketClientManager(SocketConfig socketConfig) {
        this.socketConfig = socketConfig;
    }

    public static SocketClientConfigItem getSocketConfigItemByPort(String port) {
        return socketClientConfigMap.get(port);
    }

    /**
     * 连接socket服务端
     */
    @PostConstruct
    public void startAll() {
        List<SocketClientConfigItem> socketConfigItemList = socketConfig.getSocketServerClientList();
        if (socketConfigItemList != null) {
            socketConfigItemList.forEach(item -> {
                // 打印SSL配置信息（仅TCP协议）
                if (item.getType() == SocketTypeEnum.TCP) {
                    logSslConfig(item);
                }
                
                SocketClient socketServer = createSocketClient(item);
                socketServerList.add(socketServer);
                socketClientConfigMap.put(item.getAddress(), item);
                if (socketServer != null) {
                    socketServer.connect(item);
                }
            });
        }
    }

    /**
     * 打印SSL配置信息
     */
    private void logSslConfig(SocketClientConfigItem item) {
        SocketSslConfig sslConfig = item.getSslConfig();
        if (sslConfig != null && sslConfig.isEnabled()) {
            if (sslConfig.hasKeyStore()) {
                List<String> keyAliases = SocketSslContextFactory.getCertificateAliases(sslConfig, true);
                LOGGER.info("客户端[{}] SSL已启用 - 密钥库路径: {}, 证书别名: {}", 
                        item.getName(), sslConfig.getKeyStorePath(), 
                        keyAliases.isEmpty() ? "无" : String.join(", ", keyAliases));
            }
            if (sslConfig.hasTrustStore()) {
                List<String> trustAliases = SocketSslContextFactory.getCertificateAliases(sslConfig, false);
                LOGGER.info("客户端[{}] SSL已启用 - 信任库路径: {}, 证书别名: {}", 
                        item.getName(), sslConfig.getTrustStorePath(), 
                        trustAliases.isEmpty() ? "无" : String.join(", ", trustAliases));
            }
            if (!sslConfig.isVerifyPeerCertificate()) {
                LOGGER.warn("客户端[{}] SSL已启用但未校验证书 - 将信任所有证书，存在安全风险", item.getName());
            }
        } else {
            LOGGER.warn("客户端[{}] SSL未启用 - 连接将使用明文传输，存在安全风险", item.getName());
        }
    }

    /**
     * 创建SocketClient实例
     */
    private SocketClient createSocketClient(SocketClientConfigItem item) {
        SocketClient socketClient = null;
        if (item.getType() == SocketTypeEnum.TCP) {
            socketClient = new TcpClient();
        } else if (item.getType() == SocketTypeEnum.MODBUS_TCP) {
            socketClient = new ModBusClient();
        }
        return socketClient;
    }

    @PreDestroy
    public void destroyAll() {
        for (SocketClient socketServer : socketServerList) {
            socketServer.destroy();
        }
    }
}
