package com.af.v4.system.common.socket.core.client;

import cn.hutool.cache.impl.TimedCache;
import com.af.v4.system.common.plugins.core.ConvertTools;
import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.core.channel.ChannelData;
import io.netty.channel.Channel;
import io.netty.channel.ChannelId;
import io.netty.util.HashedWheelTimer;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 客户端管理器
 */
public class ClientManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClientManager.class);

    private static final TimedCache<String, ChannelData> clientChannels;

    static {
        clientChannels = new TimedCache<>(1000 * 60 * 10, new ConcurrentHashMap<>());
    }

    private static final ScopedValue<String> CHANNEL_SCOPED_VALUE = ScopedValue.newInstance();
    private static volatile HashedWheelTimer TIMER;
    private static final ReentrantLock LOCK = new ReentrantLock();

    public static ScopedValue<String> getChannelScopedValue() {
        return CHANNEL_SCOPED_VALUE;
    }

    private static HashedWheelTimer getTimer() {
        if (TIMER == null) {
            LOCK.lock();
            try {
                if (TIMER == null) {
                    TIMER = new HashedWheelTimer();
                }
            } finally {
                LOCK.unlock();
            }
        }

        return TIMER;
    }

    /**
     * 添加channel
     * @param channelId channelId
     * @param channelData channelData
     */
    public static void add(String channelId, ChannelData channelData) {
        clientChannels.put(channelId, channelData);
    }

    /**
     * 发送数据
     *
     * @param message 数据
     */
    public static void send(String message) {
        String channelId = CHANNEL_SCOPED_VALUE.get();
        ChannelData channelData = clientChannels.get(channelId);
        Channel channel = channelData.channel();
        byte[] sendMsgBytes;
        if (channel != null) {
            sendMsgBytes = switch (channelData.configItem().getEncodeType()) {
                case BASE64 -> ConvertTools.base64Decode(message.getBytes());
                case HEX -> ConvertTools.hexStrToByte(message);
                case HEX_BASE64 -> ConvertTools.base64Decode(ConvertTools.hexStrToByte(message));
                default -> message.getBytes();
            };
            channel.writeAndFlush(channel.alloc().buffer().writeBytes(sendMsgBytes));
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("channelId:{}, 服务端回应数据:{}", channelId, message);
            }
        }
    }

    /**
     * 延迟发送数据
     * @param message 数据
     * @param sleepTime 延迟时间
     */
    public static void delayedSend(String message, Integer sleepTime) {
        getTimer().newTimeout((timeout) -> send(message), sleepTime.longValue(), TimeUnit.MILLISECONDS);
    }

    /**
     * 获取配置项
     *
     * @return 配置项
     */
    public static SocketConfigItem getConfig() {
        return getChannelData().configItem();
    }

    /**
     * 获取channelData
     *
     * @return channelData
     */
    public static ChannelData getChannelData() {
        String channelId = CHANNEL_SCOPED_VALUE.get();
        return clientChannels.get(channelId);
    }

    /**
     * 判断channel是否存在
     *
     * @param channelId channelId
     * @return true:存在
     */
    public static boolean isExistChannel(ChannelId channelId) {
        return clientChannels.containsKey(channelId.asLongText());
    }

    /**
     * 获取channel
     *
     * @param channelId channelId
     * @return channel
     */
    public static Channel getChannel(ChannelId channelId) {
        ChannelData channelData = clientChannels.get(channelId.asLongText());
        return channelData.channel();
    }

    /**
     * 移除channel
     *
     * @param channelId channelId
     */
    public static void remove(ChannelId channelId) {
        clientChannels.remove(channelId.asLongText());
    }

    /**
     * 设置全局参数
     *
     * @param key   参数key
     * @param value 值
     */
    public static void setGlobalParams(String key, Object value) {
        ChannelData channelData = ClientManager.getChannelData();
        JSONObject globalParamsItem = channelData.globalParams();
        globalParamsItem.put(key, value);
    }

    /**
     * 获取全局参数
     *
     * @param key 参数key
     */
    public static Object getGlobalParams(String key) {
        ChannelData channelData = ClientManager.getChannelData();
        JSONObject globalParamsItem = channelData.globalParams();
        return globalParamsItem.opt(key);
    }
}
