package com.af.v4.system.common.socket.core.channel;

import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.proxy.logic.ILogicServiceProxy;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.log.enums.OperatorType;
import com.af.v4.system.common.log.service.LogResolveService;
import com.af.v4.system.common.plugins.core.ConvertTools;
import com.af.v4.system.common.socket.SocketServerManager;
import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.enums.MsgTypeEnum;
import com.af.v4.system.common.socket.core.client.ClientManager;
import java.nio.charset.StandardCharsets;
import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.timeout.IdleState;
import io.netty.handler.timeout.IdleStateEvent;
import io.netty.channel.socket.DatagramChannel;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;
import java.net.SocketAddress;

public abstract class AbstractChannelHandler<T> extends SimpleChannelInboundHandler<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractChannelHandler.class);

    @Override
    protected final void channelRead0(ChannelHandlerContext ctx, T msg) {
        String channelId = ctx.channel().id().asLongText();
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("channelId:{}, 收到消息", channelId);
        }
        ScopedValue.where(ClientManager.getChannelScopedValue(), channelId).run(() -> {
            if (ClientManager.getChannelData() == null) {
                initChannelData(ctx);
            }
            // 执行读方法
            read(ctx, msg);
        });
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        String channelId = ctx.channel().id().asLongText();
        SocketAddress socketAddress = ctx.channel().remoteAddress();
        String address;
        if (socketAddress == null) {
            address = "-";
        } else {
            address = socketAddress.toString();
        }
        LOGGER.info("channelId:{}, address:{}, 成功建立连接", channelId, address);
        initChannelData(ctx);
        super.channelActive(ctx);
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        LOGGER.warn("channelId:{}, 触发断线", ctx.channel().id().asLongText());
        if (!(ctx.channel() instanceof DatagramChannel)) {
            try {
                ctx.close();
            } catch (Exception e) {
                LOGGER.error("关闭通道时发生异常", e);
            }
        }
        super.channelInactive(ctx);
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        if (evt instanceof IdleStateEvent idleStateEvent) {
            IdleState state = idleStateEvent.state();
            LOGGER.warn("channelId:{}, 触发空闲事件:{}", ctx.channel().id().asLongText(), state);
            if (!(ctx.channel() instanceof DatagramChannel)) {
                ctx.close();
            }
        }
        super.userEventTriggered(ctx, evt);
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        String channelId = ctx.channel().id().asLongText();
        LOGGER.error("channelId:{} 发生异常", channelId, cause);
        try {
            String plain = STR."ERROR:\{cause.getMessage()}";
            ScopedValue.where(ClientManager.getChannelScopedValue(), channelId).run(() -> {
                MsgTypeEnum encodeType = ClientManager.getConfig().getEncodeType();
                String payload = switch (encodeType) {
                    case HEX -> ConvertTools.byteToHexStr(plain.getBytes(StandardCharsets.UTF_8));
                    case BASE64 -> ConvertTools.base64Encode(plain.getBytes(StandardCharsets.UTF_8));
                    case HEX_BASE64 -> {
                        String b64 = ConvertTools.base64Encode(plain.getBytes(StandardCharsets.UTF_8));
                        yield ConvertTools.byteToHexStr(b64.getBytes(StandardCharsets.UTF_8));
                    }
                    default -> plain;
                };
                ClientManager.send(payload);
            });
        } catch (Exception sendEx) {
            LOGGER.warn("异常通知发送失败, channelId:{}", channelId, sendEx);
        } finally {
            // UDP通道单独处理：只记录日志，不关闭，以避免端口被关掉
            if (!(ctx.channel() instanceof DatagramChannel)) {
                try {
                    ctx.close();
                } catch (Exception e) {
                    LOGGER.error("异常处理阶段关闭通道失败", e);
                }
            }
        }
    }

    protected abstract void read(ChannelHandlerContext ctx, T msg);

    private void initChannelData(ChannelHandlerContext ctx) {
        // 存入管道信息
        Channel channel = ctx.channel();
        String port = String.valueOf(((InetSocketAddress) channel.localAddress()).getPort());
        SocketConfigItem configItem = SocketServerManager.getSocketConfigItemByPort(port);
        ClientManager.add(channel.id().asLongText(), new ChannelData(configItem, channel, new JSONObject()));
    }

    protected Object runBusiness(ByteBuf content, String remoteAddress) {
        long beginTime = System.currentTimeMillis();
        SocketConfigItem configItem = ClientManager.getConfig();
        byte[] data = new byte[content.readableBytes()];
        content.readBytes(data);
        String realData;
        try {
            realData = switch (configItem.getDecodeType()) {
                case HEX -> ConvertTools.byteToHexStr(data);
                default -> ConvertTools.bytesToStr(data);
            };
        } catch (Exception e) {
            LOGGER.error("数据解析失败：", e);
            return null;
        }

        JSONObject params = new JSONObject();
        params.put("value", realData);
        params.put("address", remoteAddress);

        try {
            Object result = SpringUtils.getBean(ILogicServiceProxy.class).run(configItem.getLogicName(), params);
            long endTime = System.currentTimeMillis();
            handleLog(configItem.getLogicName(), params, null, endTime - beginTime);
            return result;
        } catch (Exception e) {
            if (e instanceof LogicException ex) {
                LOGGER.error("请求来源'{}',发生业务异常.\n{}", remoteAddress, ex.getStack());
            } else {
                LOGGER.error("请求来源'{}',发生服务端异常.", remoteAddress, e);
            }
            long endTime = System.currentTimeMillis();
            handleLog(configItem.getLogicName(), params, e, endTime - beginTime);
            throw e;
        }
    }

    protected void handleLog(String logicName, Object operParams, final Throwable e, long costTime) {
        SpringUtils.getBean(LogResolveService.class).resolveLog(
                this.getClass().getName(),
                "runBusiness",
                BusinessType.LOGIC.name(),
                STR."Socket调用Logic：\{logicName}",
                OperatorType.SYSTEM.name(),
                operParams.toString(),
                costTime,
                e
        );
    }

}
