package com.af.v4.system.common.security.service;

import cn.hutool.crypto.SecureUtil;
import com.af.v4.system.api.RemoteUserService;
import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.utils.JwtUtils;
import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.core.utils.ip.IpUtils;
import com.af.v4.system.common.core.utils.uuid.IdUtils;
import com.af.v4.system.common.plugins.core.SecureTools;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.security.config.TokenProperties;
import com.af.v4.system.common.security.utils.SecurityUtils;
import jakarta.servlet.http.HttpServletRequest;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * token验证处理
 */
@Component
public class TokenService {
    private static final Logger LOGGER = LoggerFactory.getLogger(TokenService.class);
    protected static final long MILLIS_SECOND = 1000;

    protected static final long MILLIS_MINUTE = 60 * MILLIS_SECOND;

    private final static String ACCESS_TOKEN = CacheConstants.LOGIN_TOKEN_KEY;

    private final RedisService redisService;
    private final TokenProperties tokenProperties;
    private final RemoteUserService remoteUserService;

    public TokenService(RedisService redisService, TokenProperties tokenProperties,
                        RemoteUserService remoteUserService) {
        this.redisService = redisService;
        this.tokenProperties = tokenProperties;
        this.remoteUserService = remoteUserService;
    }

    /**
     * 创建令牌
     */
    public Map<String, Object> createToken(LoginUser loginUser) {
        String token = IdUtils.fastUUID();
        Long userId = loginUser.getSysUser().getUserId();
        String userName = loginUser.getSysUser().getUserName();
        loginUser.setToken(token);
        loginUser.setUserid(userId);
        loginUser.setUsername(userName);
        loginUser.setIpaddr(IpUtils.getIpAddr());
        loginUser.setSessionEncryptKey(SecureUtil.sha256(token).substring(0, 32));
        refreshToken(loginUser);

        // 异步记录会话信息
        recordSessionAsync(loginUser);

        // Jwt存储信息
        Map<String, Object> claimsMap = new HashMap<>(3);
        claimsMap.put(SecurityConstants.USER_KEY, token);
        claimsMap.put(SecurityConstants.DETAILS_USER_ID, userId);
        claimsMap.put(SecurityConstants.DETAILS_USERNAME, userName);

        // 接口返回信息
        Map<String, Object> rspMap = new HashMap<>(3);
        rspMap.put("access_token", JwtUtils.createToken(claimsMap));
        rspMap.put("expires_in", getExpirationTime(loginUser)); // 根据用户类型返回不同过期时间
        rspMap.put("session", SecureTools.RSAEncrypt(loginUser.getSessionEncryptKey(), CacheConstants.RSA_PUBLIC_KEY));
        rspMap.put("resources", loginUser.getResources());
        return rspMap;
    }

    /**
     * 获取用户身份信息
     *
     * @return 用户信息
     */
    public LoginUser getLoginUser() {
        return getLoginUser(ServletUtils.getRequest());
    }

    /**
     * 获取用户身份信息
     *
     * @return 用户信息
     */
    public LoginUser getLoginUser(HttpServletRequest request) {
        // 获取请求携带的令牌
        String token = SecurityUtils.getToken(request);
        return getLoginUser(token);
    }

    /**
     * 获取用户身份信息
     *
     * @return 用户信息
     */
    public LoginUser getLoginUser(String token) {
        LoginUser user;
        try {
            if (StringUtils.isNotEmpty(token)) {
                String userKey = JwtUtils.getUserKey(token);
                if (userKey == null) {
                    return null;
                }
                user = redisService.get(getTokenKey(userKey));
                return user;
            }
        } catch (Exception e) {
            LOGGER.error("TOKEN({})获取用户信息异常：{}", token, e.getMessage());
        }
        return null;
    }

    /**
     * 设置用户身份信息
     */
    public void setLoginUser(LoginUser loginUser) {
        if (StringUtils.isNotNull(loginUser) && StringUtils.isNotEmpty(loginUser.getToken())) {
            refreshToken(loginUser);
        }
    }

    /**
     * 删除用户缓存信息
     */
    public void delLoginUser(String token) {
        if (StringUtils.isNotEmpty(token)) {
            String userKey = JwtUtils.getUserKey(token);
            if (userKey == null) {
                return;
            }
            redisService.delete(getTokenKey(userKey));
            redisService.delete(getSecureKey(userKey));
        }
    }

    /**
     * 验证令牌有效期，相差不足指定刷新时间，自动刷新缓存
     *
     * @param loginUser 登录信息
     */
    public void verifyToken(LoginUser loginUser) {
        long expireTime = loginUser.getExpireTime();
        long currentTime = System.currentTimeMillis();
        long refreshThresholdMillis = tokenProperties.getRefreshTime() * MILLIS_MINUTE;
        if (expireTime - currentTime <= refreshThresholdMillis) {
            refreshToken(loginUser);
        }
    }

    /**
     * 刷新令牌有效期
     *
     * @param loginUser 登录信息
     */
    public void refreshToken(LoginUser loginUser) {
        loginUser.setLoginTime(System.currentTimeMillis());

        // 根据用户类型获取不同的过期时间
        Long expirationMinutes = getExpirationTime(loginUser);
        loginUser.setExpireTime(loginUser.getLoginTime() + expirationMinutes * MILLIS_MINUTE);
        // 使用 Long 重载方法，直接传递秒数，避免类型转换问题
        Long expireSeconds = expirationMinutes * 60L;
        // 根据uuid将loginUser缓存
        String userKey = getTokenKey(loginUser.getToken());
        redisService.set(userKey, loginUser, expireSeconds);
        LOGGER.warn("根据uuid将loginUser缓存 userKey {} ：{}", userKey, expireSeconds);
        // 根据uuid缓存登录安全信息
        String secureKey = getSecureKey(loginUser.getToken());
        redisService.set(secureKey, loginUser.getSessionEncryptKey(), expireSeconds);
    }

    /**
     * 根据用户类型获取Token过期时间
     *
     * @param loginUser 登录用户
     * @return 过期时间（分钟）
     */
    private Long getExpirationTime(LoginUser loginUser) {
        // 外部用户使用较短的过期时间（默认2小时）
        if (loginUser.isExternalUser()) {
            return tokenProperties.getExternalUserExpiration();
        }
        // 系统用户使用默认过期时间（默认7天）
        return tokenProperties.getExpiration();
    }

    private String getTokenKey(@NonNull String token) {
        return ACCESS_TOKEN + token;
    }

    private String getSecureKey(String token) {
        return CacheConstants.SESSION_SECURE_KEY + token;
    }

    /**
     * 异步记录会话信息
     *
     * @param loginUser 登录用户信息
     */
    @Async
    protected void recordSessionAsync(LoginUser loginUser) {
        try {
            LOGGER.debug("开始记录用户会话: token={}, user={}", loginUser.getToken(), loginUser.getUsername());

            // 构建会话数据JSON
            JSONObject sessionData = new JSONObject();

            // 基本信息
            sessionData.put("token_key", loginUser.getToken());
            sessionData.put("auth_id", String.valueOf(loginUser.getUserid()));
            sessionData.put("auth_identifier", loginUser.getUsername());
            sessionData.put("auth_type", loginUser.isExternalUser() ? "EXTERNAL_USER" : "SYSTEM_USER");
            sessionData.put("login_ip", loginUser.getIpaddr());
            sessionData.put("status", 1); // 在线状态

            // 时间信息
            LocalDateTime now = LocalDateTime.now();
            sessionData.put("login_time", now.toString());
            sessionData.put("last_access_time", now.toString());

            // 计算过期时间
            long expireMinutes = getExpirationTime(loginUser);
            LocalDateTime expireTime = now.plusMinutes(expireMinutes);
            sessionData.put("expire_time", expireTime.toString());

            // 解析用户代理信息
            HttpServletRequest httpRequest = ServletUtils.getRequest();
            if (httpRequest != null) {
                String userAgent = httpRequest.getHeader("User-Agent");
                if (StringUtils.isNotEmpty(userAgent)) {
                    parseUserAgent(userAgent, sessionData);
                } else {
                    sessionData.put("browser", "Unknown");
                    sessionData.put("os", "Unknown");
                    sessionData.put("client_type", "WEB");
                }
            } else {
                sessionData.put("browser", "Unknown");
                sessionData.put("os", "Unknown");
                sessionData.put("client_type", "WEB");
            }

            // 租户信息
            sessionData.put("tenant_id", 0);

            // 调用远程服务记录会话
            remoteUserService.recordSession(sessionData.toString(), SecurityConstants.INNER);

            LOGGER.debug("会话记录完成: token={}", loginUser.getToken());

        } catch (Exception e) {
            LOGGER.error("记录会话信息时发生异常: token={}, user={}",
                    loginUser.getToken(), loginUser.getUsername(), e);
        }
    }

    /**
     * 简单解析User-Agent信息
     *
     * @param userAgent   User-Agent字符串
     * @param sessionData 会话数据JSON
     */
    private void parseUserAgent(String userAgent, JSONObject sessionData) {
        try {
            // 简单的浏览器检测
            String browser = "Unknown";
            String os = "Unknown";
            String clientType = "WEB";

            userAgent = userAgent.toLowerCase();

            // 检测浏览器
            if (userAgent.contains("chrome")) {
                browser = "Chrome";
            } else if (userAgent.contains("firefox")) {
                browser = "Firefox";
            } else if (userAgent.contains("safari") && !userAgent.contains("chrome")) {
                browser = "Safari";
            } else if (userAgent.contains("edge")) {
                browser = "Edge";
            } else if (userAgent.contains("opera")) {
                browser = "Opera";
            }

            // 检测操作系统
            if (userAgent.contains("windows")) {
                os = "Windows";
            } else if (userAgent.contains("mac")) {
                os = "macOS";
            } else if (userAgent.contains("linux")) {
                os = "Linux";
            } else if (userAgent.contains("android")) {
                os = "Android";
                clientType = "MOBILE";
            } else if (userAgent.contains("iphone") || userAgent.contains("ipad")) {
                os = "iOS";
                clientType = "MOBILE";
            }

            sessionData.put("browser", browser);
            sessionData.put("os", os);
            sessionData.put("client_type", clientType);

        } catch (Exception e) {
            LOGGER.warn("解析User-Agent失败: {}", userAgent, e);
            sessionData.put("browser", "Unknown");
            sessionData.put("os", "Unknown");
            sessionData.put("client_type", "WEB");
        }
    }
}
