package com.af.v4.system.common.resource.mapper;

import cn.hutool.core.lang.Tuple;
import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.proxy.liuli.ILiuLiConfigServiceProxy;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.plugins.io.IOTools;
import com.af.v4.system.common.resource.enums.ResourceType;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import jakarta.annotation.PostConstruct;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.json.JSONArray;
import org.json.JSONObject;
import org.jspecify.annotations.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * 抽象资源映射器
 */
@Component
public abstract class AbstractResourceMapper<T extends AbstractResourceMapper.CommonResource> {
    /**
     * 租户模块名称
     */
    public static final String TENANT_MODULE_NAME = "tenants";
    /**
     * 云端数据的兼容路径
     */
    public static final String DEFAULT_CLOUD_DATA_PATH_VALUE = "cloudData";
    protected static final Logger LOGGER = LoggerFactory.getLogger(AbstractResourceMapper.class);
    /**
     * 是否为单元测试环境
     */
    private static final boolean IS_JUNIT_TEST;
    /**
     * 资源缓存
     */
    private static final Map<ResourceType, Cache<@NonNull String, String>> CACHE_MAP = new EnumMap<>(ResourceType.class);

    /**
     * 资源列表
     */
    protected volatile Map<String, T> allResourceMappingInfoMap;

    /**
     * 用于保护资源映射写入的锁
     */
    private final Lock resMappingLock = new ReentrantLock();

    protected final ILiuLiConfigServiceProxy liuLiConfigServiceProxy;
    protected final ApplicationService applicationService;
    private final ModuleMapper moduleMapper;

    static {
        IS_JUNIT_TEST = isJUnitTest();
        for (ResourceType type : ResourceType.values()) {
            CACHE_MAP.put(type, Caffeine.newBuilder().maximumSize(100).build());
        }
    }

    protected AbstractResourceMapper(ModuleMapper moduleMapper, ApplicationService applicationService, ILiuLiConfigServiceProxy liuLiConfigServiceProxy) {
        this.moduleMapper = moduleMapper;
        this.applicationService = applicationService;
        this.liuLiConfigServiceProxy = liuLiConfigServiceProxy;
    }

    /**
     * 是否为junit环境
     *
     * @return 判断结果
     */
    private static boolean isJUnitTest() {
        for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
            if (element.getClassName().startsWith("org.junit.")) {
                return true;
            }
        }
        return false;
    }

    /**
     * 获取资源类型
     *
     * @return 资源类型
     */
    public abstract ResourceType getResType();

    /**
     * 获取资源文件名
     *
     * @return 资源文件名
     */
    public abstract String getFileName();

    /**
     * 获取资源文件夹名
     *
     * @return 资源文件夹名
     */
    public abstract String getFolderName();

    /**
     * 加载资源列表
     */
    @PostConstruct
    protected void loadMap() {
        if (allResourceMappingInfoMap == null) {
            allResourceMappingInfoMap = new LinkedHashMap<>(128);
            putMapByRoot();
            moduleMapper.getMap().forEach((key, value) -> {
                String path = value.get("path");
                JSONArray mappingVisualModules;
                String mappingVisualModulesStr = value.get("mappingVisualModules");
                if (mappingVisualModulesStr != null) {
                    mappingVisualModules = new JSONArray(mappingVisualModulesStr);
                } else {
                    mappingVisualModules = null;
                }
                try {
                    putMapByModule(key, path, mappingVisualModules);
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
        }
    }

    /**
     * 加载根目录资源到列表
     */
    private void putMapByRoot() {
        putMapByModule(null, null, null);
    }

    /**
     * 加载模块资源到列表
     *
     * @param name   模块名
     * @param parent 父模块名
     */
    private void putMapByModule(String name, String parent, JSONArray mappingVisualModules) {
        String str;
        String fileName = getFileName();
        ResourceType resType = getResType();
        if (parent == null) {
            if (name == null) {
                str = fileName;
            } else {
                str = name + "/" + fileName;
            }
        } else {
            str = parent + "/" + name + "/" + fileName;
        }
        IOTools.getStream(str, stream -> {
            SAXReader reader = new SAXReader();
            Document document;
            try {
                document = reader.read(stream);
            } catch (DocumentException e) {
                throw new RuntimeException(e);
            }
            Element root = document.getRootElement();
            for (Iterator<Element> item = root.elementIterator(resType.getValue()); item.hasNext(); ) {
                Element element = item.next();
                buildLocalRes(element, name, parent, mappingVisualModules);
            }
        });
    }

    /**
     * 获取资源Map
     *
     * @return 全部资源
     */
    public final Map<String, T> getAllMap() {
        return allResourceMappingInfoMap;
    }

    /**
     * 根据Key获取资源
     *
     * @param key                                KEY
     * @param isThrowExceptionOnResourceNotFound 资源找不到时是否抛出异常
     * @param enableLiuliConfig                  启用从琉璃中心获取配置
     * @return 资源
     */
    public final T getResource(String key, Boolean isThrowExceptionOnResourceNotFound, Boolean enableLiuliConfig) {
        String[] candidateAlias = getCandidateAliasList(key);
        String tenantName = applicationService.getTenantName();
        boolean isTenant = !(tenantName.equals(CacheConstants.TENANT_DEFAULT_VALUE) ||
                tenantName.equals(CacheConstants.TENANT_TEST_VALUE));
        T result = null;
        for (String alias : candidateAlias) {
            // 寻找本地租户资源
            if (isTenant) {
                String aliasItem = alias + "@" + tenantName;
                if (allResourceMappingInfoMap.containsKey(aliasItem)) {
                    result = allResourceMappingInfoMap.get(aliasItem);
                    break;
                }
            }
            // 寻找公共资源
            if (allResourceMappingInfoMap.containsKey(alias)) {
                result = allResourceMappingInfoMap.get(alias);
                break;
            }
            // 从琉璃获取资源
            if (!enableLiuliConfig) {
                continue;
            }
            T res = getResourceByLiuli(alias);
            if (res != null) {
                result = res;
                break;
            }
        }
        if (result != null) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("资源信息：{}", result.getJsonParams());
            }
            return result;
        } else {
            if (isThrowExceptionOnResourceNotFound) {
                throw new ServiceException(getResType() + "资源[" + key + "]未找到", HttpStatus.BAD_REQUEST);
            } else {
                return null;
            }
        }
    }

    /**
     * 从琉璃中心获取资源
     *
     * @param alias 资源别名
     * @return 资源
     */
    private T getResourceByLiuli(String alias) {
        JSONObject liuliContent = liuLiConfigServiceProxy.get(alias);
        if (liuliContent == null) {
            return null;
        }
        // 先从资源缓存取，如果版本没有更新，直接返回资源，不再构建
        String liuLiResKey = "liuliRes-" + alias;
        String resTenantName = liuliContent.getJSONObject("$globalProp").getString("tenantAlias");
        if (!resTenantName.equals(CacheConstants.TENANT_DEFAULT_VALUE)) {
            liuLiResKey = liuLiResKey + "@" + resTenantName;
        }
        // 对比并获取资源
        Tuple tuple = reuseResourceObject(allResourceMappingInfoMap.get(liuLiResKey), liuliContent, alias);
        // 更新本地缓存
        T resource = tuple.get(1);
        if (tuple.get(0)) {
            this.safeAddResMapping(liuLiResKey, resource);
        }
        return resource;
    }

    public final T getResource(String key) {
        return getResource(key, true, true);
    }

    /**
     * 判断资源是否存在
     *
     * @param key KEY
     * @return 是否存在
     */
    public final boolean hasResource(String key) {
        if (key == null) {
            LOGGER.error("资源名不能为null");
            return false;
        }
        return Optional.ofNullable(getResource(key, false, true)).isPresent();
    }

    /**
     * 获取候选别名集合
     *
     * @return 候选别名集合
     */
    protected String[] getCandidateAliasList(String key) {
        return new String[]{key};
    }

    /**
     * 获取资源实际Key
     *
     * @param element          element
     * @param moduleName       模块名
     * @param parentModuleName 父级模块名
     */
    private String getResRealKey(Element element, String moduleName, String parentModuleName) {
        if (moduleName == null) {
            moduleName = "";
        }
        String key = element.attribute("alias").getValue();
        // 如果是单元测试环境，验证模块是否为单元测试模块，如果是，资源别名增加test_前缀
        if (IS_JUNIT_TEST && moduleName.startsWith("test_")) {
            key = "test_" + key;
        }
        // 如果是非二进制的租户资源，修改注册的别名
        if (!isBinaryResource() && Objects.equals(parentModuleName, TENANT_MODULE_NAME)) {
            key = key + "@" + moduleName;
        }
        return key;
    }

    /**
     * 构建本地资源
     *
     * @param element              XML元素
     * @param moduleName           模块名
     * @param parentModuleName     父级模块名
     * @param mappingVisualModules 映射虚拟模块列表
     */
    private void buildLocalRes(Element element, String moduleName, String parentModuleName, JSONArray mappingVisualModules) {
        String key = getResRealKey(element, moduleName, parentModuleName);
        String resPath = getResPath(element, moduleName, parentModuleName);
        // 存入资源映射对象
        if (resPath != null) {
            Map<String, Object> attrMap = element.attributes().stream()
                    .collect(Collectors.toMap(Attribute::getName, Attribute::getValue));
            T res = buildResource(moduleName, key, resPath, null, attrMap);
            this.addResMapping(key, res);
            // 对虚拟模块增加资源映射
            if (mappingVisualModules != null) {
                for (Object mappingVisualModule : mappingVisualModules) {
                    String mappingKey = getResRealKey(element, mappingVisualModule.toString(), parentModuleName);
                    this.addResMapping(mappingKey, res);
                }
            }
        }
    }

    /**
     * 存入资源映射对象
     *
     * @param key      资源key
     * @param resource 资源对象
     */
    private void addResMapping(String key, T resource) {
        allResourceMappingInfoMap.put(key, resource);
    }

    /**
     * 安全存入资源映射对象
     *
     * @param key      资源key
     * @param resource 源对象
     */
    private void safeAddResMapping(String key, T resource) {
        resMappingLock.lock();
        try {
            addResMapping(key, resource);
        } finally {
            resMappingLock.unlock();
        }
    }

    /**
     * 获取资源内容
     *
     * @return 资源内容
     */
    private String getValueByResource(CommonResource resource) {
        if (isBinaryResource()) {
            throw new ServiceException("二进制资源类型[" + getResType() + "]不支持获取资源内容");
        }
        String filePath = resource.getPath();
        // 琉璃配置不缓存内容，直接从resource中获取
        if (filePath.equals(DEFAULT_CLOUD_DATA_PATH_VALUE)) {
            return resource.getLiuLiValue().getString("source");
        }
        // 非生产环境不从缓存获取
        if (applicationService.getEnvType() != EnvType.PROD) {
            return loadValueByResource(resource);
        }
        return getCache().get(resource.getAlias(), (_) -> loadValueByResource(resource));
    }

    /**
     * 加载资源内容
     *
     * @return 资源内容
     */
    private String loadValueByResource(CommonResource resource) {
        String filePath = resource.getPath();
        String source = IOTools.readText(filePath);
        // 处理回车换行
        source = source.replace("\r\n", "\n");
        source = this.rewriteSource(source);
        return source;
    }


    /**
     * 获取资源路径
     *
     * @param element          XML元素
     * @param moduleName       模块名
     * @param parentModuleName 父级模块名
     */
    protected String getResPath(Element element, String moduleName, String parentModuleName) {
        Attribute pathAttribute = element.attribute("path");
        String path = pathAttribute.getValue();
        String folderName = getFolderName();
        return (parentModuleName == null ?
                moduleName + "/" + folderName + "/" :
                parentModuleName + "/" + moduleName + "/" + folderName + "/") + path;
    }

    /**
     * 构建资源映射
     *
     * @param moduleName 模块名
     * @param key        资源Key
     * @param path       资源路径
     * @param attrMap    attrMap
     * @return 资源映射对象
     */
    private T buildResource(String moduleName, String key, String path, JSONObject liuliValue, Map<String, Object> attrMap) {
        T resource = buildResourceImpl(moduleName, key, path, liuliValue, attrMap);
        // 验证别名是否重复
        if (!path.equals(DEFAULT_CLOUD_DATA_PATH_VALUE) && allResourceMappingInfoMap.containsKey(key)) {
            T t = allResourceMappingInfoMap.get(key);
            ResourceType resType = getResType();
            // 对于相同路径的二进制资源，允许覆盖注册
            if (isBinaryResource() && t.getPath().equals(resource.getPath())) {
                return resource;
            }
            throw new ServiceException(resType + "资源[" + key + "]重复注册：已存在资源：模块[" + t.getModuleName() + "],路径[" + t.getPath() + "];待注册资源：模块[" + resource.getModuleName() + "],路径[" + resource.getPath() + "];", HttpStatus.CONFIG_ERROR);
        }
        return resource;
    }

    /**
     * 构建资源映射具体实现
     *
     * @param moduleName 模块名
     * @param key        资源Key
     * @param path       资源路径
     * @param liuliValue 资源为云端资源时，得到的资源内容
     * @param attrMap    attrMap
     * @return 资源映射对象
     */
    protected abstract T buildResourceImpl(String moduleName, String key, String path, JSONObject liuliValue, Map<String, Object> attrMap);

    /**
     * 重写源
     *
     * @param source 源内容
     * @return 重写后结果
     */
    protected String rewriteSource(String source) {
        if (isBinaryResource()) {
            throw new ServiceException("二进制资源类型[" + getResType() + "]不支持重写源");
        }
        return source;
    }

    /**
     * 获取资源缓存对象
     */
    private Cache<@NonNull String, String> getCache() {
        return CACHE_MAP.get(getResType());
    }

    /**
     * 是否为二进制资源
     */
    protected boolean isBinaryResource() {
        return false;
    }

    /**
     * 是否支持编译
     */
    public boolean isSupportCompile() {
        return true;
    }

    /**
     * 数据源资源支持
     */
    public interface DataSourceResourceSupport {

        String getDataSource();
    }


    /**
     * 移动端资源支持
     */
    public interface MobileResourceSupport {

        boolean isMobile();
    }

    /**
     * 复用资源对象以减少内存占用
     * 由于琉璃资源的继承机制，不同子租户可能会获取到相同的父租户资源
     * 通过比较版本号，复用内存中已存在的资源对象，避免重复创建相同的资源对象
     * 如果新资源版本更高，则使用新资源；否则复用已有资源
     *
     * @param resource     内存中已存在的资源对象
     * @param liuliContent 新获取到的配置内容
     * @param alias        资源别名
     * @return [0] 如果新资源版本较新则返回true，否则返回false
     * [1] 复用后的资源对象
     */
    private Tuple reuseResourceObject(T resource, JSONObject liuliContent, String alias) {
        JSONObject globalProp = liuliContent.getJSONObject("$globalProp");
        Supplier<T> func = () -> {
            // 为琉璃资源构建映射
            return this.buildResource(globalProp.optString("module", null),
                    alias,
                    DEFAULT_CLOUD_DATA_PATH_VALUE,
                    liuliContent,
                    liuliContent.getJSONObject("$configProp").toMap());
        };
        if (resource == null) {
            return new Tuple(true, func.get());
        }
        // 获取已存在资源的版本号,如果不存在则默认为0
        int existingVersion = Integer.parseInt(resource.getLiuLiValue().optString("version", "0"));
        // 获取新资源的版本号,如果不存在则默认为1
        int newVersion = Integer.parseInt(globalProp.optString("version", "1"));

        // 返回版本号较大的配置对象
        // 如果新配置版本号大于已存在配置,则构建新配置
        // 否则继续使用已存在的配置,避免重复创建相同的配置对象
        boolean isNew = newVersion > existingVersion;
        return new Tuple(isNew, isNew ? func.get() : resource);
    }

    /**
     * 资源结果集
     */
    public static class CommonResource {
        /**
         * 解析器
         */
        private final AbstractResourceMapper<?> mapper;
        /**
         * 所属模块名
         */
        private final String moduleName;
        /**
         * 资源别名
         */
        private final String alias;
        /**
         * 资源路径
         */
        private final String path;
        /**
         * 云资源信息
         */
        private final JSONObject liuLiValue;


        CommonResource(AbstractResourceMapper<? extends CommonResource> mapper, String moduleName, String alias, String path, JSONObject liuLiValue) {
            this.mapper = mapper;
            this.moduleName = moduleName;
            this.alias = alias;
            this.path = path;
            this.liuLiValue = liuLiValue;
        }


        public String getAlias() {
            return alias;
        }

        public String getPath() {
            return path;
        }

        public String getModuleName() {
            return moduleName;
        }

        public JSONObject getLiuLiValue() {
            return liuLiValue;
        }

        /**
         * 获取JSON内容
         *
         * @return JSON内容
         */
        public JSONObject getJsonParams() {
            return new JSONObject()
                    .put("alias", getAlias())
                    .put("path", getPath())
                    .put("moduleName", getModuleName())
                    .put("resType", mapper.getResType().getValue())
                    .put("liuliValue", liuLiValue);
        }

        /**
         * 获取资源内容
         *
         * @return 资源内容
         */
        public String getSource() {
            return mapper.getValueByResource(this);
        }
    }
}
