package com.af.v4.system.common.resource.mapper;

import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.plugins.io.IOTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Stream;

/**
 * 模块映射器
 *
 * @author Mr.river
 */
@Component
public class ModuleMapper {
    private static final Logger LOGGER = LoggerFactory.getLogger(ModuleMapper.class);

    /**
     * 模块映射文件名称
     */
    private static final String MODULE_FILE_NAME = "module.xml";

    /**
     * 模块扩展资源文件名称
     */
    private static final String EXTRA_RESOURCE_FILE_NAME = "/resource.json";

    /**
     * 模块属性：名称
     */
    private static final String MODE_NAME = "module";

    /**
     * 模块属性：导入
     */
    private static final String IMPORT_NAME = "import";

    /**
     * 基础租户目录路径
     */
    private static final String BASE_TENANT_PATH = STR."/\{AbstractResourceMapper.TENANT_MODULE_NAME}/";
    /**
     * 基础租户目录配置路径
     */
    private static final String BASE_TENANT_CONFIG_PATH = "/tenants.json";
    private static Map<String, Map<String, String>> map;

    private final ApplicationService applicationService;

    public ModuleMapper(ApplicationService applicationService) {
        this.applicationService = applicationService;
        map = new LinkedHashMap<>(8);
        loadModuleBootstrap(MODULE_FILE_NAME, true);
    }

    /**
     * 获取所有模块映射文件
     */
    public Map<String, Map<String, String>> getMap() {
        return map;
    }

    /**
     * 遍历租户配置追加模块
     */
    private void parseTenantModule() {
        if (applicationService.getEnvType() != EnvType.LOCAL) {
            try {
                JsonTools.readJsonArrayFile(BASE_TENANT_CONFIG_PATH).forEach(moduleObj -> {
                    String moduleName = moduleObj.toString();
                    addTenantModule(moduleName);
                });
            } catch (Exception e) {
                LOGGER.error("通过读取JSON配置加载租户模块失败", e);
            }
        } else {
            try {
                URL url = ModuleMapper.class.getResource(BASE_TENANT_PATH);
                if (url != null) {
                    Path dir = Paths.get(url.toURI());
                    try (Stream<Path> stream = Files.list(dir)) {
                        stream.filter(Files::isDirectory)
                                .map(Path::getFileName)
                                .map(Path::toString)
                                .forEach(this::addTenantModule);
                    }
                }
            } catch (IOException e) {
                LOGGER.error("通过读取目录加载租户模块失败", e);
            } catch (URISyntaxException e) {
                LOGGER.error("URI语法错误", e);
            }
        }
    }

    /**
     * 增加租户模块
     *
     * @param moduleName 模块名
     */
    private void addTenantModule(String moduleName) {
        Map<String, String> module = new HashMap<>(3);
        module.put("path", AbstractResourceMapper.TENANT_MODULE_NAME);
        try {
            // 读取模块资源扩展文件
            JSONObject resourceJson = JsonTools.readJsonFile(BASE_TENANT_PATH + moduleName + EXTRA_RESOURCE_FILE_NAME);
            JSONArray mappingVisualModules = resourceJson.optJSONArray("mappingVisualModules");
            if (mappingVisualModules != null) {
                // 添加虚拟模块映射
                String mappingVisualModulesStr = mappingVisualModules.toString();
                module.put("mappingVisualModules", mappingVisualModulesStr);
                LOGGER.info(">>> 模块：[{}]加载虚拟映射模块{}", moduleName, mappingVisualModulesStr);
            }
        } catch (RuntimeException ignore) {}
        map.put(moduleName, module);
        LOGGER.info(">>> 模块：[{}]加载完成", moduleName);
    }

    /**
     * 遍历依赖模块
     *
     * @param root 根节点
     */
    private void parseImportModule(Element root) {
        List<Element> ite = root.elements(IMPORT_NAME);
        if (!ite.isEmpty()) {
            for (Element elm : ite) {
                String name = elm.attribute("name").getValue();
                LOGGER.info("====从模块集[{}]导入模块====", name);
                loadModuleBootstrap(STR."\{name}/\{MODULE_FILE_NAME}", false);
            }
        }
    }

    /**
     * 遍历项目模块
     */
    private void parseProjectModule(Element root) {
        for (Iterator<Element> it = root.elementIterator(MODE_NAME); it.hasNext(); ) {
            Element elm = it.next();
            String name = elm.attribute("name").getValue();
            String upload = elm.attributeValue("upload");
            String path = elm.attributeValue("path");
            // 设置module属性
            Map<String, String> module = new HashMap<>(3);
            module.put("upload", upload);
            module.put("path", path);
            map.put(name, module);
            LOGGER.info(">>> 模块：[{}]加载完成", name);
        }
    }

    private void loadModuleBootstrap(String path, boolean isRoot) {
        IOTools.getStream(path, stream -> {
            try {
                SAXReader reader = new SAXReader();
                Document document = reader.read(stream);
                Element root = document.getRootElement();
                parseImportModule(root);
                if (isRoot) {
                    LOGGER.info("====从租户目录导入模块====");
                    parseTenantModule();
                    LOGGER.info("====从本地目录导入模块====");
                }
                parseProjectModule(root);
            } catch (DocumentException e) {
                throw new RuntimeException(e);
            }
        }, errorPath -> {
            throw new ServiceException(STR."模块定义文件[module.xml]未找到，相关路径：\{errorPath}", HttpStatus.CONFIG_ERROR);
        });
    }
}
