package com.af.v4.system.common.redis.entity;

import org.json.JSONObject;

import java.io.Serializable;
import java.util.UUID;

/**
 * @Author: 小六
 * @Description: Redis 队列消息通用实体，统一使用 JSONObject 作为消息体
 */
public class FastCacheQueueEntity implements Serializable {

    /**
     * 消息唯一ID，每次入队时自动生成
     */
    private final String id;

    /**
     * 当前重试次数
     */
    private final int currRetryTimes;

    /**
     * 最大重试次数，默认3次
     */
    private final int maxRetryTimes;

    /**
     * 入队时间戳（毫秒）
     */
    private final long timestamp;

    /**
     * 实际业务数据，泛型在定义使用配置方式启动服务时创建队列时候会有问题，所以这里统一为 JSONObject
     */
    private final JSONObject data;

    private FastCacheQueueEntity(Builder builder) {
        this.id = builder.id != null ? builder.id : UUID.randomUUID().toString();
        this.timestamp = builder.timestamp > 0 ? builder.timestamp : System.currentTimeMillis();
        this.currRetryTimes = builder.currRetryTimes;
        this.maxRetryTimes = builder.maxRetryTimes;
        this.data = builder.data;
    }

    public String getId() {
        return id;
    }

    public int getCurrRetryTimes() {
        return currRetryTimes;
    }

    public int getMaxRetryTimes() {
        return maxRetryTimes;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public JSONObject getData() {
        return data;
    }

    @Override
    public String toString() {
        return STR."FastCacheQueueEntity --> {id='\{id}', currRetryTimes=\{currRetryTimes}, maxRetryTimes=\{maxRetryTimes}, timestamp=\{timestamp}, data=\{data}}";
    }

    /**
     * Builder 模式构建消息实体
     */
    public static class Builder {
        private String id;
        private int currRetryTimes = 0;
        private int maxRetryTimes = 3;
        private long timestamp;
        private JSONObject data;

        public Builder data(JSONObject data) {
            this.data = data;
            return this;
        }

        public Builder currRetryTimes(int currRetryTimes) {
            this.currRetryTimes = currRetryTimes;
            return this;
        }

        public Builder maxRetryTimes(int maxRetryTimes) {
            this.maxRetryTimes = maxRetryTimes;
            return this;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder timestamp(long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        /**
         * 构建实体对象，若未指定 id 和 timestamp 将自动赋值
         */
        public FastCacheQueueEntity build() {
            return new FastCacheQueueEntity(this);
        }
    }
}
