package com.af.v4.system.common.redis.utils;

import com.af.v4.system.common.redis.exception.LockInterruptedException;
import com.af.v4.system.common.redis.exception.LockTimeoutException;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * Redis分布式锁工具
 */
@Component
public class DistributedLockUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(DistributedLockUtil.class);

    // 默认等待时间（秒）
    private static final int DEFAULT_WAIT_TIME = 5;
    // 默认锁持有时间（秒）
    private static final int DEFAULT_LEASE_TIME = 20;

    private final RedissonClient redissonClient;

    public DistributedLockUtil(RedissonClient redissonClient) {
        this.redissonClient = redissonClient;
    }

    /**
     * 获取分布式锁并执行操作（阻塞模式）。
     * <p>
     * 当前线程将阻塞直到成功获取锁，锁的最大持有时间由 leaseTime 参数决定。适用于需要确保操作
     * 必须获取锁的场景，例如长时间任务或关键业务逻辑。
     *
     * @param key       锁的唯一标识
     * @param leaseTime 锁的最大持有时间（单位：秒）
     * @param supplier  提供具体操作逻辑的函数式接口
     * @param <T>       返回值的类型
     * @return supplier 提供的逻辑的执行结果
     */
    public <T> T syncLock(String key, Integer leaseTime, Supplier<T> supplier) throws LockInterruptedException {
        try {
            return executeWithLock(key, null, leaseTime, supplier, true);
        } catch (LockTimeoutException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 尝试获取分布式锁并执行操作（非阻塞模式）。
     * <p>
     * 当前线程将在 waitTime 时间内尝试获取锁。如果在此时间内未能获取锁，则抛出异常。
     * 锁的最大持有时间由 leaseTime 参数决定。适用于需要快速失败的场景。
     *
     * @param key       锁的唯一标识
     * @param waitTime  获取锁的最大等待时间（单位：秒）
     * @param leaseTime 锁的最大持有时间（单位：秒）
     * @param supplier  提供具体操作逻辑的函数式接口
     * @param <T>       返回值的类型
     * @return supplier 提供的逻辑的执行结果
     */
    public <T> T lock(String key, Integer waitTime, Integer leaseTime, Supplier<T> supplier) throws LockTimeoutException, LockInterruptedException {
        return executeWithLock(key, waitTime, leaseTime, supplier, false);
    }

    /**
     * 默认阻塞分布式锁
     *
     * @param key      锁的唯一标识
     * @param supplier 提供具体操作逻辑的函数式接口
     * @param <T>      返回值的类型
     * @return supplier 提供的逻辑的执行结果
     */
    public <T> T syncLock(String key, Supplier<T> supplier) throws LockInterruptedException {
        return syncLock(key, DEFAULT_LEASE_TIME, supplier);
    }

    /**
     * 默认非阻塞分布式锁
     *
     * @param key      锁的唯一标识
     * @param supplier 提供具体操作逻辑的函数式接口
     * @param <T>      返回值的类型
     * @return supplier 提供的逻辑的执行结果
     */
    public <T> T lock(String key, Supplier<T> supplier) throws LockInterruptedException, LockTimeoutException {
        return lock(key, DEFAULT_WAIT_TIME, DEFAULT_LEASE_TIME, supplier);
    }

    /**
     * 执行分布式锁操作
     *
     * @param key        锁的唯一标识
     * @param waitTime   用于非阻塞模式中等待获取锁的最大时间（单位：秒，0 代表立即尝试获取锁，不会等待，失败了会直接抛出异常）
     * @param leaseTime  锁自动释放时间（单位：秒）
     * @param supplier   在获取锁成功后执行的具体业务逻辑
     * @param isBlocking 是否使用阻塞模式
     * @param <T>        返回值的类型
     * @return supplier 具体业务逻辑的执行结果
     * @throws LockTimeoutException     非阻塞模式下在 waitTime 内未获取到锁时抛出
     * @throws LockInterruptedException 线程被中断时抛出
     * @apiNote
     * <p>
     * 参数设置建议：
     * </p>
     *
     * <pre>
     * | 目的      | waitTime 设置建议   | leaseTime 设置建议  |
     * |----------|------------|------------------|
     * | 快速失败   | 1~3 秒     | 业务执行时间 + buffer（如 5~10 秒）|
     * | 可以等一会 | 5~10 秒    | 业务执行时间 + buffer |
     * | 一定要执行 | 使用阻塞模式 | 同样设置足够的 leaseTime  |
     * </pre>
     *
     * <p>
     * 注意事项：
     * <ul>
     *   <li>leaseTime 应大于实际业务执行时间，否则可能在业务执行中锁已被释放。</li>
     *   <li>阻塞模式不会考虑 waitTime，线程会一直等待直到获取锁或被中断。</li>
     * </ul>
     * </p>
     */
    private <T> T executeWithLock(String key, Integer waitTime, Integer leaseTime, Supplier<T> supplier, boolean isBlocking) throws LockTimeoutException, LockInterruptedException {
        RLock lock = redissonClient.getLock(STR."LOCK-\{key}");
        boolean isLock = false;

        try {
            if (isBlocking) {
                lock.lock(leaseTime, TimeUnit.SECONDS); // 阻塞模式
                isLock = true;
            } else {
                isLock = lock.tryLock(waitTime, leaseTime, TimeUnit.SECONDS); // 非阻塞模式
            }

            if (isLock) {
                return supplier.get();
            } else {
                LOGGER.warn("获取锁：{}超时，未能获取到锁", lock.getName());
                throw new LockTimeoutException(STR."获取锁超时，未能获取到锁：\{key}");
            }
        } catch (InterruptedException e) {
            LOGGER.error("锁操作被中断：{}，原因：{}", lock.getName(), e.getMessage());
            // 恢复中断状态
            Thread.currentThread().interrupt();
            throw new LockInterruptedException("线程被中断", e);
        } finally {
            if (isLock && lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }
}
