package com.af.v4.system.common.plugins.security;

import com.af.v4.system.common.core.proxy.liuli.ILiuLiConfigServiceProxy;
import com.af.v4.system.common.core.utils.SpringUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * 基于固定密码表的单表代换加密工具类
 * 用于对字符串进行可逆加密与解密，主要用于对数据库敏感字段进行可检索（支持 LIKE）的加密存储
 * <p>
 * 特性：
 * - 高性能、线程安全
 * - 固定映射表（JSON 格式）
 * - 支持汉字、数字、英文
 * - 未匹配字符保持原样
 * - 密文固定为 4 位字符
 * </p>
 */
public class SubstitutionCipherUtil {

    private static final Logger log = LoggerFactory.getLogger(SubstitutionCipherUtil.class);

    /**
     * 密文固定长度
     */
    private static final int CIPHER_LENGTH = 4;

    /**
     * 当前启用的版本号
     */
    private static volatile String currentVersion;

    /**
     * 多版本密码表缓存：version -> {encryptMap, decryptMap}
     */
    private static final Map<String, VersionCipherTable> versionTables = new ConcurrentHashMap<>();

    /**
     * 版本密码表记录类
     */
    private record VersionCipherTable(
            Map<String, String> encryptMap,
            Map<String, String> decryptMap
    ) {}

    /**
     * 读写锁，用于初始化时的线程安全
     */
    private static final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();

    /**
     * 初始化标识
     */
    private static volatile boolean initialized = false;

    // 静态初始化块，通过琉璃中心加载密码表
    static {
        try {
            init();
        } catch (Exception e) {
            log.error("初始化密码表失败", e);
            // 不抛出异常，允许后续手动初始化或运行时刷新
        }
    }

    /**
     * 私有构造函数，防止实例化
     */
    private SubstitutionCipherUtil() {
        throw new UnsupportedOperationException("Utility class");
    }

    /**
     * 从琉璃中心加载密码表
     * 配置名：cipherTableConfig
     * 结构：{"enableVersion":"v1","v1":{ "明文":"密文", ... },"v2":{...}}
     */
    public static void init() {
        lock.writeLock().lock();
        try {
            log.info("开始初始化密码表");

            ILiuLiConfigServiceProxy liuLi = SpringUtils.getBean(ILiuLiConfigServiceProxy.class);

            JSONObject cfg = liuLi.get("cipherTableConfig");
            if (cfg == null) {
                throw new RuntimeException("LiuLi 未返回 cipherTableConfig");
            }

            String enableVersion = cfg.optString("enableVersion", null);
            if (enableVersion == null) {
                throw new RuntimeException("cipherTableConfig 缺少 enableVersion 字段");
            }

            // 清空旧数据
            versionTables.clear();

            // 加载所有版本的密码表
            for (String key : cfg.keySet()) {
                // 跳过配置元数据字段和启用版本配置项
                if ("enableVersion".equals(key) || key.startsWith("$")) {
                    continue;
                }
                
                JSONObject versionObj = cfg.optJSONObject(key);
                if (versionObj != null && !versionObj.isEmpty()) {
                    VersionCipherTable table = loadVersionTable(key, versionObj);
                    versionTables.put(key, table);
                    log.info("加载密码表版本 {}，映射数量: {}", key, table.encryptMap().size());
                }
            }

            // 验证启用版本是否存在
            if (!versionTables.containsKey(enableVersion)) {
                throw new RuntimeException("启用版本 " + enableVersion + " 在配置中不存在");
            }

            currentVersion = enableVersion;
            initialized = true;
            log.info("密码表初始化成功，当前启用版本: {}，总版本数: {}", currentVersion, versionTables.size());

        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            throw new RuntimeException("密码表初始化失败", e);
        } finally {
            lock.writeLock().unlock();
        }
    }

    /**
     * 加载单个版本的密码表
     *
     * @param version    版本号
     * @param versionObj 版本配置对象
     * @return 版本密码表
     */
    private static VersionCipherTable loadVersionTable(String version, JSONObject versionObj) {
        Map<String, String> encryptMap = new HashMap<>();
        for (String key : versionObj.keySet()) {
            encryptMap.put(key, versionObj.getString(key));
        }

        if (encryptMap.isEmpty()) {
            throw new RuntimeException("密码表版本 " + version + " 映射为空");
        }

        // 验证密码表格式
        validateCipherTable(encryptMap);

        // 构建解密表
        Map<String, String> decryptMap = new HashMap<>(encryptMap.size());
        for (Map.Entry<String, String> entry : encryptMap.entrySet()) {
            String plaintext = entry.getKey();
            String ciphertext = entry.getValue();
            if (decryptMap.containsKey(ciphertext)) {
                throw new RuntimeException(
                        String.format("密码表版本 %s 中存在重复的密文: %s 对应多个明文: %s 和 %s",
                                version, ciphertext, decryptMap.get(ciphertext), plaintext));
            }
            decryptMap.put(ciphertext, plaintext);
        }

        return new VersionCipherTable(
                Collections.unmodifiableMap(encryptMap),
                Collections.unmodifiableMap(decryptMap)
        );
    }

    /**
     * 外部刷新：强制重新加载密码表
     */
    public static void refresh() {
        init();
    }

    /**
     * 验证密码表格式
     *
     * @param cipherTable 密码表
     */
    private static void validateCipherTable(Map<String, String> cipherTable) {
        for (Map.Entry<String, String> entry : cipherTable.entrySet()) {
            String plaintext = entry.getKey();
            String ciphertext = entry.getValue();

            // 验证明文不为空
            if (plaintext == null || plaintext.isEmpty()) {
                throw new RuntimeException("密码表中存在空的明文键");
            }

            // 验证密文格式：必须是 4 位
            if (ciphertext == null || ciphertext.length() != CIPHER_LENGTH) {
                throw new RuntimeException(
                        String.format("密文格式错误，明文 '%s' 的密文 '%s' 长度不为 %d",
                                plaintext, ciphertext, CIPHER_LENGTH));
            }

            // 验证密文格式：第一位必须是小写字母
            char firstChar = ciphertext.charAt(0);
            if (firstChar < 'a' || firstChar > 'z') {
                throw new RuntimeException(
                        String.format("密文格式错误，明文 '%s' 的密文 '%s' 第一位不是小写字母",
                                plaintext, ciphertext));
            }

            // 验证密文格式：后三位必须是字母或数字
            for (int i = 1; i < CIPHER_LENGTH; i++) {
                char c = ciphertext.charAt(i);
                if (!Character.isLetterOrDigit(c)) {
                    throw new RuntimeException(
                            String.format("密文格式错误，明文 '%s' 的密文 '%s' 第 %d 位不是字母或数字",
                                    plaintext, ciphertext, i + 1));
                }
            }
        }
    }

    /**
     * 检查是否已初始化
     *
     * @throws IllegalStateException 未初始化时抛出
     */
    private static void checkInitialized() {
        if (!initialized) {
            throw new IllegalStateException("密码表未初始化，请先调用 init() 方法");
        }
    }

    /**
     * 加密字符串（带版本前缀）
     * <p>
     * 遍历输入字符串的每个字符：
     * - 若字符在密码表中存在映射，替换为对应的密文（4位）
     * - 否则保持原字符不变
     * - 最终返回格式：version:ciphertext（例如：v1:a1b2c3d4）
     * </p>
     *
     * @param plaintext 明文字符串
     * @return 带版本前缀的密文字符串
     * @throws IllegalArgumentException 输入为 null 时抛出
     * @throws IllegalStateException    密码表未初始化时抛出
     */
    public static String encrypt(String plaintext) {
        if (plaintext == null) {
            log.warn("明文不能为 null");
            return null;
        }

        checkInitialized();

        if (plaintext.isEmpty()) {
            return plaintext;
        }

        String cipherBody = encryptWithoutVersion(plaintext);
        return currentVersion + ":" + cipherBody;
    }

    /**
     * 加密字符串（不带版本前缀，内部使用）
     *
     * @param plaintext 明文字符串
     * @return 密文字符串（不含版本前缀）
     */
    private static String encryptWithoutVersion(String plaintext) {
        VersionCipherTable table = versionTables.get(currentVersion);
        if (table == null) {
            throw new IllegalStateException("当前版本 " + currentVersion + " 的密码表不存在");
        }

        StringBuilder result = new StringBuilder(plaintext.length() * CIPHER_LENGTH);

        // 遍历每个字符
        for (int i = 0; i < plaintext.length(); i++) {
            String character = String.valueOf(plaintext.charAt(i));
            String cipher = table.encryptMap().get(character);
            result.append(Objects.requireNonNullElse(cipher, character));
        }

        return result.toString();
    }

    /**
     * 解密字符串（支持版本识别）
     * <p>
     * 解析版本前缀，使用对应版本的密码表解密：
     * - 格式：version:ciphertext（例如：v1:a1b2c3d4）
     * - 遍历密文字符串，每 4 位作为一个密文单元
     * - 若该单元在解密表中存在，替换为对应的明文字符
     * - 否则按单个字符逐个保留（处理未加密的字符）
     * </p>
     *
     * @param ciphertext 带版本前缀的密文字符串
     * @return 明文字符串
     * @throws IllegalArgumentException 密文格式错误或输入为 null 时抛出
     * @throws IllegalStateException    密码表未初始化或版本不支持时抛出
     */
    public static String decrypt(String ciphertext) {
        if (ciphertext == null) {
            throw new IllegalArgumentException("密文不能为 null");
        }

        checkInitialized();

        if (ciphertext.isEmpty()) {
            return ciphertext;
        }

        // 解析版本前缀
        int colonIndex = ciphertext.indexOf(':');
        if (colonIndex <= 0 || colonIndex > 10) {
            throw new IllegalArgumentException("密文格式错误：缺少版本前缀，正确格式为 version:cipher");
        }

        String version = ciphertext.substring(0, colonIndex);
        String cipherBody = ciphertext.substring(colonIndex + 1);

        // 获取对应版本的解密表
        VersionCipherTable table = versionTables.get(version);
        if (table == null) {
            throw new IllegalStateException("不支持的密文版本: " + version + "，当前支持版本: " + versionTables.keySet());
        }

        // 使用对应版本解密
        return decryptWithTable(cipherBody, table.decryptMap());
    }

    /**
     * 使用指定解密表解密（内部方法）
     *
     * @param ciphertext 密文字符串（不含版本前缀）
     * @param decryptMap 解密映射表
     * @return 明文字符串
     */
    private static String decryptWithTable(String ciphertext, Map<String, String> decryptMap) {
        StringBuilder result = new StringBuilder(ciphertext.length());
        int i = 0;

        while (i < ciphertext.length()) {
            // 尝试读取 4 位作为一个密文单元
            if (i + CIPHER_LENGTH <= ciphertext.length()) {
                String cipherUnit = ciphertext.substring(i, i + CIPHER_LENGTH);
                String plainChar = decryptMap.get(cipherUnit);

                if (plainChar != null) {
                    // 找到映射，解密成功
                    result.append(plainChar);
                    i += CIPHER_LENGTH;
                    continue;
                }
            }

            // 未找到映射，保留当前字符
            result.append(ciphertext.charAt(i));
            i++;
        }

        return result.toString();
    }

    /**
     * 比较辅助：判断明文对应的密文是否与已有密文相等
     */
    public static boolean equalsCipher(String newPlain, String existingCipher) {
        if (newPlain == null && (existingCipher == null || existingCipher.isEmpty())) return true;
        if (newPlain == null) return false;
        String newCipher = encrypt(newPlain);
        return Objects.equals(newCipher, existingCipher);
    }

    /**
     * 比较辅助：判断明文对应的密文是否发生变化
     */
    public static boolean hasCipherChanged(String newPlain, String existingCipher) {
        return !equalsCipher(newPlain, existingCipher);
    }

    /**
     * 严格判定是否为替代密码密文（支持版本前缀）：
     * 1) 非空且长度 > 6（至少v1:xxxx）
     * 2) 包含版本前缀（格式：version:cipher）
     * 3) 版本在支持的版本列表中
     * 4) 密文主体长度必须是4的倍数
     * 5) 每个4字符块都能在对应版本的解密映射中命中
     */
    public static boolean isStrictCipher(String value) {
        try {
            if (value == null || value.length() <= 6) return false; // v1:xxxx 至少6字符
            checkInitialized();
            
            // 检查版本前缀格式
            int colonIndex = value.indexOf(':');
            if (colonIndex <= 0 || colonIndex > 10) return false;
            
            String version = value.substring(0, colonIndex);
            if (!versionTables.containsKey(version)) return false;
            
            String cipherBody = value.substring(colonIndex + 1);
            if (cipherBody.isEmpty() || cipherBody.length() % CIPHER_LENGTH != 0) return false;
            
            // 使用对应版本的解密表验证
            Map<String, String> decryptMap = versionTables.get(version).decryptMap();
            for (int i = 0; i < cipherBody.length(); i += CIPHER_LENGTH) {
                String block = cipherBody.substring(i, i + CIPHER_LENGTH);
                if (!decryptMap.containsKey(block)) return false;
            }
            
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 获取当前启用的版本号
     *
     * @return 当前版本号
     * @throws IllegalStateException 密码表未初始化时抛出
     */
    public static String getCurrentVersion() {
        checkInitialized();
        return currentVersion;
    }

    /**
     * 获取支持的所有版本
     *
     * @return 版本号集合（不可变）
     * @throws IllegalStateException 密码表未初始化时抛出
     */
    public static Set<String> getSupportedVersions() {
        checkInitialized();
        return Collections.unmodifiableSet(versionTables.keySet());
    }

    /**
     * 解析密文的版本号
     *
     * @param ciphertext 密文字符串
     * @return 版本号，如果无法解析则返回 null
     */
    public static String parseVersion(String ciphertext) {
        if (ciphertext == null || ciphertext.isEmpty()) {
            return null;
        }
        int colonIndex = ciphertext.indexOf(':');
        return (colonIndex > 0 && colonIndex < 10) ? ciphertext.substring(0, colonIndex) : null;
    }

    /**
     * 获取指定版本的加密映射表（只读）
     * 主要用于调试和验证
     *
     * @param version 版本号
     * @return 加密映射表的只读副本
     * @throws IllegalStateException 密码表未初始化或版本不存在时抛出
     */
    public static Map<String, String> getEncryptMap(String version) {
        checkInitialized();
        VersionCipherTable table = versionTables.get(version);
        if (table == null) {
            throw new IllegalStateException("版本 " + version + " 不存在");
        }
        return table.encryptMap();
    }

    /**
     * 获取指定版本的解密映射表（只读）
     * 主要用于调试和验证
     *
     * @param version 版本号
     * @return 解密映射表的只读副本
     * @throws IllegalStateException 密码表未初始化或版本不存在时抛出
     */
    public static Map<String, String> getDecryptMap(String version) {
        checkInitialized();
        VersionCipherTable table = versionTables.get(version);
        if (table == null) {
            throw new IllegalStateException("版本 " + version + " 不存在");
        }
        return table.decryptMap();
    }

    /**
     * 获取当前版本的加密映射表（只读）
     *
     * @return 当前版本的加密映射表
     * @throws IllegalStateException 密码表未初始化时抛出
     */
    public static Map<String, String> getEncryptMap() {
        return getEncryptMap(currentVersion);
    }

    /**
     * 获取当前版本的解密映射表（只读）
     *
     * @return 当前版本的解密映射表
     * @throws IllegalStateException 密码表未初始化时抛出
     */
    public static Map<String, String> getDecryptMap() {
        return getDecryptMap(currentVersion);
    }

    /**
     * 检查是否已初始化
     *
     * @return true 如果已初始化，否则 false
     */
    public static boolean isInitialized() {
        return initialized;
    }

    /**
     * 获取密文固定长度
     *
     * @return 密文长度（固定为 4）
     */
    public static int getCipherLength() {
        return CIPHER_LENGTH;
    }
}


