package com.af.v4.system.common.plugins.charts;

import org.json.JSONArray;
import org.json.JSONObject;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;

public class ChartsTools {
    // 常量定义
    private static final String DEFAULT_SERIES = "计数";
    private static final String SERIES_DELIMITER = "-";
    private static final String KEY_DELIMITER = "_";
    private static final int DEFAULT_PRECISION = 2;
    private static final Set<String> VALID_MOLDS = new HashSet<>(Arrays.asList("sum", "max", "min", "avg", "count"));

    /**
     * 验证图表配置
     */
    private void validateConfig(JSONObject chartConfig) {
        if (!chartConfig.has("xField")) {
            throw new IllegalArgumentException("必须配置 xField");
        }
        if ("sum".equals(chartConfig.getString("yFieldType")) && !chartConfig.has("yField")) {
            throw new IllegalArgumentException("必须配置 yField");
        }
        // 验证yFieldType - 从外层配置获取
        String yFieldType = chartConfig.optString("yFieldType", "sum").toLowerCase();
        if (!VALID_MOLDS.contains(yFieldType)) {
            throw new IllegalArgumentException("yFieldType 必须是 sum, max, min, avg 或 count");
        }
    }

    /**
     * 通用图表数据处理方法
     */
    public JSONArray processChartData(JSONArray data, JSONObject chartConfig, ChartType chartType) {
        // 验证配置
        validateConfig(chartConfig);

        ChartConfig config = ChartConfig.fromJSON(chartConfig);

        // 处理柱状图和折线图数据
        return processAxisChartData(data, config);
    }

    /**
     * 格式化数字，控制精度
     */
    private double formatNumber(double value, int precision) {
        // 使用BigDecimal处理精度问题
        BigDecimal bd = new BigDecimal(value);
        bd = bd.setScale(precision, RoundingMode.HALF_UP);
        return bd.doubleValue();
    }

    private JSONArray processAxisChartData(JSONArray data, ChartConfig config) {
        DataAggregator aggregator = new DataAggregator();
        boolean isCount = "count".equals(config.getYFieldType());

        // 第一次遍历：收集唯一值
        data.forEach(obj -> {
            JSONObject item = (JSONObject) obj;
            Optional.ofNullable(item.optString(config.getXField()))
                    .filter(x -> !x.trim().isEmpty())
                    .ifPresent(xValue -> {
                        String groupValue = config.getGroupField() != null ?
                                item.optString(config.getGroupField()) : null;
                        aggregator.collectValue(xValue.trim(), groupValue);
                    });
        });

        // 第二次遍历：处理数据
        if (isCount) {
            processCountType(data, config, aggregator);
        } else {
            processSumType(data, config, aggregator);
        }

        // 补充缺失值
        fillMissingValues(config, aggregator);

        return aggregator.buildResult(config.getXField());
    }

    // 其他辅助方法...
    private void processCountType(JSONArray data, ChartConfig config, DataAggregator aggregator) {
        // 处理计数逻辑
        for (int j = 0; j < data.length(); j++) {
            JSONObject item = data.getJSONObject(j);
            if (item.isNull(config.getXField())) continue;
            String xValue = item.optString(config.getXField(), "").trim();
            if (xValue.isEmpty()) continue;

            if (config.getGroupField() != null) {
                String groupValue = item.isNull(config.getGroupField()) ? "" : item.optString(config.getGroupField());
                // 只在 groupValue 不为空时添加分隔符
                String seriesName = groupValue.isEmpty() ? DEFAULT_SERIES : groupValue + SERIES_DELIMITER + DEFAULT_SERIES;
                String key = xValue + KEY_DELIMITER + seriesName;
                aggregator.addValue(key, 1.0, seriesName, config.getPrecision());
            } else {
                String key = xValue + KEY_DELIMITER + DEFAULT_SERIES;
                aggregator.addValue(key, 1.0, DEFAULT_SERIES, config.getPrecision());
            }
        }
    }

    private void processSumType(JSONArray data, ChartConfig config, DataAggregator aggregator) {
        // 处理求和逻辑
        for (int j = 0; j < data.length(); j++) {
            JSONObject item = data.getJSONObject(j);
            if (item.isNull(config.getXField())) continue;
            String xValue = item.optString(config.getXField(), "").trim();
            if (xValue.isEmpty()) continue;

            for (int k = 0; k < config.getYFields().length(); k++) {
                JSONObject yField = config.getYFields().getJSONObject(k);
                String field = yField.getString("field");
                String label = yField.optString("label", field);
                String mold = yField.optString("mold", "sum").toLowerCase();

                if (!VALID_MOLDS.contains(mold)) {
                    continue; // 跳过无效的统计模式
                }

                if (config.getGroupField() != null) {
                    String groupValue = item.isNull(config.getGroupField()) ? "" : item.optString(config.getGroupField());
                    // 只在 groupValue 不为空时添加分隔符
                    String seriesName = groupValue.isEmpty() ? label : groupValue + SERIES_DELIMITER + label;
                    String key = xValue + KEY_DELIMITER + seriesName;

                    // 移除了多余的 config.getYFieldType() 参数
                    processValue(item, field, mold, config.getPrecision(), aggregator, key, seriesName);
                } else {
                    String key = xValue + KEY_DELIMITER + label;
                    // 移除了多余的 config.getYFieldType() 参数
                    processValue(item, field, mold, config.getPrecision(), aggregator, key, label);
                }
            }
        }
    }

    private void fillMissingValues(ChartConfig config, DataAggregator aggregator) {
        if (config.getGroupField() != null && "count".equals(config.getYFieldType())) {
            for (String xValue : aggregator.xValues) {
                for (String groupValue : aggregator.groupValues) {
                    // 只在组值不为空时添加分隔符
                    String seriesName = groupValue.isEmpty() ? DEFAULT_SERIES : groupValue + SERIES_DELIMITER + DEFAULT_SERIES;
                    String key = xValue + KEY_DELIMITER + seriesName;
                    if (!aggregator.valueMap.containsKey(key)) {
                        aggregator.addValue(key, 0.0, seriesName, config.getPrecision());
                    }
                }
            }
        } else if (config.getGroupField() != null) {
            for (String xValue : aggregator.xValues) {
                for (String groupValue : aggregator.groupValues) {
                    for (int k = 0; k < config.getYFields().length(); k++) {
                        JSONObject yField = config.getYFields().getJSONObject(k);
                        String label = yField.optString("label", yField.getString("field"));
                        // 只在组值不为空时添加分隔符
                        String seriesName = groupValue.isEmpty() ? label : groupValue + SERIES_DELIMITER + label;
                        String key = xValue + KEY_DELIMITER + seriesName;

                        if (!aggregator.valueMap.containsKey(key)) {
                            aggregator.addValue(key, 0.0, seriesName, config.getPrecision());
                        }
                    }
                }
            }
        }
    }

    // 抽取处理单个值的方法
    private void processValue(JSONObject item, String field, String mold, int precision,
                              DataAggregator aggregator, String key, String seriesName) {
        try {
            if (!item.isNull(field)) {
                double value = item.getDouble(field);
                if (!Double.isNaN(value) && !Double.isInfinite(value)) {
                    value = formatNumber(value, precision);
                    switch (mold.toLowerCase()) {
                        case "sum":
                            aggregator.addValue(key, value, seriesName, precision);
                            break;
                        case "max":
                            aggregator.updateMax(key, value, seriesName);
                            break;
                        case "min":
                            aggregator.updateMin(key, value, seriesName);
                            break;
                        case "avg":
                            aggregator.addToAverage(key, value, seriesName);
                            break;
                        default:
                            // count处理保持不变
                            aggregator.addValue(key, 1.0, seriesName, precision);
                    }
                }
            }
        } catch (Exception e) {
            // 忽略解析错误的数据
        }
    }

    public JSONArray mergeGroupData(JSONArray totalData, JSONArray newData, JSONObject chartConfig) {
        // 先处理新数据
        JSONArray processedNewData = processChartData(newData, chartConfig, ChartType.LINE);

        // 如果totalData为空，直接返回处理后的新数据
        if (totalData == null || totalData.isEmpty()) {
            return processedNewData;
        }

        String xField = chartConfig.getString("xField");
        JSONArray yFields = chartConfig.getJSONArray("yField");
        String groupField = chartConfig.optString("groupField", null);

        // 使用Map进行数据合并
        Map<String, JSONObject> mergedMap = new LinkedHashMap<>();

        // 处理已有数据
        for (int i = 0; i < totalData.length(); i++) {
            JSONObject item = totalData.getJSONObject(i);
            if (item.isNull(xField)) {
                continue; // 跳过xField为null的数据
            }
            String xValue = item.optString(xField, "");
            if (xValue.isEmpty()) {
                continue;
            }

            String key = groupField != null ?
                    STR."\{item.isNull(groupField) ? "" : item.optString(groupField)}_\{xValue}" :
                    xValue;
            mergedMap.put(key, new JSONObject(item.toString()));
        }

        // 合并新数据
        for (int i = 0; i < processedNewData.length(); i++) {
            JSONObject item = processedNewData.getJSONObject(i);
            if (item.isNull(xField)) {
                continue;
            }
            String xValue = item.optString(xField, "");
            if (xValue.isEmpty()) {
                continue;
            }

            String key = groupField != null ?
                    STR."\{item.isNull(groupField) ? "" : item.optString(groupField)}_\{xValue}" :
                    xValue;

            if (mergedMap.containsKey(key)) {
                // 合并相同key的数据
                JSONObject existing = mergedMap.get(key);
                // 合并所有yField的值
                for (int j = 0; j < yFields.length(); j++) {
                    JSONObject yField = yFields.getJSONObject(j);
                    String field = yField.getString("field");
                    double sum = existing.getDouble(field) + item.getDouble(field);
                    // 合并时处理精度
                    existing.put(field, formatNumber(sum, chartConfig.optInt("precision", 2))); // 直接放入数值
                }
            } else {
                mergedMap.put(key, new JSONObject(item.toString()));
            }
        }

        // 转换回JSONArray
        JSONArray result = new JSONArray();
        for (JSONObject value : mergedMap.values()) {
            result.put(value);
        }

        return result;
    }

    // 图表类型枚举
    public enum ChartType {
        LINE,   // 折线图
        BAR,    // 柱状图
        PIE     // 饼图
    }

    // 图表配置类
    public static class ChartConfig {
        private final String xField;
        private final JSONArray yFields;
        private final String groupField;
        private final String yFieldType;
        private final int precision;

        private ChartConfig(Builder builder) {
            this.xField = builder.xField;
            this.yFields = builder.yFields;
            this.groupField = builder.groupField;
            this.yFieldType = builder.yFieldType;
            this.precision = builder.precision;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static ChartConfig fromJSON(JSONObject config) {
            return ChartConfig.builder()
                    .xField(config.getString("xField"))
                    .yFields(config.optJSONArray("yField"))
                    .groupField(config.optString("groupField"))
                    .yFieldType(config.optString("yFieldType", "sum").toLowerCase())
                    .precision(config.optInt("precision", DEFAULT_PRECISION))
                    .build();
        }

        public String getXField() {
            return xField;
        }

        public JSONArray getYFields() {
            return yFields;
        }

        public String getGroupField() {
            return groupField;
        }

        public String getYFieldType() {
            return yFieldType;
        }

        public int getPrecision() {
            return precision;
        }

        public static class Builder {
            private String xField;
            private JSONArray yFields;
            private String groupField;
            private String yFieldType;
            private int precision;

            public Builder xField(String xField) {
                this.xField = xField;
                return this;
            }

            public Builder yFields(JSONArray yFields) {
                this.yFields = yFields;
                return this;
            }

            public Builder groupField(String groupField) {
                this.groupField = groupField;
                return this;
            }

            public Builder yFieldType(String yFieldType) {
                this.yFieldType = yFieldType;
                return this;
            }

            public Builder precision(int precision) {
                this.precision = precision;
                return this;
            }

            public ChartConfig build() {
                return new ChartConfig(this);
            }
        }
    }

    // 数据聚合器
    private class DataAggregator {
        private final Set<String> xValues = new LinkedHashSet<>();
        private final Set<String> groupValues = new LinkedHashSet<>();
        private final Map<String, Double> valueMap = new LinkedHashMap<>();
        private final Map<String, String> seriesMap = new LinkedHashMap<>();
        private final Map<String, AggregateData> aggregateData = new LinkedHashMap<>();

        public void collectValue(String xValue, String groupValue) {
            xValues.add(xValue);
            if (groupValue != null) {
                groupValues.add(groupValue);
            }
        }

        public void addValue(String key, double value, String seriesName, int precision) {
            valueMap.merge(key, value, (a, b) ->
                    formatNumber(a + b, precision));
            seriesMap.put(key, seriesName);
        }

        public void updateMax(String key, double value, String seriesName) {
            aggregateData.compute(key, (k, v) -> {
                if (v == null) {
                    return new AggregateData(value, seriesName);
                }
                v.value = Math.max(v.value, value);
                return v;
            });
        }

        public void updateMin(String key, double value, String seriesName) {
            aggregateData.compute(key, (k, v) -> {
                if (v == null) {
                    return new AggregateData(value, seriesName);
                }
                v.value = Math.min(v.value, value);
                return v;
            });
        }

        public void addToAverage(String key, double value, String seriesName) {
            aggregateData.compute(key, (k, v) -> {
                if (v == null) {
                    return new AggregateData(value, seriesName);
                }
                v.value += value;
                v.count++;
                return v;
            });
        }

        public JSONArray buildResult(String xField) {
            JSONArray result = new JSONArray();
            // 处理聚合数据
            aggregateData.forEach((key, data) -> {
                String[] parts = key.split(KEY_DELIMITER);
                JSONObject point = new JSONObject()
                        .put(xField, parts[0])
                        .put("value", data.value / (data.count > 0 ? data.count : 1))
                        .put("series", data.seriesName);
                result.put(point);
            });
            // 处理其他数据
            valueMap.forEach((key, value) -> {
                if (!aggregateData.containsKey(key)) {
                    String[] parts = key.split(KEY_DELIMITER);
                    JSONObject point = new JSONObject()
                            .put(xField, parts[0])
                            .put("value", value)
                            .put("series", seriesMap.get(key));
                    result.put(point);
                }
            });
            return result;
        }

        private static class AggregateData {
            final String seriesName;
            double value;
            long count;

            AggregateData(double value, String seriesName) {
                this.value = value;
                this.count = 1;
                this.seriesName = seriesName;
            }
        }
    }
}
