package com.af.v4.system.common.plugins.other;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;

/**
 * SHA1加密工具类
 * 提供SHA1哈希、验证、HMAC等相关功能
 * 
 * @author Claude
 */
public class SHA1Tools {
    
    /**
     * 对字符串进行SHA1哈希计算
     * 
     * @param data 待哈希的字符串
     * @return 40位十六进制哈希值，如果输入为空则返回null
     */
    public static String sha1(String data) {
        if (data == null || data.isEmpty()) {
            return null;
        }
        try {
            MessageDigest sha1 = MessageDigest.getInstance("SHA-1");
            byte[] hash = sha1.digest(data.getBytes(StandardCharsets.UTF_8));
            return bytesToHex(hash);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-1算法不可用", e);
        }
    }
    
    /**
     * 对字符串进行SHA1哈希计算（别名方法）
     * 
     * @param data 待哈希的字符串
     * @return 40位十六进制哈希值，如果输入为空则返回null
     */
    public static String sha1Hex(String data) {
        return sha1(data);
    }
    
    /**
     * 对字符串进行SHA1哈希计算并返回Base64编码
     * 
     * @param data 待哈希的字符串
     * @return Base64编码的哈希值，如果输入为空则返回null
     */
    public static String sha1Base64(String data) {
        if (data == null || data.isEmpty()) {
            return null;
        }
        try {
            MessageDigest sha1 = MessageDigest.getInstance("SHA-1");
            byte[] hash = sha1.digest(data.getBytes(StandardCharsets.UTF_8));
            return Base64.getEncoder().encodeToString(hash);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-1算法不可用", e);
        }
    }
    
    /**
     * 对字符串进行SHA1哈希计算并返回字节数组
     * 
     * @param data 待哈希的字符串
     * @return 20字节的哈希值数组，如果输入为空则返回null
     */
    public static byte[] sha1Bytes(String data) {
        if (data == null || data.isEmpty()) {
            return null;
        }
        try {
            MessageDigest sha1 = MessageDigest.getInstance("SHA-1");
            return sha1.digest(data.getBytes(StandardCharsets.UTF_8));
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-1算法不可用", e);
        }
    }
    
    /**
     * 对字节数组进行SHA1哈希计算
     * 
     * @param data 待哈希的字节数组
     * @return 40位十六进制哈希值，如果输入为空则返回null
     */
    public static String sha1FromBytes(byte[] data) {
        if (data == null || data.length == 0) {
            return null;
        }
        try {
            MessageDigest sha1 = MessageDigest.getInstance("SHA-1");
            byte[] hash = sha1.digest(data);
            return bytesToHex(hash);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-1算法不可用", e);
        }
    }
    
    /**
     * 验证数据的SHA1哈希值是否匹配
     * 
     * @param data 原始数据
     * @param hash 期望的十六进制哈希值
     * @return 验证是否通过
     */
    public static boolean verify(String data, String hash) {
        if (data == null || hash == null) {
            return false;
        }
        return hash.equals(sha1(data));
    }
    
    /**
     * 验证数据的SHA1哈希值是否匹配（Base64格式）
     * 
     * @param data 原始数据
     * @param base64Hash 期望的Base64编码哈希值
     * @return 验证是否通过
     */
    public static boolean verifyBase64(String data, String base64Hash) {
        if (data == null || base64Hash == null) {
            return false;
        }
        return base64Hash.equals(sha1Base64(data));
    }
    
    /**
     * 对字符串进行加盐SHA1哈希计算
     * 将盐值附加到原数据后再进行哈希，增强安全性
     * 
     * @param data 待哈希的字符串
     * @param salt 盐值
     * @return 40位十六进制哈希值，如果任一参数为空则返回null
     */
    public static String sha1WithSalt(String data, String salt) {
        if (data == null || salt == null) {
            return null;
        }
        return sha1(data + salt);
    }
    
    /**
     * 计算HMAC-SHA1值
     * 基于密钥的哈希消息认证码，用于验证数据完整性和真实性
     * 
     * @param data 待计算HMAC的数据
     * @param key 密钥
     * @return 40位十六进制HMAC值，如果任一参数为空则返回null
     */
    public static String sha1HMAC(String data, String key) {
        if (data == null || key == null) {
            return null;
        }
        try {
            MessageDigest sha1 = MessageDigest.getInstance("SHA-1");
            byte[] keyBytes = key.getBytes(StandardCharsets.UTF_8);
            byte[] dataBytes = data.getBytes(StandardCharsets.UTF_8);
            
            if (keyBytes.length > 64) {
                sha1.update(keyBytes);
                keyBytes = sha1.digest();
                sha1.reset();
            }
            
            byte[] ipad = new byte[64];
            byte[] opad = new byte[64];
            
            System.arraycopy(keyBytes, 0, ipad, 0, keyBytes.length);
            System.arraycopy(keyBytes, 0, opad, 0, keyBytes.length);
            
            for (int i = 0; i < 64; i++) {
                ipad[i] ^= 0x36;
                opad[i] ^= 0x5c;
            }
            
            sha1.update(ipad);
            sha1.update(dataBytes);
            byte[] innerHash = sha1.digest();
            sha1.reset();
            
            sha1.update(opad);
            sha1.update(innerHash);
            byte[] hmac = sha1.digest();
            
            return bytesToHex(hmac);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-1算法不可用", e);
        }
    }
    
    /**
     * 计算文件数据的SHA1哈希值
     * 用于文件完整性校验
     * 
     * @param fileData 文件的字节数据
     * @return 40位十六进制哈希值，如果输入为空则返回null
     */
    public static String hashFile(byte[] fileData) {
        return sha1FromBytes(fileData);
    }
    
    /**
     * 计算文件数据的SHA1哈希值并返回Base64编码
     * 用于文件完整性校验
     * 
     * @param fileData 文件的字节数据
     * @return Base64编码的哈希值，如果输入为空则返回null
     */
    public static String hashFileBase64(byte[] fileData) {
        if (fileData == null || fileData.length == 0) {
            return null;
        }
        try {
            MessageDigest sha1 = MessageDigest.getInstance("SHA-1");
            byte[] hash = sha1.digest(fileData);
            return Base64.getEncoder().encodeToString(hash);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-1算法不可用", e);
        }
    }
    
    /**
     * 将字节数组转换为十六进制字符串
     * 
     * @param bytes 字节数组
     * @return 十六进制字符串
     */
    private static String bytesToHex(byte[] bytes) {
        StringBuilder result = new StringBuilder();
        for (byte b : bytes) {
            result.append(String.format("%02x", b));
        }
        return result.toString();
    }
}