package com.af.v4.system.common.plugins.security;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * 敏感数据增强处理工具类 包含所有敏感数据处理功能：
 * 1. 哈希优化：通过 SHA256 哈希对比避免重复处理
 * 2. 智能分类：自动识别数据状态，跳过已处理数据
 * 3. 同表模式：支持在主表中直接存储哈希和加密字段
 * 4. 完整脱敏：提供各种脱敏策略（姓名、手机、身份证、地址等）
 * 5. AES加密：独立的加密解密功能
 * 6. 性能优化：大幅提升敏感数据处理性能
 */
public class SensitiveDataEnhancer {

    private static final Logger log = LoggerFactory.getLogger(SensitiveDataEnhancer.class);

    // AES 加密密钥
    public static final String AES_KEY = "7xjgtQc4M8FOXikU7JkwcUI0wKhYkREt";

    // 加密算法
    private static final String AES_ALGORITHM = "AES/ECB/PKCS5Padding";

    // 预编译正则表达式，提升性能 (Java 21 现代化)
    private static final Pattern DIGITS_ONLY_PATTERN = Pattern.compile("\\D");
    private static final Pattern PHONE_FORMAT_PATTERN = Pattern.compile("\\d+");
    private static final Pattern ID_NUMBER_PATTERN = Pattern.compile("\\d{6}\\d{8}[\\dXx]");

    // 策略匹配关键词集合 (使用 Set.of 不可变集合，Java 21 优化)
    private static final Set<String> NAME_STRATEGIES = Set.of("name", "姓名", "用户名");
    private static final Set<String> PHONE_STRATEGIES = Set.of("phone", "mobile", "手机", "电话");
    private static final Set<String> ADDRESS_STRATEGIES = Set.of("address", "addr", "地址", "住址");
    private static final Set<String> ID_STRATEGIES = Set.of("idnumber", "idcard", "身份证", "证件号");

    /**
     * 应用脱敏策略 (Java 21 现代化)
     * 使用 switch expressions 和自动策略推断
     */
    public static String applyMaskingStrategy(String fieldName, String originalValue, String strategy) {
        if (originalValue == null || originalValue.isEmpty()) {
            return originalValue;
        }

        // 策略推断：如果未指定策略，根据字段名自动推断
        String actualStrategy = strategy != null ? strategy : inferStrategy(fieldName);

        try {
            // 使用 Java 21 switch 表达式，更简洁和类型安全
            return switch (actualStrategy.toLowerCase()) {
                case "name" -> maskUserName(originalValue);
                case "phone", "mobile" -> maskPhone(originalValue);
                case "address", "addr" -> maskAddress(originalValue);
                case "idnumber", "idcard" -> maskIdNumber(originalValue);
                default -> maskDefault(originalValue);
            };
        } catch (Exception e) {
            log.error("脱敏处理失败: {}, 策略: {}", fieldName, actualStrategy, e);
            return "**MASKED**";
        }
    }

    /**
     * 根据字段名推断脱敏策略 (Java 21 性能优化)
     * 使用预编译的策略集合和Stream API，避免重复创建集合对象
     */
    private static String inferStrategy(String fieldName) {
        String lowerFieldName = fieldName.toLowerCase();

        // 使用预编译的策略集合，性能更优
        if (NAME_STRATEGIES.stream().anyMatch(lowerFieldName::contains) ||
                Set.of("username", "realname", "nickname").stream().anyMatch(lowerFieldName::contains)) {
            return "name";
        }

        if (PHONE_STRATEGIES.stream().anyMatch(lowerFieldName::contains) ||
                Set.of("tel", "telephone").stream().anyMatch(lowerFieldName::contains)) {
            return "phone";
        }

        if (ADDRESS_STRATEGIES.stream().anyMatch(lowerFieldName::contains) ||
                Set.of("location").stream().anyMatch(lowerFieldName::contains)) {
            return "address";
        }

        if (ID_STRATEGIES.stream().anyMatch(lowerFieldName::contains) ||
                (lowerFieldName.contains("id") &&
                        Set.of("number", "card").stream().anyMatch(lowerFieldName::contains)) ||
                Set.of("cardnumber").stream().anyMatch(lowerFieldName::contains)) {
            return "idnumber";
        }

        return "default";
    }

    /**
     * 姓名脱敏 (Java 21 现代化)
     * 使用 switch 表达式和现代 String API
     */
    private static String maskUserName(String name) {
        if (name == null || name.isEmpty()) {
            return name;
        }

        int length = name.length();
        return switch (length) {
            case 1 -> name; // 单字符不脱敏
            case 2 -> "*" + name.charAt(1); // 两字符：保留最后一个
            default -> name.charAt(0) + "*".repeat(length - 2) + name.charAt(length - 1); // 保留首尾
        };
    }

    /**
     * 手机号脱敏 (Java 21 现代化 + 性能优化)
     * 支持多种格式：手机号、座机号(带/不带区号)，使用预编译正则表达式和 switch 表达式
     */
    private static String maskPhone(String phone) {
        if (phone == null || phone.isEmpty()) {
            return phone;
        }

        // 使用预编译正则表达式提取数字，性能更优
        String digitsOnly = DIGITS_ONLY_PATTERN.matcher(phone).replaceAll("");
        int length = digitsOnly.length();

        if (length < 4) {
            return phone; // 太短，不脱敏
        }

        // 根据数字长度使用不同的脱敏策略，使用预编译正则表达式替换
        return switch (length) {
            case 4, 5, 6, 7 -> {
                // 4-7位：保留前2位和后1位
                String prefix = digitsOnly.substring(0, 2);
                String suffix = digitsOnly.substring(length - 1);
                yield PHONE_FORMAT_PATTERN.matcher(phone).replaceAll(prefix + "*".repeat(length - 3) + suffix);
            }
            case 8, 9, 10, 11 -> {
                // 8-11位：保留前3位和后2位
                String prefix = digitsOnly.substring(0, 3);
                String suffix = digitsOnly.substring(length - 2);
                yield PHONE_FORMAT_PATTERN.matcher(phone).replaceAll(prefix + "****" + suffix);
            }
            default -> {
                // 超过11位：保留前3位和后4位
                String prefix = digitsOnly.substring(0, 3);
                String suffix = digitsOnly.substring(length - 4);
                yield PHONE_FORMAT_PATTERN.matcher(phone).replaceAll(prefix + "****" + suffix);
            }
        };
    }

    /**
     * 身份证脱敏 (Java 21 现代化 + 格式验证)
     * 使用 String.repeat()、条件表达式和预编译正则表达式验证
     */
    private static String maskIdNumber(String idNumber) {
        if (idNumber == null || idNumber.isEmpty()) {
            return idNumber;
        }

        // 基本长度验证 + 可选的格式验证
        return switch (idNumber.length()) {
            case 15 -> // 15位老身份证
                    idNumber.substring(0, 6) + "*".repeat(6) + idNumber.substring(12);
            case 18 -> { // 18位新身份证
                // 可选：使用预编译正则表达式进行更严格的格式验证
                if (ID_NUMBER_PATTERN.matcher(idNumber).matches()) {
                    yield idNumber.substring(0, 6) + "*".repeat(8) + idNumber.substring(14);
                } else {
                    // 格式不匹配，采用通用脱敏策略
                    yield idNumber.substring(0, 6) + "*".repeat(8) + idNumber.substring(14);
                }
            }
            default -> idNumber.length() >= 10
                    ? idNumber.substring(0, 6) + "*".repeat(Math.min(8, idNumber.length() - 10)) +
                    idNumber.substring(Math.max(6, idNumber.length() - 4))
                    : idNumber; // 太短，不脱敏
        };
    }

    /**
     * 地址脱敏 (Java 21 现代化)
     * 使用 switch 表达式按长度分类处理
     */
    private static String maskAddress(String address) {
        if (address == null || address.isEmpty()) {
            return address;
        }

        int length = address.length();
        return switch (length) {
            case 1, 2, 3 -> address; // 太短，不脱敏
            case 4, 5, 6 -> address.substring(0, 2) + "***";
            case 7, 8, 9, 10 -> address.substring(0, 3) + "***" + address.substring(length - 2);
            default -> address.substring(0, 4) + "***" + address.substring(length - 3);
        };
    }

    /**
     * 默认脱敏策略 (Java 21 现代化)
     * 使用 String.repeat() 和 switch 表达式
     */
    private static String maskDefault(String value) {
        if (value == null || value.isEmpty()) {
            return value;
        }

        int length = value.length();
        return switch (length) {
            case 1, 2 -> "*".repeat(length); // 短字符串全部脱敏
            case 3, 4, 5, 6 -> value.charAt(0) + "*".repeat(length - 2) + value.charAt(length - 1);
            default -> value.substring(0, 2) + "*".repeat(length - 4) + value.substring(length - 2);
        };
    }

    // 删除了 repeatString 方法，因为 Java 11+ 已经有 String.repeat() 方法

    /**
     * 性能优化工具方法 (Java 21 现代化)
     * 检查字段名是否匹配预编译的策略关键词
     */
    private static boolean matchesStrategy(String fieldName, Set<String> strategies) {
        return strategies.stream().anyMatch(fieldName::contains);
    }

    /**
     * 快速字符串验证工具 (Java 21 性能优化)
     * 避免多次 null 检查和字符串操作
     */
    private static boolean isValidInput(String input) {
        return input == null || input.trim().isEmpty();
    }

    /**
     * 生成 SHA256 哈希
     */
    // 移除 SHA256 Hash 方案，统一改为 cipher（SubstitutionCipherUtil.encrypt）

    /**
     * AES 加密 (Java 21 现代化 + 性能优化)
     * 移除不必要的KeyGenerator，直接使用SecretKeySpec
     */
    public static String aesEncrypt(String input) throws Exception {
        if (isValidInput(input)) {
            return null;
        }

        // 直接创建密钥规范，无需KeyGenerator（性能优化）
        SecretKeySpec keySpec = new SecretKeySpec(AES_KEY.getBytes(StandardCharsets.UTF_8), "AES");
        Cipher cipher = Cipher.getInstance(AES_ALGORITHM);
        cipher.init(Cipher.ENCRYPT_MODE, keySpec);

        byte[] encryptedBytes = cipher.doFinal(input.getBytes(StandardCharsets.UTF_8));
        return java.util.Base64.getEncoder().encodeToString(encryptedBytes);
    }

    /**
     * AES 解密 (Java 21 现代化 + 性能优化)
     * 移除不必要的KeyGenerator，直接使用SecretKeySpec
     */
    public static String aesDecrypt(String encrypted) throws Exception {
        if (isValidInput(encrypted)) {
            return null;
        }

        try {
            // 直接创建密钥规范，无需KeyGenerator（性能优化）
            SecretKeySpec keySpec = new SecretKeySpec(AES_KEY.getBytes(StandardCharsets.UTF_8), "AES");
            Cipher cipher = Cipher.getInstance(AES_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, keySpec);

            byte[] encryptedBytes = java.util.Base64.getDecoder().decode(encrypted);
            byte[] decryptedBytes = cipher.doFinal(encryptedBytes);
            return new String(decryptedBytes, StandardCharsets.UTF_8);
        } catch (Exception e) {
            log.error("AES解密失败，可能是数据格式错误或密钥不匹配", e);
            throw e; // 重新抛出异常，让调用方处理
        }
    }

    /**
     * 敏感字段统一处理工具函数 - 一次性完成hash、加密、脱敏
     *
     * @param fieldName     字段名
     * @param originalValue 原始值
     * @param maskStrategy  脱敏策略
     * @return 处理结果，包含hash、加密、脱敏值
     */
    public static SensitiveProcessResult processSensitiveField(String fieldName, String originalValue, String maskStrategy) {
        // 使用现代化的输入验证工具方法
        if (isValidInput(originalValue)) {
            return null;
        }

        try {
            // 统一改为 cipher + AES + mask
            String cipherValue = SubstitutionCipherUtil.encrypt(originalValue);
            String encryptedValue = aesEncrypt(originalValue);
            String maskedValue = applyMaskingStrategy(fieldName, originalValue, maskStrategy);

            // 验证所有结果都成功
            if (cipherValue != null && encryptedValue != null && maskedValue != null) {
                return new SensitiveProcessResult(cipherValue, encryptedValue, maskedValue);
            } else {
                log.warn("敏感字段处理部分失败: {}, hash={}, encrypted={}, masked={}",
                        fieldName, cipherValue != null, encryptedValue != null, maskedValue != null);
                return null;
            }

        } catch (Exception e) {
            log.error("敏感字段处理失败: {}", fieldName, e);
            return null;
        }
    }

    /**
     * 敏感字段处理结果记录类
     * 提供不可变的数据载体，自动生成 equals(), hashCode(), toString() 方法
     *
     * @param cipherValue    密文值
     * @param encryptedValue AES 加密值，用于安全存储
     * @param maskedValue    脱敏显示值，用于前端展示
     */
    public record SensitiveProcessResult(
            String cipherValue,
            String encryptedValue,
            String maskedValue
    ) {
        /**
         * 紧凑构造器：提供输入验证
         */
        public SensitiveProcessResult {
            // 验证关键参数不为null（允许为空字符串）
            if (cipherValue == null || encryptedValue == null || maskedValue == null) {
                throw new IllegalArgumentException("所有字段都不能为null");
            }
        }

        /**
         * 验证处理结果是否有效
         */
        public boolean isValid() {
            return !cipherValue.isBlank() && !encryptedValue.isBlank() && !maskedValue.isBlank();
        }

        /**
         * 静态工厂方法：创建空结果
         */
        public static SensitiveProcessResult empty() {
            return new SensitiveProcessResult("", "", "");
        }
    }

    /**
     * 敏感数据处理结果记录类 (Java 21 Enhanced Record)
     * 用于beforeSave方法返回跨表模式的加密数据
     * 提供不可变的数据载体和丰富的验证方法
     *
     * @param tableName     加密表名
     * @param data          待保存的加密数据
     * @param relationField 业务表关联字段名
     */
    public record SensitiveDataResult(
            String tableName,
            JSONObject data,
            String relationField
    ) {
        /**
         * 紧凑构造器：提供输入验证和数据规范化
         */
        public SensitiveDataResult {
            // 参数验证
            if (tableName == null || tableName.isBlank()) {
                throw new IllegalArgumentException("表名不能为空");
            }
            if (relationField == null || relationField.isBlank()) {
                throw new IllegalArgumentException("关联字段名不能为空");
            }
            // data 允许为null或空，表示没有需要保存的数据

            // 数据规范化
            tableName = tableName.trim();
            relationField = relationField.trim();
        }

        /**
         * 检查是否有有效的跨表数据需要保存
         */
        public boolean hasData() {
            return data != null && !data.isEmpty();
        }

        /**
         * 检查是否为空结果（无数据需要保存）
         */
        public boolean isEmpty() {
            return !hasData();
        }

        /**
         * 获取数据大小（字段数量）
         */
        public int dataSize() {
            return hasData() ? data.length() : 0;
        }

        /**
         * 安全获取数据的字符串表示（避免敏感信息泄漏）
         */
        public String getDataSummary() {
            if (!hasData()) {
                return "无数据";
            }
            return "包含" + data.length() + "个字段的加密数据";
        }

        /**
         * 静态工厂方法：创建空结果
         */
        public static SensitiveDataResult empty(String tableName, String relationField) {
            return new SensitiveDataResult(tableName, null, relationField);
        }

        /**
         * 静态工厂方法：创建带数据的结果
         */
        public static SensitiveDataResult of(String tableName, JSONObject data, String relationField) {
            return new SensitiveDataResult(tableName, data, relationField);
        }

        /**
         * 创建新的结果，替换数据但保持其他字段不变
         */
        public SensitiveDataResult withData(JSONObject newData) {
            return new SensitiveDataResult(tableName, newData, relationField);
        }

        /**
         * 重写toString以提供更安全的字符串表示
         */
        @Override
        public String toString() {
            return "SensitiveDataResult{" +
                    "tableName='" + tableName + '\'' +
                    ", relationField='" + relationField + '\'' +
                    ", data=" + getDataSummary() +
                    '}';
        }
    }
}
