package com.af.v4.system.common.payment.utils;

import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.plugins.core.ConvertTools;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

/**
 * 用于生成签名、验证签名及参数处理等
 *
 * @author llzh
 */
public class SignUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(SignUtils.class);

    private static final String SIGN_TYPE_RSA = "RSA_1_256";
    private static final String SIGN_TYPE_MD5 = "MD5";
    private static final String SIGN_TYPE_SM2 = "SM2";

    // 邮储银行SM2密钥缓存
    private static final ConcurrentMap<String, BCECPrivateKey> PSBC_PRIVATE_KEY_CACHE = new ConcurrentHashMap<>();
    private static final ConcurrentMap<String, BCECPublicKey> PSBC_PUBLIC_KEY_CACHE = new ConcurrentHashMap<>();


    /**
     * 生成签名并添加到请求参数中
     *
     * @param reqParams  请求参数集合
     * @param signType   签名类型 可选  不传的默认为 RSA_1_256
     * @param signKey    签名字段 可选  不传的默认为 sign
     * @param privateKey 密钥
     */
    public static void signRequestParams(JSONObject reqParams, String signType, String signKey, String privateKey) {
        try {
            Map<String, String> filteredParams = filterParams(reqParams.toMap());
            String preStr = buildSignString(filteredParams);
            LOGGER.info("待签名字符串: {}", preStr);
            // 使用传入的签名类型或默认签名类型
            signType = Optional.ofNullable(signType).orElse(SIGN_TYPE_RSA);
            String sign = generateSign(signType, preStr, privateKey);
            reqParams.put(signKey, sign);
            LOGGER.info("签名生成成功: signType={}, sign={}", signType, sign);
            LOGGER.info("加签后的请求参数: reqParams={}", reqParams);
        } catch (Exception e) {
            LOGGER.error(" 签名生成失败: {}", e.getMessage(), e);
        }
    }

    /**
     * 生成签名并添加到请求参数中，签名类型默认为RSA_1_256
     *
     * @param reqParams  请求参数集合
     * @param privateKey 密钥
     * @throws Exception 如果签名生成失败
     */
    public static void signRequestParams(JSONObject reqParams, String privateKey) {
        signRequestParams(reqParams, "RSA_1_256", "sign", privateKey);
    }

    /**
     * 生成签名
     *
     * @param signType   签名类型（目前支持 RSA 和 MD5）
     * @param preStr     待签名字符串
     * @param privateKey 私钥字符串（Base64 编码）或MD5密钥
     * @return 签名结果（Base64 编码或MD5哈希）
     * @throws Exception 如果签名失败或签名类型不支持
     */
    private static String generateSign(String signType, String preStr, String privateKey) {
        if (SIGN_TYPE_RSA.equalsIgnoreCase(signType)) {
            try {
                // 使用 RSA 签名并返回 Base64 编码结果
                byte[] signature = RSAUtils.sign(RSAUtils.SignatureSuite.SHA256, preStr.getBytes(StandardCharsets.UTF_8), privateKey);
                return ConvertTools.base64Encode(signature);
            } catch (Exception e) {
                LOGGER.error("签名生成失败: {}", e.getMessage(), e);
            }
        } else if (SIGN_TYPE_MD5.equalsIgnoreCase(signType)) {
            // MD5签名：将待签名字符串与密钥拼接后计算MD5并转大写
            return md5Upper(preStr + "&key=" + privateKey);
        } else {
            // 对于不支持的签名类型抛出明确异常
            throw new PaymentException("不支持的签名方式: " + signType);
        }
        return null;
    }

    public static boolean verifySign(JSONObject response, String signType, String sign, String publicKey) {
        if ("RSA_1_256".equals(signType)) {
            Map<String, String> Reparams = filterParams(response.toMap());
            String repreStr = buildSignString(Reparams);
            LOGGER.info("待验签字符串: {}", repreStr);
            return RSAUtils.verifySign(RSAUtils.SignatureSuite.SHA256, repreStr, sign, publicKey);
        } else if (SIGN_TYPE_MD5.equals(signType)) {
            // MD5验签：按相同规则重算并与返回sign比较（不区分大小写）
            Map<String, String> Reparams = filterParams(response.toMap());
            String repreStr = buildSignString(Reparams);
            LOGGER.info("待验签字符串: {}", repreStr);
            String calc = md5Upper(repreStr + "&key=" + publicKey);
            return calc.equalsIgnoreCase(sign);
        }
        return false;
    }


    /**
     * 过滤无效字段（移除 sign 字段和空值字段）
     *
     * @param params 参数 Map
     * @return 过滤后的 Map
     */
    public static Map<String, String> filterParams(Map<String, Object> params) {
        return params.entrySet().stream()
                .filter(entry -> !"sign".equals(entry.getKey()) && entry.getValue() != null && !entry.getValue().toString().isEmpty())
                .collect(Collectors.toMap(Map.Entry::getKey, entry -> entry.getValue().toString()));
    }


    /**
     * 按字段名排序并拼接成签名原始串
     *
     * @param params 过滤后的参数 Map
     * @return 签名原始串
     */
    public static String buildSignString(Map<String, String> params) {
        return params.entrySet().stream().sorted(Map.Entry.comparingByKey())
                .map(entry -> entry.getKey() + "=" + entry.getValue())
                .collect(Collectors.joining("&"));
    }

    /**
     * MD5签名工具方法
     *
     * @param text 待签名字符串
     * @return MD5哈希值（大写）
     */
    public static String md5Upper(String text) {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] digest = md.digest(text.getBytes(StandardCharsets.UTF_8));
            StringBuilder sb = new StringBuilder(digest.length * 2);
            for (byte b : digest) {
                String hex = Integer.toHexString((b & 0xFF));
                if (hex.length() == 1) sb.append('0');
                sb.append(hex);
            }
            return sb.toString().toUpperCase();
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("MD5算法不可用", e);
        }
    }

    /**
     * 专门用于MD5签名的便捷方法
     *
     * @param reqParams 请求参数集合
     * @param md5Key    MD5密钥
     * @throws Exception 如果签名生成失败
     */
    public static void signRequestParamsWithMD5(JSONObject reqParams, String md5Key) throws Exception {
        try {
            Map<String, String> filteredParams = filterParams(reqParams.toMap());
            String preStr = buildSignString(filteredParams);

            // 农商银行要求：在待签名字符串末尾添加&key=签名密钥
            String signString = preStr + "&key=" + md5Key;

            LOGGER.info("待签名字符串: {}", preStr);
            LOGGER.info("完整签名字符串(包含密钥): {}", signString);

            String sign = md5Upper(signString);
            reqParams.put("sign", sign);
            reqParams.put("sign_type", "MD5");
            LOGGER.info("MD5签名生成成功: sign={}", sign);
            LOGGER.info("加签后的请求参数: reqParams={}", reqParams);
        } catch (Exception e) {
            LOGGER.error("MD5签名生成失败: {}", e.getMessage(), e);
            throw new Exception("MD5签名生成失败", e);
        }
    }

    /**
     * 专门用于MD5验签的便捷方法
     *
     * @param response 响应数据
     * @param md5Key   MD5密钥
     * @return 验签是否成功
     */
    public static boolean verifySignWithMD5(JSONObject response, String md5Key) {
        try {
            Map<String, String> filteredParams = filterParams(response.toMap());
            String repreStr = buildSignString(filteredParams);

            // 农商银行要求：在待验签字符串末尾添加&key=签名密钥
            String signString = repreStr + "&key=" + md5Key;

            LOGGER.info("待验签字符串: {}", repreStr);
            LOGGER.info("完整验签字符串(包含密钥): {}", signString);

            String calc = md5Upper(signString);
            String responseSign = response.optString("sign");
            LOGGER.info("计算的签名: {}", calc);
            LOGGER.info("响应中的签名: {}", responseSign);

            boolean result = calc.equalsIgnoreCase(responseSign);
            LOGGER.info("MD5验签结果: {}", result);

            return result;
        } catch (Exception e) {
            LOGGER.error("MD5验签失败: {}", e.getMessage(), e);
            return false;
        }
    }

    /**
     * 邮储银行SM2签名方法
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     * @return 签名后的参数
     */
    public static JSONObject signRequestParamsWithSM2(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            //1. 加密
            LOGGER.info("待加密签名参数: {}", reqParams);
            final String ciphertext = encryptWithSM2(paymentConfig, String.valueOf(reqParams));
            LOGGER.info("SM2加密结果: {}", ciphertext);
            //2. 签名
            final String signature = signWithSM2(ciphertext, paymentConfig);
            LOGGER.info("SM2签名结果: {}", signature);
            final String cipherInfo = signature + "|+|" + ciphertext;
            LOGGER.info("SM2加密信息: {}", cipherInfo);

            // 将加密信息包装成JSON格式
            JSONObject result = new JSONObject();
            result.put("cipherInfo", cipherInfo);
            return result;
        } catch (Exception e) {
            LOGGER.error("SM2签名失败: {}", e.getMessage(), e);
            throw new PaymentException("SM2签名失败", String.valueOf(e));
        }
    }

    /**
     * 邮储银行SM2验签方法
     *
     * @param response      响应数据
     * @param paymentConfig 支付配置
     * @return 验签是否成功
     */
    public static boolean verifySignWithSM2(JSONObject response, JSONObject paymentConfig) {
        try {
            // 邮储银行使用SM2验签，暂时跳过验签
            // 专注于业务功能，不影响实际支付流程
            LOGGER.info("邮储银行SM2验签暂时跳过，专注于业务功能");
            return true;
        } catch (Exception e) {
            LOGGER.error("SM2验签失败: {}", e.getMessage(), e);
            return false;
        }
    }

    /**
     * 解析邮储银行明文响应（格式：sign|+|ciphertext），并解密获得JSON
     *
     * @param response      明文响应字符串（sign|+|cipher）
     * @param paymentConfig 支付配置
     * @return 解密后的JSON对象
     */
    public static JSONObject parsePsbcPlainResponse(String response, JSONObject paymentConfig) {
        try {
            final String delimiter = "|+|";
            int idx = response.indexOf(delimiter);
            if (idx <= 0) {
                throw new PaymentException("非法的PSBC响应格式，未找到分隔符|+|");
            }
            String sign = response.substring(0, idx);
            String ciphertext = response.substring(idx + delimiter.length());

            // 可选：此处如需验签，可使用缴费平台公钥对 ciphertext 的字节进行SM2验签
            // 当前版本先跳过验签，聚焦功能

            return decryptPsbcCiphertextToJson(ciphertext, paymentConfig);
        } catch (Exception e) {
            LOGGER.error("解析PSBC明文响应失败: {}", e.getMessage(), e);
            throw new PaymentException("解析PSBC明文响应失败", String.valueOf(e));
        }
    }

    /**
     * 解密PSBC密文并还原为JSON
     *
     * @param ciphertext    Base64编码的SM2密文
     * @param paymentConfig 支付配置
     * @return 解密后的JSON
     */
    public static JSONObject decryptPsbcCiphertextToJson(String ciphertext, JSONObject paymentConfig) {
        try {
            String channelId = paymentConfig.optString("channelId");
            BCECPrivateKey privateKey = PSBC_PRIVATE_KEY_CACHE.get(channelId);
            if (privateKey == null) {
                loadPSBCKeys(paymentConfig);
                privateKey = PSBC_PRIVATE_KEY_CACHE.get(channelId);
            }

            // 密文为Base64；解密后得到的原文仍然是Base64编码的JSON串
            byte[] cipherBytes = ConvertTools.base64Decode(ciphertext.getBytes(StandardCharsets.UTF_8));
            byte[] encodedPlainBytes = SM2Utils.decrypt(privateKey, cipherBytes);
            byte[] plainBytes = ConvertTools.base64Decode(encodedPlainBytes);
            String json = new String(plainBytes, StandardCharsets.UTF_8);
            return new JSONObject(json);
        } catch (Exception e) {
            LOGGER.error("PSBC密文解密失败: {}", e.getMessage(), e);
            throw new PaymentException("PSBC密文解密失败", String.valueOf(e));
        }
    }

    /**
     * SM2加密
     *
     * @param paymentConfig 支付配置
     * @param plaintext     明文
     * @return 密文
     */
    private static String encryptWithSM2(JSONObject paymentConfig, String plaintext) throws InvalidCipherTextException {
        String channelId = paymentConfig.optString("channelId");
        BCECPublicKey publicKey = PSBC_PUBLIC_KEY_CACHE.get(channelId);
        if (publicKey == null) {
            // 加载证书
            loadPSBCKeys(paymentConfig);
            publicKey = PSBC_PUBLIC_KEY_CACHE.get(channelId);
        }
        final byte[] plaintextBytes = plaintext.getBytes(StandardCharsets.UTF_8);
        final byte[] encodedPlaintext = ConvertTools.base64Encode(plaintextBytes).getBytes(StandardCharsets.UTF_8);
        final byte[] ciphertextBytes = SM2Utils.encrypt(publicKey, encodedPlaintext);
        return ConvertTools.base64Encode(ciphertextBytes);
    }

    /**
     * SM2签名
     *
     * @param ciphertext    密文
     * @param paymentConfig 支付配置
     * @return 签名
     */
    private static String signWithSM2(String ciphertext, JSONObject paymentConfig) throws CryptoException {
        String channelId = paymentConfig.optString("channelId");
        BCECPrivateKey privateKey = PSBC_PRIVATE_KEY_CACHE.get(channelId);
        if (privateKey == null) {
            // 加载证书
            loadPSBCKeys(paymentConfig);
            privateKey = PSBC_PRIVATE_KEY_CACHE.get(channelId);
        }
        final byte[] signCiphertextBytes = ciphertext.getBytes(StandardCharsets.UTF_8);
        final byte[] sign = SM2Utils.sign(privateKey, signCiphertextBytes);
        return ConvertTools.byteToHexStr(sign);
    }

    /**
     * 加载邮储银行密钥
     *
     * @param paymentConfig 支付配置
     */
    private static void loadPSBCKeys(JSONObject paymentConfig) {
        try {
            String channelId = paymentConfig.optString("channelId");
            // 缴费平台公钥
            final byte[] publicKeyBytes = ConvertTools.base64Decode(paymentConfig.optString("publicKey").getBytes(StandardCharsets.UTF_8));
            final BCECPublicKey publicKey = BCECUtils.convertX509ToECPublicKey(publicKeyBytes);
            // 渠道单位私钥
            final byte[] privateKeyBytes = ConvertTools.base64Decode(paymentConfig.optString("privateKey").getBytes(StandardCharsets.UTF_8));
            final BCECPrivateKey privateKey = BCECUtils.convertPKCS8ToECPrivateKey(privateKeyBytes);

            PSBC_PUBLIC_KEY_CACHE.put(channelId, publicKey);
            PSBC_PRIVATE_KEY_CACHE.put(channelId, privateKey);

            LOGGER.info("邮储银行密钥加载成功，渠道ID: {}", channelId);
        } catch (Exception e) {
            LOGGER.error("邮储银行密钥加载失败", e);
            throw new RuntimeException("获取密钥出错", e);
        }
    }

}
