package com.af.v4.system.common.payment.utils;

import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.KeyGenerationParameters;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECKeyGenerationParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;


/**
 * 该工具类的方法适用于基于BC库的ECC算法
 *
 * @author Qu Dihuai
 */
public final class BCECUtils
{
	private static final String EC_ALGO_NAME = "EC";
	
	// 静态初始化块，确保BouncyCastle提供程序被注册
	static {
		try {
			if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
				Security.addProvider(new BouncyCastleProvider());
			}
		} catch (Exception e) {
			throw new RuntimeException("无法注册BouncyCastle提供程序", e);
		}
	}

	private BCECUtils()
	{
	}

	/**
	 * 生成ECC密钥对
	 *
	 * @param domainParams
	 * @param random
	 *
	 * @return ECC密钥对
	 */
	public static AsymmetricCipherKeyPair generateKeyPairParameter(final ECDomainParameters domainParams, final SecureRandom random)
	{
		final ECKeyPairGenerator generator = new ECKeyPairGenerator();

		final KeyGenerationParameters params = new ECKeyGenerationParameters(domainParams, random);
		generator.init(params);

		return generator.generateKeyPair();
	}

	/**
	 * @param domainParams
	 * @param random
	 * @return KeyPair
	 * @throws InvalidAlgorithmParameterException
	 */
	public static KeyPair generateKeyPair(final ECDomainParameters domainParams, final SecureRandom random) throws InvalidAlgorithmParameterException
	{
		final ECCurve curve = domainParams.getCurve();
		final ECPoint g = domainParams.getG();
		final BigInteger n = domainParams.getN();
		final BigInteger h = domainParams.getH();
		final ECParameterSpec parameterSpec = new ECParameterSpec(curve, g, n, h);

		final KeyPairGenerator generator = getECKeyPairGenerator();
		generator.initialize(parameterSpec, random);

		return generator.generateKeyPair();
	}

	/**
	 * @param privateKey
	 * @return ECPrivateKeyParameters
	 */
	public static ECPrivateKeyParameters convertPrivateKeyToParameters(final BCECPrivateKey privateKey)
	{
		final ECParameterSpec parameterSpec = privateKey.getParameters();
		final ECCurve curve = parameterSpec.getCurve();
		final ECPoint g = parameterSpec.getG();
		final BigInteger n = parameterSpec.getN();
		final BigInteger h = parameterSpec.getH();
		final ECDomainParameters domainParams = new ECDomainParameters(curve, g, n, h);
		return new ECPrivateKeyParameters(privateKey.getD(), domainParams);
	}

	/**
	 * @param publicKey
	 * @return ECPublicKeyParameters
	 */
	public static ECPublicKeyParameters convertPublicKeyToParameters(final BCECPublicKey publicKey)
	{
		final ECParameterSpec parameterSpec = publicKey.getParameters();

		final ECCurve curve = parameterSpec.getCurve();
		final ECPoint g = parameterSpec.getG();
		final BigInteger n = parameterSpec.getN();
		final BigInteger h = parameterSpec.getH();
		final ECDomainParameters domainParams = new ECDomainParameters(curve, g, n, h);

		final ECPoint q = publicKey.getQ();
		return new ECPublicKeyParameters(q, domainParams);
	}

	/**
	 * 将ECC私钥转换为PKCS8标准的字节流
	 *
	 * @param privateKeyParams
	 * @param publicKeyParams
	 *                                可以为空，但是如果为空的话得到的结果OpenSSL可能解析不了
	 * @return a PKCS8 representation of the key.
	 */
	public static byte[] convertECPrivateKeyToPKCS8(final ECPrivateKeyParameters privateKeyParams, final ECPublicKeyParameters publicKeyParams)
	{
		final ECDomainParameters domainParams = privateKeyParams.getParameters();

		final ECCurve curve = domainParams.getCurve();
		final ECPoint g = domainParams.getG();
		final BigInteger n = domainParams.getN();
		final BigInteger h = domainParams.getH();
		final ECParameterSpec spec = new ECParameterSpec(curve, g, n, h);

		BCECPublicKey publicKey = null;
		if (publicKeyParams != null)
		{
			publicKey = new BCECPublicKey(EC_ALGO_NAME, publicKeyParams, spec, BouncyCastleProvider.CONFIGURATION);
		}

		final BCECPrivateKey privateKey = new BCECPrivateKey(EC_ALGO_NAME, privateKeyParams, publicKey, spec, BouncyCastleProvider.CONFIGURATION);
		return privateKey.getEncoded();
	}

	/**
	 * 将PKCS8标准的私钥字节流转换为私钥对象
	 *
	 * @param pkcs8Key
	 * @return BCECPrivateKey
	 * @throws InvalidKeySpecException
	 */
	public static BCECPrivateKey convertPKCS8ToECPrivateKey(final byte[] pkcs8Key) throws InvalidKeySpecException
	{
		final PKCS8EncodedKeySpec peks = new PKCS8EncodedKeySpec(pkcs8Key);
		final KeyFactory keyFactory = getECKeyFactory();
		return (BCECPrivateKey) keyFactory.generatePrivate(peks);
	}

	/**
	 * 将ECC公钥对象转换为X509标准的字节流
	 *
	 * @param publicKeyParams
	 * @return byte[]
	 */
	public static byte[] convertECPublicKeyToX509(final ECPublicKeyParameters publicKeyParams)
	{
		final ECDomainParameters domainParams = publicKeyParams.getParameters();

		final ECCurve curve = domainParams.getCurve();
		final ECPoint g = domainParams.getG();
		final BigInteger n = domainParams.getN();
		final BigInteger h = domainParams.getH();
		final ECParameterSpec spec = new ECParameterSpec(curve, g, n, h);

		final BCECPublicKey publicKey = new BCECPublicKey(EC_ALGO_NAME, publicKeyParams, spec, BouncyCastleProvider.CONFIGURATION);
		return publicKey.getEncoded();
	}

	/**
	 * 将X509标准的公钥字节流转为公钥对象
	 *
	 * @param x509Bytes
	 * @return BCECPublicKey
	 * @throws InvalidKeySpecException
	 */
	public static BCECPublicKey convertX509ToECPublicKey(final byte[] x509Bytes) throws InvalidKeySpecException
	{
		final X509EncodedKeySpec eks = new X509EncodedKeySpec(x509Bytes);
		final KeyFactory keyFactory = getECKeyFactory();
		return (BCECPublicKey) keyFactory.generatePublic(eks);
	}

	private static KeyFactory getECKeyFactory()
	{
		try
		{
			return KeyFactory.getInstance(EC_ALGO_NAME, BouncyCastleProvider.PROVIDER_NAME);
		}
		catch (final NoSuchAlgorithmException | NoSuchProviderException e)
		{
			// 不会发生
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	private static KeyPairGenerator getECKeyPairGenerator()
	{
		try
		{
			return KeyPairGenerator.getInstance(EC_ALGO_NAME, BouncyCastleProvider.PROVIDER_NAME);
		}
		catch (final NoSuchAlgorithmException | NoSuchProviderException e)
		{
			// 不会发生
			throw new RuntimeException(e.getMessage(), e);
		}
	}
}
