package com.af.v4.system.common.payment.service;

import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.payment.config.PaymentConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.factory.PaymentHandlerFactory;
import com.af.v4.system.common.payment.handler.PaymentHandler;
import com.af.v4.system.common.payment.utils.PaymentUtils;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.redis.RedisService;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author llzh
 * @ClassName PaymentService
 * @Description 支付服务
 * @DateTime 2024/12/24 17:24
 */
@Service
public class PaymentService {

    private static final Logger LOGGER = LoggerFactory.getLogger(PaymentService.class);
    private static final AtomicInteger COUNTER = new AtomicInteger(0);
    private static final int MAX_COUNTER = 99999;
    private final LiuLiConfigService liuLiConfigService;
    private final PaymentConfig paymentConfig;
    private final RedisService redisService;

    public PaymentService(LiuLiConfigService liuLiConfigService, PaymentConfig paymentConfig, RedisService redisService) {
        this.liuLiConfigService = liuLiConfigService;
        this.paymentConfig = paymentConfig;
        this.redisService = redisService;
    }

    /**
     * 注册支付配置
     *
     * @param config 支付配置
     */
    public void registerPaymentConfig(JSONObject config) {
        paymentConfig.setPaymentConfig(config);
        paymentConfig.setBankName(BankName.findByName(config.getString("bankName")));
        paymentConfig.setIntegrationType(IntegrationType.findByDescription(config.getString("integrationType")));
    }

    /**
     * 获取支付配置
     */
    private JSONObject getPaymentConfig(String bankName, String orgName) {
        JSONObject allConfig = liuLiConfigService.get("PaymentConfig", true);
        if (!allConfig.has(bankName)) {
            LOGGER.warn("未获取到银行配置，请检查琉璃中心配置-PaymentConfig");
        }
        JSONObject bankConfig = allConfig.getJSONObject(bankName);
        if (!bankConfig.has(orgName)) {
            LOGGER.warn("未获取到分公司配置，请检查琉璃中心配置-PaymentConfig");
        }
        return bankConfig.getJSONObject(orgName);
    }

    /**
     * 处理支付请求
     * <p>
     * 该方法通过提交支付请求到对应的支付处理器进行异步处理，并等待支付结果
     * 如果支付处理超过预设时间未完成，则自动撤销当前订单，抛出支付异常, 以异常形式返回给调用方,
     *
     * @param request 支付请求对象，包含支付所需的信息
     * @return 返回支付响应对象，包含支付结果
     * @throws PaymentException 当支付处理失败或超时
     */
    public PaymentOrderResponse paymentOrder(PaymentOrderRequest request) {
        try {
            LOGGER.info("处理支付请求 > {}", request.getBankName());
            registerPaymentConfig(getPaymentConfig(request.getBankName(), request.getOrgName()));
            PaymentHandler handler = getPaymentHandler(paymentConfig.getBankName(), paymentConfig.getIntegrationType());
            return handler.paymentOrder(request, paymentConfig.getPaymentConfig());
        } catch (PaymentException e) {
            throw e;
        }
        catch (Exception e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }

    public JSONObject paymentOrder(JSONObject requestJson) {
        try {
            PaymentOrderRequest request = PaymentOrderRequest.Builder.fromJsonObject(requestJson).build();
            return PaymentUtils.toJson(paymentOrder(request));
        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), requestJson.toString(), e.getMessage());
        }
    }

    public QueryPaymentStatusResponse queryPaymentStatus(QueryPaymentStatusRequest request) {
        try {
            registerPaymentConfig(getPaymentConfig(request.getBankName(), request.getOrgName()));
            PaymentHandler handler = getPaymentHandler(paymentConfig.getBankName(), paymentConfig.getIntegrationType());
            return handler.queryPaymentStatus(request, paymentConfig.getPaymentConfig());
        } catch (Exception e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }

    public JSONObject queryPaymentStatus(JSONObject requestJson) {
        try {
            QueryPaymentStatusRequest request = QueryPaymentStatusRequest.Builder.fromJsonObject(requestJson).build();
            return PaymentUtils.toJson(queryPaymentStatus(request));
        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), requestJson.toString(), e.getMessage());
        }
    }

    public CancelPaymentResponse cancelPayment(CancelPaymentRequest request) {
        try {
            registerPaymentConfig(getPaymentConfig(request.getBankName(), request.getOrgName()));
            PaymentHandler handler = getPaymentHandler(paymentConfig.getBankName(), paymentConfig.getIntegrationType());
            return handler.cancelOrder(request, paymentConfig.getPaymentConfig());

        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }

    public JSONObject cancelPayment(JSONObject requestJson) {
        try {
            CancelPaymentRequest request = CancelPaymentRequest.Builder.fromJsonObject(requestJson).build();
            return PaymentUtils.toJson(cancelPayment(request));
        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), requestJson.toString(), e.getMessage());
        }
    }

    public RefundPaymentResponse refundPayment(RefundPaymentRequest request) {
        try {
            registerPaymentConfig(getPaymentConfig(request.getBankName(), request.getOrgName()));
            PaymentHandler handler = getPaymentHandler(paymentConfig.getBankName(), paymentConfig.getIntegrationType());
            return handler.refundPayment(request, paymentConfig.getPaymentConfig());
        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }

    public JSONObject refundPayment(JSONObject requestJson) {
        try {
            RefundPaymentRequest request = RefundPaymentRequest.Builder.fromJsonObject(requestJson).build();
            return PaymentUtils.toJson(refundPayment(request));
        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), requestJson.toString(), e.getMessage());
        }
    }

    public QueryRefundStatusResponse queryRefundStatus(QueryRefundStatusRequest request) {
        try {
            registerPaymentConfig(getPaymentConfig(request.getBankName(), request.getOrgName()));
            PaymentHandler handler = getPaymentHandler(paymentConfig.getBankName(), paymentConfig.getIntegrationType());
            return handler.queryRefundStatus(request, paymentConfig.getPaymentConfig());
        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }

    public JSONObject queryRefundStatus(JSONObject requestJson) {
        try {
            QueryRefundStatusRequest request = QueryRefundStatusRequest.Builder.fromJsonObject(requestJson).build();
            return PaymentUtils.toJson(queryRefundStatus(request));
        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), requestJson.toString(), e.getMessage());
        }
    }

    /**
     * 获取支付处理器
     */
    private PaymentHandler getPaymentHandler(BankName bankName, IntegrationType integrationType) {
        try {
            return PaymentHandlerFactory.getHandler(bankName, integrationType);
        }catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.INVALID_CONNECTION_METHOD_OR_BANK_NAME.getCode(), "{}", PaymentStatus.INVALID_CONNECTION_METHOD_OR_BANK_NAME.getMessage());
        }
    }

    /**
     * 生成订单号
     *
     * @param length 订单号长度（范围 20-30）
     * @return 生成的订单号
     */
    public String generateOrderNumber(int length) {
        String key = STR."order_number_counter_lock_\{Thread.currentThread().threadId()}";
        if (length < 20 || length > 30) {
            throw new IllegalArgumentException("订单号长度必须在 20 到 30 位之间！");
        }
        return redisService.syncLock(key, () -> {
            StringBuilder orderNumber = new StringBuilder("SN");
            String timestamp = DateTools.getNow("yyyyMMddHHmmssSSS");
            orderNumber.append(timestamp);
            int counterValue = COUNTER.updateAndGet(current -> {
                if (current >= MAX_COUNTER) {
                    return 0;
                }
                return current + 1;
            });
            if (counterValue > MAX_COUNTER) {
                // 计数器归零
                COUNTER.set(0);
                counterValue = COUNTER.getAndIncrement();
            }
            // 固定5位，不足补0
            String counterString = String.format("%05d", counterValue);
            orderNumber.append(counterString);

            // 3. 随机数补充，确保总长度满足要求
            int remainingLength = length - orderNumber.length();
            if (remainingLength > 0) {
                String randomString = generateRandomDigits(remainingLength);
                orderNumber.append(randomString);
            }
            return orderNumber.toString();
        });


    }

    /**
     * 生成指定长度的随机数字符串
     *
     * @param length 随机数长度
     * @return 随机数字符串
     */
    public String generateRandomDigits(int length) {
        StringBuilder randomDigits = new StringBuilder();
        for (int i = 0; i < length; i++) {
            int randomDigit = (int) (Math.random() * 10);
            randomDigits.append(randomDigit);
        }
        return randomDigits.toString();
    }

}
