package com.af.v4.system.common.payment.utils;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.plugins.date.DateTools;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.security.SecureRandom;
import java.util.Random;
import java.util.UUID;

/**
 * @author llzh
 * @ClassName PaymentUtils
 * @Description 支付工具类
 * @DateTime 2024/12/24 16:59
 */
public class PaymentUtils {
    private static final Random SECURE_RANDOM = new SecureRandom();
    private static final Logger LOGGER = LoggerFactory.getLogger(PaymentUtils.class);

    // 定义模式枚举
    public enum Mode {
        UPPERCASE_ONLY,         // 仅大写字母
        LOWERCASE_ONLY,         // 仅小写字母
        MIXED_CASE,             // 大小写混合
        DIGITS_ONLY,            // 仅数字
        UPPERCASE_WITH_DIGITS,  // 大写字母和数字混合
        LOWERCASE_WITH_DIGITS,  // 小写字母和数字混合
        MIXED_CASE_WITH_DIGITS  // 大小写字母和数字混合
    }

    /**
     * 生成唯一流水号
     * 格式：YYYYMMDDHHMMSS + 随机6位数字
     */
    public static String generateSerialNumber() {
        // 获取当前时间并格式化
        String timestamp = DateTools.getNow("yyyyMMddHHmmss");

        // 生成6位随机数字
        // 保证6位随机数
        int randomNum = SECURE_RANDOM.nextInt(900000) + 100000;

        return timestamp + randomNum;
    }

    /**
     * 生成唯一订单号
     * 格式：UUID 去除横线
     */
    public static String generateOrderId() {
        try {
            return UUID.randomUUID().toString().replace("-", "").toUpperCase();
        } catch (Exception e) {
            // 记录异常日志
            LOGGER.error("生成订单号时发生异常：{}", e.getMessage());
            throw new RuntimeException("生成订单号失败", e);
        }
    }

    /**
     * 将对象转换为 JSONObject，自动忽略值为 null 的字段
     *
     * @param obj 要转换的对象
     * @return 转换后的 JSONObject
     */
    public static JSONObject toJson(Object obj) {
        JSONObject json = new JSONObject();
        if (obj == null) {
            return json;
        }
        try {
            Field[] fields = obj.getClass().getDeclaredFields();
            for (Field field : fields) {
                // 允许访问私有字段
                field.setAccessible(true);
                Object value = field.get(obj);
                if (value != null) {
                    json.put(field.getName(), value);
                }
            }
        } catch (Exception e) {
            throw new ServiceException("对象转换为 JSON 失败", 500);
        }
        return json;
    }

    /**
     * 生成指定长度的随机字符串
     *
     * @param length    字符串长度
     * @param mode      字符串模式：详见内部枚举
     * @return          随机字符串
     */
    public static String generateRandomString(int length, Mode mode) {
        String charPool = getCharPool(mode);

        SecureRandom random = new SecureRandom();
        StringBuilder randomString = new StringBuilder();

        for (int i = 0; i < length; i++) {
            int randomIndex = random.nextInt(charPool.length());
            char randomChar = charPool.charAt(randomIndex);
            randomString.append(randomChar);
        }

        return randomString.toString();
    }

    /**
     * 生成指定长度的随机字符串(默认包含大小写字母和数字)
     *
     * @param length    字符串长度
     * @return          随机字符串
     */
    public static String generateRandomString(int length) {
        return generateRandomString(length, Mode.MIXED_CASE_WITH_DIGITS);
    }

    /**
     * 根据模式选择字符池
     *
     * @param mode  模式名称
     * @return      字符池
     */
    private static String getCharPool(Mode mode) {
        String uppercaseLetters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        String lowercaseLetters = "abcdefghijklmnopqrstuvwxyz";
        String digits = "0123456789";

        return switch (mode) {
            case UPPERCASE_ONLY -> uppercaseLetters;
            case LOWERCASE_ONLY -> lowercaseLetters;
            case MIXED_CASE -> uppercaseLetters + lowercaseLetters;
            case DIGITS_ONLY -> digits;
            case LOWERCASE_WITH_DIGITS -> lowercaseLetters + digits;
            case UPPERCASE_WITH_DIGITS -> uppercaseLetters + digits;
            default -> uppercaseLetters + lowercaseLetters + digits;
        };
    }
}
