package com.af.v4.system.common.mq.rabbitMQ;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.uuid.IdUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

/**
 * 多连接RabbitMQ生产者实现
 * 支持向不同的RabbitMQ服务器发送消息
 */
@Component
@ConditionalOnProperty(prefix = "rabbitmq", name = "enabled", havingValue = "true")
public class MultiRabbitProducer {

    private static final Logger LOGGER = LoggerFactory.getLogger(MultiRabbitProducer.class);

    private final MultiRabbitConnectionManager connectionManager;

    public MultiRabbitProducer(MultiRabbitConnectionManager connectionManager) {
        this.connectionManager = connectionManager;
    }

    /**
     * 发送消息到指定连接的交换机
     *
     * @param connectionName 连接名称
     * @param exchange       交换机名称
     * @param routingKey     路由键
     * @param message        消息内容
     * @return 结果
     */
    public JSONObject sendToConnection(String connectionName, String exchange, String routingKey, String message) {
        try {
            RabbitTemplate template = connectionManager.getRabbitTemplate(connectionName);

            // 设置消息ID和属性
            MessageProperties properties = new MessageProperties();
            properties.setMessageId(IdUtils.fastSimpleUUID());
            properties.setContentType("text/plain");

            Message msg = new Message(message.getBytes(), properties);

            template.send(exchange, routingKey, msg);

            JSONObject result = new JSONObject();
            result.put("messageId", properties.getMessageId());
            result.put("success", true);
            result.put("connectionName", connectionName);
            result.put("exchange", exchange);
            result.put("routingKey", routingKey);

            LOGGER.debug("[RabbitMQ] 消息发送成功: 连接=[{}], messageId={}", connectionName, properties.getMessageId());

            return result;
        } catch (Exception e) {
            LOGGER.error("[RabbitMQ] 消息发送失败: 连接=[{}], exchange={}, routingKey={}", connectionName, exchange, routingKey, e);
            throw new ServiceException("RabbitMQ消息发送失败: " + e.getMessage());
        }
    }

    /**
     * 发送消息到指定连接的队列（使用默认交换机）
     *
     * @param connectionName 连接名称
     * @param queueName      队列名称
     * @param message        消息内容
     * @return 结果
     */
    public JSONObject sendToQueue(String connectionName, String queueName, String message) {
        return sendToConnection(connectionName, "", queueName, message);
    }

    /**
     * 发送JSON消息到指定连接
     *
     * @param connectionName 连接名称
     * @param exchange       交换机名称
     * @param routingKey     路由键
     * @param data           JSON数据
     * @return 结果
     */
    public JSONObject sendJsonToConnection(String connectionName, String exchange, String routingKey, JSONObject data) {
        try {
            RabbitTemplate template = connectionManager.getRabbitTemplate(connectionName);

            MessageProperties properties = new MessageProperties();
            properties.setMessageId(IdUtils.fastSimpleUUID());
            properties.setContentType("application/json");

            Message msg = new Message(data.toString().getBytes(), properties);

            template.send(exchange, routingKey, msg);

            JSONObject result = new JSONObject();
            result.put("messageId", properties.getMessageId());
            result.put("success", true);
            result.put("connectionName", connectionName);
            result.put("exchange", exchange);
            result.put("routingKey", routingKey);

            LOGGER.debug("[RabbitMQ] JSON消息发送成功: 连接=[{}], messageId={}", connectionName, properties.getMessageId());

            return result;
        } catch (Exception e) {
            LOGGER.error("[RabbitMQ] JSON消息发送失败: 连接=[{}], exchange={}, routingKey={}", connectionName, exchange, routingKey, e);
            throw new ServiceException("RabbitMQ JSON消息发送失败: " + e.getMessage());
        }
    }


    /**
     * 检查连接是否可用
     *
     * @param connectionName 连接名称
     * @return 是否可用
     */
    public boolean isConnectionAvailable(String connectionName) {
        return connectionManager.hasConnection(connectionName);
    }

}