package com.af.v4.system.common.mq.rabbitMQ;

import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.connection.CachingConnectionFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.amqp.support.converter.MessageConverter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 多RabbitMQ连接管理器
 * 负责创建和管理多个RabbitMQ连接、RabbitTemplate和监听器容器工厂
 */
@Component
@ConditionalOnProperty(prefix = "rabbitmq", name = "enabled", havingValue = "true")
public class MultiRabbitConnectionManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(MultiRabbitConnectionManager.class);

    private final MultiRabbitProperties properties;
    private final MessageConverter messageConverter;

    // 连接工厂缓存
    private final Map<String, ConnectionFactory> connectionFactories = new ConcurrentHashMap<>();

    // RabbitTemplate缓存
    private final Map<String, RabbitTemplate> rabbitTemplates = new ConcurrentHashMap<>();

    // 监听器容器工厂缓存
    private final Map<String, SimpleRabbitListenerContainerFactory> containerFactories = new ConcurrentHashMap<>();

    public MultiRabbitConnectionManager(MultiRabbitProperties properties) {
        this.properties = properties;
        this.messageConverter = new Jackson2JsonMessageConverter();
    }

    /**
     * 初始化所有连接
     */
    @PostConstruct
    public void initConnections() {
        // 如果没有配置任何连接，直接返回
        if (properties.getConnections() == null || properties.getConnections().isEmpty()) {
            LOGGER.debug("[RabbitMQ] 未配置任何连接，跳过初始化");
            return;
        }

        properties.getConnections().forEach((connectionName, config) -> {
            if (Boolean.TRUE.equals(config.getEnabled())) {
                try {
                    initSingleConnection(connectionName, config);
                    LOGGER.debug("[RabbitMQ] 连接 [{}] 初始化成功: {}:{}", connectionName, config.getHost(), config.getPort());
                } catch (Exception e) {
                    LOGGER.error("[RabbitMQ] 连接 [{}] 初始化失败: {}:{}", connectionName, config.getHost(), config.getPort(), e);
                }
            } else {
                LOGGER.debug("[RabbitMQ] 连接 [{}] 已禁用，跳过初始化", connectionName);
            }
        });

        LOGGER.info("[RabbitMQ] 初始化完成，共 {} 个连接", connectionFactories.size());
    }

    /**
     * 初始化单个连接
     */
    private void initSingleConnection(String connectionName, MultiRabbitProperties.ConnectionConfig config) {
        // 验证必填参数
        validateRequiredConfig(connectionName, config);

        // 1. 创建连接工厂
        ConnectionFactory connectionFactory = createConnectionFactory(config);
        connectionFactories.put(connectionName, connectionFactory);

        // 2. 创建RabbitTemplate
        RabbitTemplate rabbitTemplate = createRabbitTemplate(connectionFactory);
        rabbitTemplates.put(connectionName, rabbitTemplate);

        // 3. 创建监听器容器工厂
        SimpleRabbitListenerContainerFactory containerFactory = createContainerFactory(connectionFactory, config);
        containerFactories.put(connectionName, containerFactory);
    }

    /**
     * 验证连接配置的必填参数
     */
    private void validateRequiredConfig(String connectionName, MultiRabbitProperties.ConnectionConfig config) {
        if (!StringUtils.hasText(config.getHost())) {
            throw new IllegalArgumentException("RabbitMQ连接 [" + connectionName + "] 缺少必填参数: host");
        }
        if (!StringUtils.hasText(config.getUsername())) {
            throw new IllegalArgumentException("RabbitMQ连接 [" + connectionName + "] 缺少必填参数: username");
        }
        if (!StringUtils.hasText(config.getPassword())) {
            throw new IllegalArgumentException("RabbitMQ连接 [" + connectionName + "] 缺少必填参数: password");
        }
    }

    /**
     * 创建连接工厂
     */
    private ConnectionFactory createConnectionFactory(MultiRabbitProperties.ConnectionConfig config) {
        CachingConnectionFactory factory = new CachingConnectionFactory();

        // 基本连接配置
        factory.setHost(config.getHost());
        factory.setPort(config.getPort());
        factory.setUsername(config.getUsername());
        factory.setPassword(config.getPassword());

        if (StringUtils.hasText(config.getVirtualHost())) {
            factory.setVirtualHost(config.getVirtualHost());
        }

        // 连接超时和心跳配置
        factory.setConnectionTimeout(config.getConnectionTimeout());
        factory.setRequestedHeartBeat(config.getRequestedHeartbeat());

        // 启用自动恢复机制（处理Socket closed等网络异常）
        factory.getRabbitConnectionFactory().setAutomaticRecoveryEnabled(true);
        factory.getRabbitConnectionFactory().setNetworkRecoveryInterval(5000); // 5秒重试间隔
        factory.getRabbitConnectionFactory().setTopologyRecoveryEnabled(true); // 恢复队列、交换机等拓扑

        // 连接池配置
        factory.setChannelCacheSize(25); // 通道缓存大小
        factory.setConnectionCacheSize(10); // 连接缓存大小

        // 基础连接工厂配置完成

        return factory;
    }

    /**
     * 创建RabbitTemplate
     */
    private RabbitTemplate createRabbitTemplate(ConnectionFactory connectionFactory) {
        RabbitTemplate template = new RabbitTemplate(connectionFactory);
        template.setMessageConverter(messageConverter);
        return template;
    }

    /**
     * 创建监听器容器工厂
     */
    private SimpleRabbitListenerContainerFactory createContainerFactory(ConnectionFactory connectionFactory,
                                                                        MultiRabbitProperties.ConnectionConfig config) {
        SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
        factory.setConnectionFactory(connectionFactory);
        factory.setMessageConverter(messageConverter);

        // 设置手动确认模式
        factory.setAcknowledgeMode(org.springframework.amqp.core.AcknowledgeMode.MANUAL);

        // 设置并发消费者数量
        factory.setConcurrentConsumers(config.getConcurrentConsumers());
        factory.setMaxConcurrentConsumers(config.getMaxConcurrentConsumers());

        // 设置预取数量
        factory.setPrefetchCount(config.getPrefetchCount());

        // 设置任务执行器
        factory.setTaskExecutor(new SimpleAsyncTaskExecutor("rabbit-" + hashCode() + "-"));

        // 设置错误处理器
        factory.setErrorHandler(t -> LOGGER.error("[RabbitMQ] 消息处理出错", t));

        return factory;
    }

    /**
     * 获取指定连接的ConnectionFactory
     */
    public ConnectionFactory getConnectionFactory(String connectionName) {
        ConnectionFactory factory = connectionFactories.get(connectionName);
        if (factory == null) {
            throw new IllegalArgumentException("RabbitMQ连接不存在: " + connectionName);
        }
        return factory;
    }

    /**
     * 获取指定连接的RabbitTemplate
     */
    public RabbitTemplate getRabbitTemplate(String connectionName) {
        RabbitTemplate template = rabbitTemplates.get(connectionName);
        if (template == null) {
            throw new IllegalArgumentException("RabbitMQ连接不存在: " + connectionName);
        }
        return template;
    }

    /**
     * 获取指定连接的监听器容器工厂
     */
    public SimpleRabbitListenerContainerFactory getContainerFactory(String connectionName) {
        SimpleRabbitListenerContainerFactory factory = containerFactories.get(connectionName);
        if (factory == null) {
            throw new IllegalArgumentException("RabbitMQ连接不存在: " + connectionName);
        }
        return factory;
    }

    /**
     * 获取所有连接名称
     */
    public Map<String, ConnectionFactory> getAllConnectionFactories() {
        return new HashMap<>(connectionFactories);
    }

    /**
     * 获取所有RabbitTemplate
     */
    public Map<String, RabbitTemplate> getAllRabbitTemplates() {
        return new HashMap<>(rabbitTemplates);
    }

    /**
     * 检查连接是否存在
     */
    public boolean hasConnection(String connectionName) {
        return connectionFactories.containsKey(connectionName);
    }

}