package com.af.v4.system.common.mq.rabbitMQ;

import com.af.v4.system.common.core.proxy.logic.ILogicServiceProxy;
import com.af.v4.system.common.core.utils.uuid.IdUtils;
import com.rabbitmq.client.Channel;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerEndpoint;
import org.springframework.amqp.rabbit.listener.RabbitListenerEndpointRegistry;
import org.springframework.amqp.rabbit.listener.api.ChannelAwareMessageListener;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 统一消息处理器
 * 支持多连接动态监听，所有消息统一入口处理
 */
@Component
@ConditionalOnProperty(prefix = "rabbitmq", name = "enabled", havingValue = "true")
public class UnifiedMessageProcessor {

    private static final Logger LOGGER = LoggerFactory.getLogger(UnifiedMessageProcessor.class);

    private final MultiRabbitProperties properties;
    private final MultiRabbitConnectionManager connectionManager;
    private final RabbitListenerEndpointRegistry registry;
    private final ILogicServiceProxy logicService;

    // 监听器计数器
    private final AtomicInteger listenerCounter = new AtomicInteger(0);

    public UnifiedMessageProcessor(MultiRabbitProperties properties,
                                   MultiRabbitConnectionManager connectionManager,
                                   RabbitListenerEndpointRegistry registry,
                                   ILogicServiceProxy logicService) {
        this.properties = properties;
        this.connectionManager = connectionManager;
        this.registry = registry;
        this.logicService = logicService;
    }

    /**
     * 应用完全启动后，为所有配置的队列动态创建监听器
     * 等到所有Spring Boot自动配置完成后再执行
     */
    @EventListener(ApplicationReadyEvent.class)
    public void initializeRabbitListeners() {
        // 如果没有配置任何连接，直接返回
        if (properties.getConnections() == null || properties.getConnections().isEmpty()) {
            LOGGER.info("[RabbitMQ] 未配置任何RabbitMQ连接，跳过初始化");
            return;
        }

        LOGGER.info("[RabbitMQ] 开始动态注册RabbitMQ消息监听器...");

        int totalListeners = 0;

        // 为每个连接的每个队列创建监听器
        for (Map.Entry<String, MultiRabbitProperties.ConnectionConfig> entry : properties.getConnections().entrySet()) {
            String connectionName = entry.getKey();
            MultiRabbitProperties.ConnectionConfig config = entry.getValue();

            if (Boolean.TRUE.equals(config.getEnabled()) && !config.getListenQueues().isEmpty()) {
                for (String queueName : config.getListenQueues()) {
                    if (StringUtils.hasText(queueName)) {
                        registerDynamicListener(connectionName, queueName);
                        totalListeners++;
                    }
                }
                LOGGER.info("[RabbitMQ] 连接 [{}] 注册了 {} 个队列监听器", connectionName, config.getListenQueues().size());
            }
        }

        LOGGER.info("动态注册RabbitMQ监听器完成，共注册 {} 个监听器", totalListeners);
    }

    /**
     * 为指定连接和队列动态注册监听器
     */
    private void registerDynamicListener(String connectionName, String queueName) {
        try {
            // 检查连接是否存在
            if (!connectionManager.hasConnection(connectionName)) {
                LOGGER.error("[RabbitMQ] 连接 [{}] 不存在，无法注册队列 [{}] 的监听器", connectionName, queueName);
                return;
            }

            // 创建监听器端点
            SimpleRabbitListenerEndpoint endpoint = new SimpleRabbitListenerEndpoint();
            endpoint.setId(generateListenerId(connectionName, queueName));
            endpoint.setQueueNames(queueName);

            // 设置消息监听器（支持手动确认）
            endpoint.setMessageListener((ChannelAwareMessageListener) (message, channel) -> processUnifiedMessage(message, channel, connectionName, queueName));

            // 获取对应连接的容器工厂
            SimpleRabbitListenerContainerFactory containerFactory = connectionManager.getContainerFactory(connectionName);

            // 注册监听器
            registry.registerListenerContainer(endpoint, containerFactory, true);

            LOGGER.info("[RabbitMQ] 成功注册监听器: 连接=[{}], 队列=[{}], ID=[{}]",
                    connectionName, queueName, endpoint.getId());

        } catch (Exception e) {
            LOGGER.error("[RabbitMQ] 注册监听器失败: 连接=[{}], 队列=[{}]", connectionName, queueName, e);
        }
    }

    /**
     * 统一消息处理入口
     * 所有来自不同连接、不同队列的消息都会进入这个方法
     */
    private void processUnifiedMessage(Message message, Channel channel, String connectionName, String queueName) {
        String messageId = IdUtils.fastSimpleUUID();
        long deliveryTag = message.getMessageProperties().getDeliveryTag();

        LOGGER.info("[RabbitMQ] 接收消息: 连接=[{}], 队列=[{}], 消息ID=[{}], 投递标签=[{}]",
                connectionName, queueName, messageId, deliveryTag);

        try {
            // 构造统一的数据格式传递给Logic
            JSONObject data = buildMessageData(message, connectionName, queueName, messageId);

            // 直接使用队列名作为Logic名称

            LOGGER.debug("[RabbitMQ] 调用Logic处理消息: logic=[{}], 连接=[{}], 队列=[{}]", queueName, connectionName, queueName);

            // 执行Logic处理
            Object result = logicService.run(queueName, data.toString());

            // 手动确认消息（ACK）
            channel.basicAck(deliveryTag, false);

            LOGGER.info("[RabbitMQ] 消息处理成功并已确认: 连接=[{}], 队列=[{}], Logic=[{}]", connectionName, queueName, queueName);

        } catch (Exception e) {
            LOGGER.error("[RabbitMQ] 消息处理失败: 队列=[{}], Logic=[{}], 错误: {}", queueName, queueName, e.getMessage(), e);

            try {
                // 消息处理失败时，拒绝消息并重新入队（可配置是否重新入队）
                boolean requeue = shouldRequeueOnError(e);
                channel.basicNack(deliveryTag, false, requeue);

                LOGGER.warn("[RabbitMQ] 消息已拒绝: 队列=[{}], 重新入队=[{}]", queueName, requeue);

            } catch (Exception ackException) {
                LOGGER.error("[RabbitMQ] 消息确认操作失败: 队列=[{}]", queueName, ackException);
            }
        }
    }

    /**
     * 判断消息处理失败时是否应该重新入队
     * 可根据异常类型或业务需求自定义逻辑
     */
    private boolean shouldRequeueOnError(Exception e) {
        // 默认不重新入队，避免无限循环
        // 可根据具体异常类型决定是否重新入队
        // 参数错误类异常通常不应重新入队
        // 其他异常可以考虑重新入队，但建议设置最大重试次数
        return false;
    }

    /**
     * 构造传递给Logic的消息数据
     */
    private JSONObject buildMessageData(Message message, String connectionName, String queueName, String messageId) {
        JSONObject data = new JSONObject();

        // 基本信息
        data.put("messageId", messageId);
        data.put("connectionName", connectionName);
        data.put("queueName", queueName);
        data.put("message", new String(message.getBody()));
        data.put("timestamp", System.currentTimeMillis());

        // 消息属性
        if (message.getMessageProperties() != null) {
            data.put("exchange", message.getMessageProperties().getReceivedExchange());
            data.put("routingKey", message.getMessageProperties().getReceivedRoutingKey());
            data.put("contentType", message.getMessageProperties().getContentType());

            // 消息头
            if (message.getMessageProperties().getHeaders() != null) {
                data.put("headers", new JSONObject(message.getMessageProperties().getHeaders()));
            }
        }

        return data;
    }


    /**
     * 生成监听器ID
     */
    private String generateListenerId(String connectionName, String queueName) {
        return String.format("rabbit-listener-%s-%s-%d",
                connectionName, queueName.replace(".", "_"), listenerCounter.incrementAndGet());
    }
}
