package com.af.v4.system.common.mq;

import com.af.v4.system.common.core.proxy.logic.ILogicServiceProxy;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.log.enums.OperatorType;
import com.af.v4.system.common.log.service.LogResolveService;
import com.af.v4.system.common.plugins.core.ConvertTools;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.spring.annotation.RocketMQMessageListener;
import org.apache.rocketmq.spring.core.RocketMQListener;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;

/**
 * Logic消息消费监听器
 *
 * @author Mr.river
 */
@Component
@ConditionalOnProperty(prefix = "rocketmq", name = "name-server")
@RocketMQMessageListener(topic = RocketMQProducer.DEFAULT_LOGIC_TOPIC, consumerGroup = "${spring.application.name:}-GROUP")
public class LogicMessageListener implements RocketMQListener<MessageExt> {
    /**
     * 日志对象
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(LogicMessageListener.class);
    private final ILogicServiceProxy logicService;
    private final LogResolveService logResolveService;

    public LogicMessageListener(ILogicServiceProxy logicService, LogResolveService logResolveService) {
        this.logicService = logicService;
        this.logResolveService = logResolveService;
    }

    @Override
    public void onMessage(MessageExt message) {
        long beginTime = System.currentTimeMillis();
        //消费Logic请求
        String logicName = message.getTags();
        String body = new String(message.getBody(), StandardCharsets.UTF_8);
        JSONObject content = new JSONObject(
                new String(ConvertTools.base64Decode(
                        new JSONObject(body).getString("body").getBytes(StandardCharsets.UTF_8)
                ), StandardCharsets.UTF_8)
        );
        try {
            logicService.run(logicName, content);
            long endTime = System.currentTimeMillis();
            handleLog(logicName, content, null, endTime - beginTime);
        } catch (Exception e) {
            LOGGER.error("【{}】出现异常：", logicName, e);
            long endTime = System.currentTimeMillis();
            handleLog(logicName, content, e, endTime - beginTime);
            throw e;
        }
    }

    protected void handleLog(String logicName, Object operParams, final Throwable e, long costTime) {
        logResolveService.resolveLog(
                this.getClass().getName(),
                "onMessage",
                BusinessType.LOGIC.name(),
                STR."MQ消费Logic：\{logicName}",
                OperatorType.SYSTEM.name(),
                operParams.toString(),
                costTime,
                e
        );
    }
}
