package com.af.v4.system.common.liuli.config.subscriber;

import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.redis.RedisService;
import com.github.benmanes.caffeine.cache.Cache;
import org.json.JSONObject;
import org.jspecify.annotations.NonNull;
import org.redisson.api.RTopic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 琉璃缓存清理订阅
 *
 * @author Mr.river
 * @since 1.1.14.17
 */
@Component
public class LiuLiCacheSubscriber implements ApplicationRunner {

    /**
     * 订阅发布topicName - 清除redis缓存消息
     */
    public static final String CLEAR_CACHE_TOPIC_NAME = "Topic_LiuLi_ClearCache";
    private static final Logger LOGGER = LoggerFactory.getLogger(LiuLiCacheSubscriber.class);
    private final RedisService redisService;

    public LiuLiCacheSubscriber(RedisService redisService) {
        this.redisService = redisService;
    }

    @Override
    public void run(ApplicationArguments args) {
        RTopic topic = redisService.getTopic(CLEAR_CACHE_TOPIC_NAME);
        topic.addListener(JSONObject.class, (charSequence, object) -> {
            // 存放配置的key
            String namespaceName = object.getString("namespaceName");
            String environment = object.getString("environment");
            String configName = object.getString("configName");
            String configCacheKey = LiuLiConfigService.getConfigCacheKey("*", namespaceName, environment, configName);
            // 删除本地缓存
            Cache<@NonNull String, JSONObject> cache = LiuLiConfigService.getLiuliConfigLocalCache();
            List<String> keys = cache.asMap().keySet().stream()
                    .filter(str -> isMatch(str, configCacheKey))
                    .toList();
            LOGGER.info("待清理的配置KEY:{}，匹配到的集合：{}", configCacheKey, keys);
            keys.forEach(cache::invalidate);
        });
    }

    /**
     * 给定一个字符串 (s) 和一个字符模式 (p) ，实现一个支持 '?' 和 '*' 的通配符匹配。
     * 使用动态规划
     *
     * @param ss 待匹配的字符串
     * @param pp 字符模式
     * @return 匹配结果
     */
    private boolean isMatch(String ss, String pp) {
        int n = ss.length(), m = pp.length();
        // 技巧：往原字符头部插入空格，这样得到 char 数组是从 1 开始，而且可以使得 f[0][0] = true，可以将 true 这个结果滚动下去
        ss = STR." \{ss}";
        pp = STR." \{pp}";
        char[] s = ss.toCharArray();
        char[] p = pp.toCharArray();
        // f(i,j) 代表考虑 s 中的 1~i 字符和 p 中的 1~j 字符 是否匹配
        boolean[][] f = new boolean[n + 1][m + 1];
        f[0][0] = true;
        for (int i = 0; i <= n; i++) {
            for (int j = 1; j <= m; j++) {
                if (p[j] == '*') {
                    f[i][j] = f[i][j - 1] || (i - 1 >= 0 && f[i - 1][j]);
                } else {
                    f[i][j] = i - 1 >= 0 && f[i - 1][j - 1] && (s[i] == p[j] || p[j] == '?');
                }
            }
        }
        return f[n][m];
    }
}
