package com.af.v4.system.common.jpa.sensitive;

import com.af.v4.system.common.jpa.types.MetaData;
import com.af.v4.system.common.plugins.security.SubstitutionCipherUtil;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import java.util.Set;


/**
 * 敏感数据拦截器
 * 作为EntityServer与敏感数据处理系统的唯一接口
 * <p>
 * 设计原则：
 * - 最小侵入：EntityServer只需调用一个方法
 * - 可插拔：可以通过配置完全禁用敏感数据处理
 * - 职责单一：只负责拦截和委托处理
 * - 现代优化：使用Stream API、Pattern Matching和Switch表达式
 */
@Component
@RefreshScope
@ConfigurationProperties(prefix = "security.sensitive")
public class SensitiveDataInterceptor {

    private static final Logger log = LoggerFactory.getLogger(SensitiveDataInterceptor.class);
    private static final String PROTECTED_ROW_LOG = "[PROTECTED]";

    /**
     * 是否启用敏感数据处理功能（来自配置 security.sensitive.enabled）
     */
    private boolean enabled = true;

    public boolean isEnabled() {
        return enabled;
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    /**
     * 在数据保存前处理所有敏感字段：统一在原字段就地写入替代密码密文
     *
     * @param entityName 实体名称（业务表名）
     * @param row        数据行对象（字段会在此就地修改）
     * @param metaData   实体元数据（包含敏感字段配置）
     */
    public void beforeSave(
            String entityName,
            JSONObject row,
            MetaData metaData) {

        // 功能开关检查
        if (!this.enabled) {
            return;
        }

        // 配置检查
        if (metaData == null || !metaData.hasMetaAttributes()) {
            return;
        }

        // 获取所有敏感字段
        Set<String> sensitiveFields = metaData.getSensitiveFieldNames();
        if (sensitiveFields.isEmpty()) {
            return;
        }

        try {
            // 就地处理：检测 -> 加密回写/跳过
            applySensitiveFieldEncryption(entityName, row, sensitiveFields);

        } catch (Exception e) {
            // 敏感数据处理失败不应该影响主业务流程
            handleProcessingError(entityName, row, e);
        }
    }

    /**
     * 统一错误处理方法
     * 提供一致的错误日志记录和异常处理
     */
    private void handleProcessingError(String entityName,
                                       JSONObject row, Exception e) {
        String rowInfo = (row != null) ? PROTECTED_ROW_LOG : "null";
        log.error("敏感数据处理失败，将跳过处理: {} -> {}, row={}", "beforeSave处理失败", entityName, rowInfo, e);
    }

    /**
     * 处理单个字段错误
     */
    private void handleFieldError(String fieldName, Exception e) {
        log.warn("{} 字段处理出错: {} - {}", "字段加密回写", fieldName, e.getMessage());
        if (log.isDebugEnabled()) {
            log.debug("{} 字段处理详细错误信息:", "字段加密回写", e);
        }
    }

    /**
     * 在原字段就地处理敏感数据：已脱敏则跳过，已是密文则跳过，否则加密写回
     */
    private void applySensitiveFieldEncryption(String entityName, JSONObject row,
                                               Set<String> sensitiveFields) {

        log.debug("处理敏感字段(就地加密): {}, 字段数={}", entityName, sensitiveFields.size());

        sensitiveFields.stream()
                .filter(fieldName -> row.opt(fieldName) != null)
                .forEach(fieldName -> processSingleField(fieldName, row));
    }

    /**
     * 处理单个敏感字段
     *
     * @param fieldName 字段名
     * @param row       数据行
     */
    private void processSingleField(String fieldName, JSONObject row) {
        String val = row.opt(fieldName).toString();

        if (val.isBlank()) {
            return;
        }

        // 1) 已脱敏输入（包含 *）=> 跳过该字段的更新，避免覆盖真实密文
        if (isMaskedValue(val)) {
            row.remove(fieldName);
            log.debug("检测到已脱敏输入，跳过字段更新: {}", fieldName);
            return;
        }

        // 2) 已是替代密码密文（严格判定）=> 跳过加密，保留原值
        if (SubstitutionCipherUtil.isStrictCipher(val)) {
            log.debug("检测到已是替代密码密文，跳过加密: {} (版本: {})",
                    fieldName, SubstitutionCipherUtil.parseVersion(val));
            return;
        }

        // 3) 明文 => 加密后回写到原字段
        try {
            String cipher = SubstitutionCipherUtil.encrypt(val);
            if (cipher != null) {
                row.put(fieldName, cipher);
                log.debug("敏感字段已加密写回: {} (版本: {}, 原文长度: {}, 密文长度: {})",
                        fieldName,
                        SubstitutionCipherUtil.getCurrentVersion(),
                        val.length(),
                        cipher.length());
            }
        } catch (Exception ex) {
            handleFieldError(fieldName, ex);
        }
    }

    /**
     * 判定是否为已脱敏输入（包含 *）
     */
    private boolean isMaskedValue(String value) {
        return value != null && value.indexOf('*') >= 0;
    }

}
