package com.af.v4.system.common.jpa.action;

import com.af.v4.system.common.core.constant.CodeNormsConstants;
import com.af.v4.system.common.core.exception.CheckedException;
import com.af.v4.system.common.core.proxy.jpa.IQueryParamsProxy;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.jpa.session.SessionPool;
import com.af.v4.system.common.jpa.transformer.StandardAliasTransformer;
import com.af.v4.system.common.jpa.utils.DesensitizationTool;
import com.af.v4.system.common.jpa.utils.QueryParams;
import org.hibernate.Session;
import org.hibernate.exception.SQLGrammarException;
import org.hibernate.query.NativeQuery;
import org.hibernate.query.TupleTransformer;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Component;
import org.springframework.util.StopWatch;
import java.util.List;
import java.util.function.Supplier;

/**
 * SQL执行器
 *
 * @author Mr.river
 */
@Component
public class SqlAction {

    private static final Logger LOGGER = LoggerFactory.getLogger(SqlAction.class);
    private final SessionPool sessionPool;
    private DesensitizationTool desensitizationTool;

    public SqlAction(SessionPool sessionPool, DesensitizationTool desensitizationTool) {
        this.sessionPool = sessionPool;
        this.desensitizationTool = desensitizationTool;
    }

    /**
     * 执行SQL查询
     *
     * @param name        SQL标识名
     * @param sql         SQL语句
     * @param queryParams 查询参数
     * @return 结果集
     */
    public JSONArray query(String name, String sql, IQueryParamsProxy queryParams) {
        return (JSONArray) sqlMonitor(() -> {
            InnerSQLCall sqlCall = new InnerSQLCall(sql, queryParams, StandardAliasTransformer.INSTANCE);
            JSONArray array = sqlCall.doInHibernate(sessionPool.getSession());
            writeSqlResult(name, array);
            array = desensitizationTool.processEncryption(array);
            return array;
        }, name, queryParams);
    }

    /**
     * 执行SQL增删改
     *
     * @param name       SQL标识名
     * @param sql        SQL语句
     * @param dataSource 数据源
     * @return 执行结果
     */
    public Integer exec(String name, String sql, String dataSource) {
        return (int) sqlMonitor(() -> {
            Session session = sessionPool.getSession();
            NativeQuery<?> queryObject = session.createNativeQuery(sql);
            return queryObject.executeUpdate();
        }, name, new QueryParams.Builder().dataSource(dataSource).build());
    }

    /**
     * SQL监控包装
     *
     * @param supplier    SQL具体执行业务
     * @param name        SQL标识名
     * @param queryParams 查询参数
     * @return 执行结果
     */
    private Object sqlMonitor(Supplier<Object> supplier, String name, IQueryParamsProxy queryParams) {
        if (StringUtils.isEmpty(name)) {
            throw new CheckedException(CodeNormsConstants.E_01);
        }
        LOGGER.info(STR."\{CodeNormsConstants.DEBUG_PREFIX}执行SQL[{}]，查询参数：{}", name, queryParams);
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        try {
            return DynamicDataSource.withDataSource(queryParams.getDataSource(), supplier);
        } finally {
            stopWatch.stop();
            long time = stopWatch.getTotalTimeMillis();
            LOGGER.info(STR."\{CodeNormsConstants.DEBUG_PREFIX}SQL[{}]耗时：{}ms", name, time);
        }
    }

    /**
     * 打印SQL执行结果
     *
     * @param name  SQL名称
     * @param array 执行结果
     */
    private void writeSqlResult(String name, JSONArray array) {
        String result = array.toString();
        int length = result.length();
        if (length > 2000) {
            result = STR."\{result.substring(0, 2000)}...后续还有\{length - 2000}个字符";
        }
        LOGGER.info(STR."""
\{CodeNormsConstants.DEBUG_PREFIX}SQL[{}]查询结果:\s
{}""", name, result);
    }

    private record InnerSQLCall(String sql, IQueryParamsProxy queryParams,
                                TupleTransformer<?> transformer) implements HibernateCallback<JSONArray> {

        @Override
        public JSONArray doInHibernate(Session session) {
            NativeQuery<?> q = session.createNativeQuery(sql);
            // 页数
            Integer pageNo = queryParams.getPageNo();
            // 页大小
            Integer pageSize = queryParams.getPageSize();
            // 预编译查询参数
            JSONObject parameterList = queryParams.getParameterList();
            if (parameterList != null) {
                for (String key : parameterList.keySet()) {
                    Object value = parameterList.get(key);
                    if (value instanceof JSONArray list) {
                        q.setParameter(key, list.toList());
                    } else if (value instanceof JSONObject object) {
                        q.setParameter(key, object.toMap());
                    } else {
                        q.setParameter(key, value);
                    }
                }
            }
            q.setTupleTransformer(transformer);
            try {
                List<?> list;

                /* 检查并解析分页参数 **/
                // 检查 pageSize 是否为 null 或小于等于 0
                if (pageSize == null || pageSize <= 0) {
                    // 如果 pageSize 无效，则获取完整列表，不进行分页
                    list = q.list();
                } else if (pageNo == null || pageNo < 0) {
                    // 如果 pageNo 无效（null 或小于 0），则从第一页（第 0 页）开始
                    // 设置查询结果的起始位置为 0，最多返回 pageSize 条数据
                    list = q.setFirstResult(0).setMaxResults(pageSize).list();
                } else {
                    int resolvedPageNo = pageNo > 0 ? pageNo - 1 : 0;
                    // 设置查询结果的起始位置为根据页面计算出的值，限制返回的数据为 pageSize 条
                    list = q.setFirstResult(resolvedPageNo * pageSize)
                            .setMaxResults(pageSize)
                            .list();
                }


                return new JSONArray(list);
            } catch (SQLGrammarException ex) {
                // 把sql语句添加到异常信息中
                String msg = STR."""
sql:
\{sql}
\{ex.getMessage()}""";
                throw new SQLGrammarException(msg, ex.getSQLException());
            }
        }
    }
}
