package com.af.v4.system.common.jpa.service;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.proxy.jpa.IQueryParamsProxy;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.expression.Expression;
import com.af.v4.system.common.jpa.action.SqlAction;
import com.af.v4.system.common.jpa.utils.QueryParams;
import com.af.v4.system.common.resource.mapper.PluginMapper;
import com.af.v4.system.common.resource.mapper.SqlMapper;
import com.af.v4.system.common.security.auth.AuthLogic;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Iterator;

@Service
@Transactional(rollbackFor = Exception.class)
public class SqlService {
    private static final Logger LOGGER = LoggerFactory.getLogger(SqlService.class);

    private final SqlMapper sqlMapper;

    private final SqlAction sqlAction;

    private final ApplicationService applicationService;

    private final AuthLogic authLogic = new AuthLogic();

    public SqlService(SqlMapper sqlMapper, SqlAction sqlAction, ApplicationService applicationService) {
        this.sqlMapper = sqlMapper;
        this.sqlAction = sqlAction;
        this.applicationService = applicationService;
    }

    /**
     * 获取SQL文件的编译后内容
     *
     * @param name   SQL文件映射名
     * @param params 参数
     * @return SQL语句
     */
    public String getSQL(String name, JSONObject params) {
        return this.getSQLByFile(name, params);
    }

    /**
     * 获取合计执行结果
     *
     * @param name           SQL文件映射名
     * @param params         参数
     * @param dataSourceName 数据源名称
     * @return 结果集
     */
    public JSONObject queryTotal(String name, JSONObject params, String dataSourceName) {
        // 获取求和字段等内容
        Object sums = null;
        if (params.has("sums")) {
            sums = params.get("sums");
        }
        // 产生sql语句编译后的结果
        String sql = this.getSQLByFile(name, params);

        // 求和时，order by会导致sql错误，过滤掉order by部分。
        sql = filterOutOrderBy(sql, sums);
        IQueryParamsProxy queryParams = new QueryParams.Builder().dataSource(dataSourceName).build();
        JSONArray array = sqlAction.query(name, sql, queryParams);
        return array.getJSONObject(0);
    }

    public JSONObject queryTotal(String name, JSONObject params) {
        return queryTotal(name, params, null);
    }

    public JSONObject queryTotal(String name, String str, String dataSourceName) {
        return queryTotal(name, StringUtils.isEmpty(str) ? new JSONObject() : new JSONObject(str), dataSourceName);
    }

    public JSONObject queryTotal(String name, String str) {
        return queryTotal(name, str, null);
    }

    /**
     * 根据SQL文件执行查询
     *
     * @param name           SQL文件映射名
     * @param params         参数
     * @param pageNo         页码
     * @param pageSize       每页大小
     * @param dataSourceName 数据源名称
     * @return 结果集
     */
    public JSONArray query(String name, JSONObject params, Integer pageNo, Integer pageSize, String dataSourceName) {
        // 产生SQL语句编译后的结果
        String sql = this.getSQLByFile(name, params);

        IQueryParamsProxy queryParams = new QueryParams.Builder()
                .pageNo(pageNo).pageSize(pageSize).dataSource(dataSourceName).build();
        return sqlAction.query(name, sql, queryParams);
    }

    public JSONArray query(String name, JSONObject params, Integer pageNo, Integer pageSize) {
        return query(name, params, pageNo, pageSize, null);
    }

    public JSONArray query(String name, JSONObject params, Integer pageSize, String dataSourceName) {
        return query(name, params, 1, pageSize, dataSourceName);
    }

    public JSONArray query(String name, JSONObject params, Integer pageSize) {
        return query(name, params, 1, pageSize);
    }

    public JSONArray query(String name, JSONObject params, String dataSourceName) {
        return query(name, params, null, null, dataSourceName);
    }

    public JSONArray query(String name, JSONObject params) {
        return query(name, params, null, null, null);
    }

    /**
     * 根据指定名称的SQL语句执行查询
     *
     * @param name           SQL查询标识名
     * @param sql            SQL语句
     * @param dataSourceName 数据源名称
     * @return 结果集
     * @apiNote 注意：不推荐依赖该方法作为SQL查询，因为该方法没有校验name参数的唯一性，这会给排查SQL问题的同事带来麻烦
     */
    @Deprecated(since = "2.8.0")
    public JSONArray querySQL(String name, String sql, String dataSourceName) {
        IQueryParamsProxy queryParams = new QueryParams.Builder().dataSource(dataSourceName).build();
        return querySQL(name, sql, queryParams);
    }

    public JSONArray querySQL(String name, String sql) {
        IQueryParamsProxy queryParams = new QueryParams.Builder().build();
        return querySQL(name, sql, queryParams);
    }

    @Deprecated(since = "2.8.0")
    public JSONArray querySQL(String name, String sql, Integer pageSize, String dataSourceName) {
        IQueryParamsProxy queryParams = new QueryParams.Builder()
                .pageSize(pageSize).dataSource(dataSourceName).build();
        return querySQL(name, sql, queryParams);
    }

    @Deprecated(since = "2.8.0")
    public JSONArray querySQL(String name, String sql, Integer pageSize) {
        IQueryParamsProxy queryParams = new QueryParams.Builder()
                .pageSize(pageSize).build();
        return querySQL(name, sql, queryParams);
    }

    @Deprecated(since = "2.8.0")
    public JSONArray querySQL(String name, String sql, Integer pageNo, Integer pageSize, String dataSourceName) {
        IQueryParamsProxy queryParams = new QueryParams.Builder()
                .pageNo(pageNo).pageSize(pageSize).dataSource(dataSourceName).build();
        return querySQL(name, sql, queryParams);
    }

    @Deprecated(since = "2.8.0")
    public JSONArray querySQL(String name, String sql, Integer pageNo, Integer pageSize) {
        IQueryParamsProxy queryParams = new QueryParams.Builder()
                .pageNo(pageNo).pageSize(pageSize).build();
        return querySQL(name, sql, queryParams);
    }

    /**
     * 根据指定名称的SQL语句执行查询
     *
     * @param name           SQL查询标识名
     * @param sql            SQL语句
     * @param queryParamsObj 查询参数
     * @return 执行结果
     */
    public JSONArray querySQL(String name, String sql, JSONObject queryParamsObj) {
        IQueryParamsProxy queryParams = QueryParams.Builder.fromJSONObject(queryParamsObj).build();
        return sqlAction.query(name, sql, queryParams);
    }

    public JSONArray querySQL(String name, String sql, IQueryParamsProxy queryParams) {
        return sqlAction.query(name, sql, queryParams);
    }

    /**
     * 根据SQL文件执行增删改
     *
     * @param name           SQL文件映射名
     * @param params         参数
     * @param dataSourceName 数据源名称
     * @return 执行结果
     */
    public Integer exec(String name, JSONObject params, String dataSourceName) {
        // 产生sql语句编译后的结果
        String sql = this.getSQLByFile(name, params);
        return execSQL(name, sql, dataSourceName);
    }

    public Integer exec(String name, JSONObject params) {
        return exec(name, params, null);
    }

    /**
     * 根据SQL语句执行增删改
     *
     * @param name           SQL查询标识名
     * @param sql            SQL语句
     * @param dataSourceName 数据源名称
     * @return 影响行数
     * @apiNote 注意：不推荐依赖该方法作为SQL查询，因为该方法没有校验name参数的唯一性，这会给排查SQL问题的同事带来麻烦
     */
    public Integer execSQL(String name, String sql, String dataSourceName) {
        return sqlAction.exec(name, sql, dataSourceName);
    }

    public Integer execSQL(String name, String sql) {
        return execSQL(name, sql, null);
    }

    /**
     * 从SQL文件中获取SQL语句
     */
    private String getSQLByFile(String sqlName, JSONObject params) {
        // 获取原始sql语句
        SqlMapper.SqlResource resource = sqlMapper.getResource(sqlName);
        String source = resource.getSource();
        // 附加用户注册的对象到SQL中
        params.put("ENV", applicationService.getValues());
        params.put("RESOURCE", resource.getJsonParams());
        params.put("authTools", authLogic);
        SpringUtils.getBean(PluginMapper.class).getPluginMap().forEach(params::put);
        return Expression.run(source, params.toMap()).toString();
    }

    /**
     * 过滤order by子句，产生求和结果
     */
    private String filterOutOrderBy(String source, Object sums) {
        int idx = source.toLowerCase().lastIndexOf("order by");
        if (idx == -1) {
            idx = source.length() - 1;
        }
        StringBuilder sql = new StringBuilder("select ");
        // 如果有求和部分，产生求和部分的语句
        if (sums != null) {
            if (sums instanceof JSONArray arraySums) {
                for (int i = 0; i < arraySums.length(); i++) {
                    String name = (String) arraySums.get(i);
                    sql.append("sum(").append(name).append(") ").append(name).append(", ");
                }
            } else if (sums instanceof JSONObject objSums) {
                Iterator<String> keys = objSums.keys();
                while (keys.hasNext()) {
                    String name = keys.next();
                    String value = objSums.getString(name);
                    sql.append("sum(").append(value).append(") ").append(name).append(", ");
                }
            }
        }
        if (idx != -1) {
            sql.append("count(*) n, 1 placeholder from ( ").append(source, 0, idx).append(") t_");
        }
        return sql.toString();
    }

    /**
     * 获取数据源类型名称
     *
     * @return 数据源类型
     * @see com.af.v4.system.common.datasource.enums.DbType
     */
    public String getDbTypeName() {
        return DynamicDataSource.getDbType().name();
    }

    /**
     * 解析符合数据源的SQL
     *
     * @return 符合
     */
    public String resolveSQL(JSONObject sqlList) {
        String dbType = getDbTypeName();
        String sqlStr = sqlList.optString(dbType, null);
        if (sqlStr == null) {
            throw new ServiceException("未找到符合数据源[" + dbType + "]的SQL");
        }
        return sqlStr;
    }


}
