package com.af.v4.system.common.jpa.sensitive;

import com.af.v4.system.common.jpa.service.SqlService;
import com.af.v4.system.common.jpa.types.MetaData;
import com.af.v4.system.common.plugins.security.SensitiveDataEnhancer;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 敏感数据拦截器
 * 作为EntityServer与敏感数据处理系统的唯一接口
 * <p>
 * 设计原则：
 * - 最小侵入：EntityServer只需调用一个方法
 * - 可插拔：可以通过配置完全禁用敏感数据处理
 * - 职责单一：只负责拦截和委托处理
 * - 现代优化：使用Stream API、Pattern Matching和Switch表达式
 */
@Component
public class SensitiveDataInterceptor {

    private static final Logger log = LoggerFactory.getLogger(SensitiveDataInterceptor.class);
    // 业务表中存储加密表ID引用的字段名
    public static final String CRYPTO_REF_ID_FIELD = "f_crypto_ref_id";
    // 常量定义
    private static final String PROTECTED_ROW_LOG = "[PROTECTED]";
    private static final String STATE_FIELD = "f_state";
    private static final String VALID_STATE = "valid";
    private static final String SOURCE_ENTITY_ID_FIELD = "source_entity_id";
    private static final String OLD_ID_FIELD = "old_id";

    // SQL 查询相关常量
    private static final String QUERY_EXISTING_CRYPTO_DESCRIPTION = "查询现有的加密数据";
    private static final String PAGE_SIZE_PARAM = "pageSize";
    private static final String PAGE_NO_PARAM = "pageNo";
    private static final String SINGLE_RESULT_SIZE = "1";

    private final SqlService sqlService;
    private final SensitiveDataProperties properties;

    // 敏感数据分类器
    private final SensitiveDataClassifier classifier = new SensitiveDataClassifier();

    public SensitiveDataInterceptor(SqlService sqlService, SensitiveDataProperties properties) {
        this.sqlService = sqlService;
        this.properties = properties;
    }

    /**
     * 新的单一拦截器入口方法
     * 在数据保存前处理所有敏感字段：同表模式直接修改row，跨表模式返回加密数据
     *
     * @param entityName 实体名称（业务表名）
     * @param row        数据行对象（同表模式字段会直接在此修改）
     * @param metaData   实体元数据（包含敏感字段配置）
     * @return SensitiveDataResult 包含跨表模式的加密数据，null表示无跨表数据
     */
    public SensitiveDataEnhancer.SensitiveDataResult beforeSave(
            String entityName,
            JSONObject row,
            MetaData metaData) {

        // 功能开关检查
        if (!properties.isEnabled()) {
            return null;
        }

        // 配置检查
        if (metaData == null || !metaData.hasMetaAttributes()) {
            return null;
        }

        // 获取所有敏感字段
        Set<String> sensitiveFields = metaData.getSensitiveFieldNames();
        if (sensitiveFields.isEmpty()) {
            return null;
        }

        try {
            // 分离同表模式和跨表模式字段
            Map<String, String> sameTableFields = sensitiveFields.stream()
                    .filter(fieldName -> entityName.equals(metaData.getSensitiveTargetTable(fieldName)))
                    .collect(Collectors.toMap(
                            fieldName -> fieldName,
                            fieldName -> fieldName
                    ));

            Map<String, String> crossTableFields = sensitiveFields.stream()
                    .filter(fieldName -> !entityName.equals(metaData.getSensitiveTargetTable(fieldName)))
                    .collect(Collectors.toMap(
                            fieldName -> fieldName,
                            metaData::getSensitiveTargetTable
                    ));

            // 1. 处理同表模式：直接修改row中的数据
            if (!sameTableFields.isEmpty()) {
                processSameTableMode(entityName, row, sameTableFields, metaData);
            }

            // 2. 处理跨表模式：返回加密数据
            SensitiveDataEnhancer.SensitiveDataResult crossTableResult = null;
            if (!crossTableFields.isEmpty()) {
                crossTableResult = processCrossTableMode(entityName, row, crossTableFields, metaData);
            }

            return crossTableResult;

        } catch (Exception e) {
            // 敏感数据处理失败不应该影响主业务流程
            return handleProcessingError("beforeSave处理失败", entityName, row, e);
        }
    }

    /**
     * 分类结果处理助手方法
     * 使用现代化的模式匹配来简化分类结果处理逻辑
     */
    private boolean shouldProcessField(SensitiveDataClassifier.DataClassificationResult classification,
                                       String fieldName, Object originalValue, String context) {
        return switch (classification) {
            case SensitiveDataClassifier.DataClassificationResult.RawData rawData -> {
                log.debug("{} 检测到原始数据: {} = {} ({})", context, fieldName, originalValue, rawData.getDescription());
                yield true; // 需要处理原始数据
            }
            case SensitiveDataClassifier.DataClassificationResult.AlreadyMasked alreadyMasked -> {
                log.debug("{} 字段已脱敏，跳过: {} = {} ({})", context, fieldName, originalValue, alreadyMasked.getDescription());
                yield false; // 已处理过，跳过
            }
            case SensitiveDataClassifier.DataClassificationResult.AlreadyEncrypted alreadyEncrypted -> {
                log.debug("{} 字段已加密，跳过: {} = {} ({})", context, fieldName, originalValue, alreadyEncrypted.getDescription());
                yield false; // 已处理过，跳过
            }
            case SensitiveDataClassifier.DataClassificationResult.NullValue nullValue -> {
                log.debug("{} 字段为空值，跳过: {} = {} ({})", context, fieldName, originalValue, nullValue.getDescription());
                yield false; // 空值，跳过
            }
            case SensitiveDataClassifier.DataClassificationResult.Uncertain uncertain -> {
                log.debug("{} 字段状态不确定，跳过: {} = {} ({})", context, fieldName, originalValue, uncertain.getDescription());
                yield false; // 不确定，跳过
            }
        };
    }

    /**
     * 统一错误处理方法
     * 提供一致的错误日志记录和异常处理
     */
    private SensitiveDataEnhancer.SensitiveDataResult handleProcessingError(String operation, String entityName,
                                                                            JSONObject row, Exception e) {
        String rowInfo = (row != null) ? PROTECTED_ROW_LOG : "null";
        log.error("敏感数据处理失败，将跳过处理: {} -> {}, row={}", operation, entityName, rowInfo, e);
        return null;
    }

    /**
     * 处理单个字段错误
     */
    private void handleFieldError(String operation, String fieldName, Exception e) {
        log.warn("{} 字段处理出错: {} - {}", operation, fieldName, e.getMessage());
        if (log.isDebugEnabled()) {
            log.debug("{} 字段处理详细错误信息:", operation, e);
        }
    }

    /**
     * 处理同表模式：直接修改row中的敏感字段为脱敏值，同时添加hash和加密字段
     *
     * @param entityName      业务表名
     * @param row             业务数据行（直接修改）
     * @param sameTableFields 同表模式的敏感字段名称
     * @param metaData        实体元数据
     */
    private void processSameTableMode(String entityName, JSONObject row,
                                      Map<String, String> sameTableFields,
                                      com.af.v4.system.common.jpa.types.MetaData metaData) {

        log.debug("处理同表模式: {}, 字段数={}", entityName, sameTableFields.size());

        // 使用 Stream API 过滤和处理字段
        sameTableFields.keySet().stream()
                .filter(fieldName -> Optional.ofNullable(row.opt(fieldName)).isPresent())
                .forEach(fieldName -> {
                    Object originalValue = row.opt(fieldName);

                    // 使用分类器判断数据状态
                    SensitiveDataClassifier.DataClassificationResult classification =
                            classifier.classify(fieldName, originalValue);

                    // 使用现代化的模式匹配处理分类结果
                    if (shouldProcessField(classification, fieldName, originalValue, "同表模式")) {
                        processSingleField(fieldName, originalValue.toString(), row, metaData);
                    }
                });
    }

    /**
     * 处理单个敏感字段
     * 统一的字段处理逻辑，减少代码重复
     */
    private void processSingleField(String fieldName, String originalValue, JSONObject row, MetaData metaData) {
        // 使用工具函数处理敏感字段
        SensitiveDataEnhancer.SensitiveProcessResult result = SensitiveDataEnhancer.processSensitiveField(
                fieldName, originalValue, metaData.getSensitiveMaskStrategy(fieldName));

        if (result != null && result.isValid()) {
            // 修改row：原字段改为脱敏值，添加hash和加密字段
            row.put(fieldName, result.maskedValue()); // 业务字段存脱敏值
            row.put(metaData.getSensitiveHashField(fieldName), result.hashValue()); // 添加hash字段
            row.put(metaData.getSensitiveEncryptedField(fieldName), result.encryptedValue()); // 添加加密字段

            log.debug("敏感字段处理完成: {} -> {}", fieldName, result.maskedValue());
        } else {
            log.warn("敏感字段处理失败或结果无效: {}", fieldName);
        }
    }

    /**
     * 处理跨表模式：查询现有加密数据，hash比较优化，只处理变化字段
     *
     * @param entityName       业务表名
     * @param row              业务数据行
     * @param crossTableFields 跨表模式的敏感字段配置
     * @param metaData         实体元数据
     * @return SensitiveDataResult 包含加密表数据
     */
    private SensitiveDataEnhancer.SensitiveDataResult processCrossTableMode(String entityName, JSONObject row,
                                                                            Map<String, String> crossTableFields,
                                                                            com.af.v4.system.common.jpa.types.MetaData metaData) {

        log.debug("处理跨表模式: {}, 字段数={}", entityName, crossTableFields.size());

        // 根据约束1：一个业务表只能有一个跨表目标，所以可以统一处理
        String targetTable = crossTableFields.keySet().stream()
                .findFirst()
                .map(metaData::getSensitiveTargetTable)
                .orElse(null);

        String relationField = crossTableFields.keySet().stream()
                .findFirst()
                .map(metaData::getSensitiveRelationField)
                .orElse(null);

        if (targetTable == null || relationField == null) {
            log.warn("跨表模式配置无效: targetTable={}, relationField={}", targetTable, relationField);
            return null;
        }

        // 获取关联ID值（用于查询现有加密数据）
        Object relationIdValue = row.opt(relationField);
        JSONObject cryptoData;

        if (relationIdValue == null) {
            // 如果没有关联ID，说明是新增数据，直接构建加密数据
            // 对于新增数据，使用临时ID或者空值，实际ID会在EntityServer中生成
            cryptoData = buildNewCryptoData(crossTableFields, row, "", metaData);
        } else {
            // 查询现有加密数据进行hash比较优化
            cryptoData = buildOptimizedCryptoData(targetTable, relationIdValue.toString(), crossTableFields, row, metaData);
        }

        if (!cryptoData.isEmpty()) {
            return new SensitiveDataEnhancer.SensitiveDataResult(targetTable, cryptoData, relationField);
        }

        return null; // 没有需要保存的跨表数据
    }

    /**
     * 构建新的加密数据（用于新增记录）
     */
    private JSONObject buildNewCryptoData(Map<String, String> crossTableFields,
                                          JSONObject row, String relationId,
                                          com.af.v4.system.common.jpa.types.MetaData metaData) {
        JSONObject cryptoData = new JSONObject();
        boolean hasProcessedFields = false; // 标记是否有字段被实际处理

        // 使用 Stream API 过滤和处理字段
        for (String fieldName : crossTableFields.keySet()) {
            Object originalValue = row.opt(fieldName);
            if (originalValue == null) continue;

            // 使用分类器判断数据状态
            SensitiveDataClassifier.DataClassificationResult classification =
                    classifier.classify(fieldName, originalValue);

            // 使用现代化的模式匹配处理分类结果
            if (shouldProcessField(classification, fieldName, originalValue, "跨表新增")) {
                // 使用工具函数处理敏感字段
                SensitiveDataEnhancer.SensitiveProcessResult result = SensitiveDataEnhancer.processSensitiveField(
                        fieldName, originalValue.toString(), metaData.getSensitiveMaskStrategy(fieldName));

                if (result != null && result.isValid()) {
                    // 添加到加密数据中
                    cryptoData.put(metaData.getSensitiveHashField(fieldName), result.hashValue());
                    cryptoData.put(metaData.getSensitiveEncryptedField(fieldName), result.encryptedValue());

                    // 业务表中的字段改为脱敏值
                    row.put(fieldName, result.maskedValue());

                    hasProcessedFields = true; // 标记有字段被处理
                    log.debug("新增加密数据: {} -> {}", fieldName, result.maskedValue());
                } else {
                    log.warn("跨表新增字段处理失败: {}", fieldName);
                }
            }
        }

        // 只有当有字段被实际处理时才添加固定字段并返回数据
        if (hasProcessedFields) {
            cryptoData.put(SOURCE_ENTITY_ID_FIELD, relationId);      // 关联业务表ID
            cryptoData.put(STATE_FIELD, VALID_STATE);                  // 状态字段
            log.debug("跨表加密数据构建完成，包含 {} 个敏感字段", (cryptoData.length() - 2) / 2);
            return cryptoData;
        } else {
            log.debug("所有敏感字段都已脱敏，跳过加密表数据保存");
            return new JSONObject(); // 返回空对象，表示不需要保存
        }
    }

    /**
     * 基于hash比较优化构建加密数据（用于更新记录）
     * 按照你的需求：查询现有数据，hash比较，只处理变更字段
     */
    private JSONObject buildOptimizedCryptoData(String targetTable, String relationId,
                                                Map<String, String> crossTableFields,
                                                JSONObject row,
                                                com.af.v4.system.common.jpa.types.MetaData metaData) {

        // 1. 查询现有加密数据
        JSONObject existingCrypto = queryExistingCryptoData(targetTable, relationId);

        if (existingCrypto == null) {
            // 没找到现有数据，按新增处理
            log.debug("未找到现有加密数据，按新增处理: {}", targetTable);
            return buildNewCryptoData(crossTableFields, row, relationId, metaData);
        }

        // 2. Hash比较，只处理变化的字段
        JSONObject newCryptoData = new JSONObject(existingCrypto.toString());
        boolean hasChanges = false;

        for (String fieldName : crossTableFields.keySet()) {

            Object newValue = row.opt(fieldName);
            if (newValue == null) continue;

            // 使用分类器判断数据状态
            SensitiveDataClassifier.DataClassificationResult classification =
                    classifier.classify(fieldName, newValue);

            // 使用现代化的模式匹配处理分类结果
            if (shouldProcessField(classification, fieldName, newValue, "跨表更新")) {
                String newStr = newValue.toString();
                String newHash = SensitiveDataEnhancer.generateSHA256Hash(newStr);
                String existingHash = existingCrypto.optString(metaData.getSensitiveHashField(fieldName));

                // Hash比较：只有值变化才重新加密
                if (!newHash.equals(existingHash)) {
                    // 使用工具函数处理敏感字段
                    SensitiveDataEnhancer.SensitiveProcessResult result = SensitiveDataEnhancer.processSensitiveField(
                            fieldName, newStr, metaData.getSensitiveMaskStrategy(fieldName));
                    if (result != null && result.isValid()) {
                        // 更新加密数据中的字段
                        newCryptoData.put(metaData.getSensitiveHashField(fieldName), result.hashValue());
                        newCryptoData.put(metaData.getSensitiveEncryptedField(fieldName), result.encryptedValue());

                        // 业务表中的字段改为脱敏值
                        row.put(fieldName, result.maskedValue());

                        hasChanges = true;
                        log.debug("字段值发生变化，更新加密数据: {} -> {}", fieldName, result.maskedValue());
                    }
                } else {
                    // 值未变化，业务表中依然使用脱敏值（直接调用静态方法）
                    String maskedValue = SensitiveDataEnhancer.applyMaskingStrategy(fieldName, newStr, metaData.getSensitiveMaskStrategy(fieldName));
                    row.put(fieldName, maskedValue);
                    log.debug("字段值未变化，跳过加密处理: {}", fieldName);
                }
            }
        }

        if (hasChanges) {
            // 有字段变化，移除ID字段，让系统生成新记录
            newCryptoData.put(SOURCE_ENTITY_ID_FIELD, relationId);
            newCryptoData.put(OLD_ID_FIELD, newCryptoData.get("id"));
            newCryptoData.put(STATE_FIELD, VALID_STATE);
            newCryptoData.remove("id");
            log.debug("跨表更新数据构建完成，有字段变化");
            return newCryptoData;
        } else {
            log.debug("跨表更新：无字段变化，不需要保存新的加密记录");
            return new JSONObject(); // 返回空对象，表示不需要保存
        }
    }

    /**
     * 查询现有的加密数据
     * 使用 StringTemplate 和常量优化查询构建
     *
     * @param targetTable 加密表名
     * @param relationId  关联ID值
     * @return 现有的加密数据，未找到返回null
     */
    private JSONObject queryExistingCryptoData(String targetTable, String relationId) {
        try {
            // 构建 SQL 查询语句，使用常量
            String sql = STR."SELECT * FROM \{targetTable} WHERE \{SOURCE_ENTITY_ID_FIELD} = '\{relationId}' AND \{STATE_FIELD} = '\{VALID_STATE}'";

            JSONObject queryParams = new JSONObject()
                    .put(PAGE_SIZE_PARAM, SINGLE_RESULT_SIZE)
                    .put(PAGE_NO_PARAM, SINGLE_RESULT_SIZE);

            JSONArray results = sqlService.querySQL(QUERY_EXISTING_CRYPTO_DESCRIPTION, sql, queryParams);

            return results.isEmpty() ? null : results.getJSONObject(0);

        } catch (Exception e) {
            handleFieldError("查询现有加密数据", STR."\{targetTable}(ID:\{relationId})", e);
            return null;
        }
    }

}
