package com.af.v4.system.common.jpa.transformer;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.TemporalAccessorUtil;
import com.af.v4.system.common.jpa.utils.ClobUtil;
import org.hibernate.query.TupleTransformer;
import org.json.JSONObject;
import org.json.JSONParserConfiguration;

import java.math.BigDecimal;
import java.sql.Clob;
import java.time.temporal.TemporalAccessor;
import java.util.Date;
import java.util.Map;

/**
 * 标准结果转换器
 */
public class StandardAliasTransformer implements TupleTransformer<JSONObject> {

    public static final StandardAliasTransformer INSTANCE = new StandardAliasTransformer();

    private StandardAliasTransformer() {
    }

    protected static void runTrans(Object obj, Object[] tuple, String[] aliases) {
        for (int i = 0; i < aliases.length; i++) {
            String alias = aliases[i];
            Object value = tuple[i];
            switch (value) {
                case null -> value = JSONObject.NULL;
                case BigDecimal bigDecimal -> {
                    if (isIntegerValue(bigDecimal)) {
                        long longValue = bigDecimal.longValue();
                        if (longValue < Integer.MAX_VALUE) {
                            value = (int) longValue;
                        } else {
                            value = longValue;
                        }
                    }
                }
                case Date date ->
                    // 处理日期格式
                        value = DateUtil.formatDateTime(date);
                case TemporalAccessor temporalAccessor ->
                    // 处理日期格式
                        value = TemporalAccessorUtil.format(temporalAccessor, "yyyy-MM-dd HH:mm:ss");
                case Clob clob -> value = ClobUtil.getClobString(clob);
                default -> {
                }
            }
            if (obj instanceof JSONObject object) {
                object.put(alias.toLowerCase(), value);
            } else if (obj instanceof Map objectMap) {
                objectMap.put(alias.toLowerCase(), value);
            }
        }
    }

    private static boolean isIntegerValue(BigDecimal bd) {
        return bd.stripTrailingZeros().scale() <= 0;
    }

    @Override
    public JSONObject transformTuple(Object[] tuple, String[] aliases) {
        JSONObject result = new JSONObject(new JSONParserConfiguration().setOrdered(true));
        runTrans(result, tuple, aliases);
        return result;
    }
}
