package com.af.v4.system.common.file.utils;

import com.af.v4.system.common.core.constant.FileConstants;
import com.af.v4.system.common.core.enums.OSType;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.resource.config.ResourceConfig;
import org.springframework.stereotype.Component;

import java.io.File;

/**
 * 文件路径构建工具
 */
@Component
public final class FilePathUtil {

    private final ApplicationService applicationService;

    private final ResourceConfig resourceConfig;

    public FilePathUtil(ApplicationService applicationService, ResourceConfig resourceConfig) {
        this.applicationService = applicationService;
        this.resourceConfig = resourceConfig;
    }

    /**
     * 获取文件扩展名
     *
     * @param filename 文件名
     * @return 扩展名
     */
    public static String fileExt(String filename) {
        String[] array = filename.split("\\.");
        return array[array.length - 1].toLowerCase();
    }

    /**
     * 获取服务器的文件存储根路径
     *
     * @return 服务器的文件存储根路径
     */
    public String getFileRootPathByServer() {
        String rootPath = resourceConfig.getFileRootPath();
        if (rootPath == null) {
            // 生成保存的相对路径
            OSType osType = ApplicationService.getOSType();
            if (osType == OSType.WINDOWS) {
                rootPath = "D:" + File.separatorChar + "resources";
            } else {
                rootPath = File.separatorChar + "usr" + File.separatorChar + "local" + File.separatorChar + "tomcat" + File.separatorChar + "files";
            }
        }
        return rootPath;
    }

    /**
     * 获取服务器的文件存储扩展路径
     *
     * @return 服务器的文件存储扩展路径
     */
    public String getFileExtraPathByServer(String resourceType) {
        return File.separatorChar + applicationService.getApplicationName()
                + File.separatorChar + FileConstants.RESOURCE_TYPE_2_DIR.get(resourceType);
    }

    /**
     * 获取服务器的文件存储完整路径
     *
     * @return 服务器的文件存储完整路径
     */
    public String getFileFullPathByServer(String resourceType, String fileName) {
        return getFileRootPathByServer() + getFileExtraPathByServer(resourceType) + fileName;
    }

    /**
     * 获取服务器的文件下载路径
     *
     * @return 服务器的文件下载路径
     */
    public String getFileDownloadUrlByServer(String resourceType, String fileName) {
        return "/resource" + getFileExtraPathByServer(resourceType) + fileName;
    }

    /**
     * 获取minio的文件存储扩展路径
     *
     * @return minio的文件存储扩展路径
     */
    public String getFileExtraPathBySSO(String resourceType, String bucketName) {
        return '/' + FileConstants.RESOURCE_TYPE_2_DIR.get(resourceType);
    }

    /**
     * 获取minio的文件存储完整路径
     *
     * @return minio的文件存储完整路径
     */
    public String getFileFullPathBySSO(String resourceType, String fileName, String bucketName) {
        return getFileExtraPathBySSO(resourceType, bucketName) + fileName;
    }

    /**
     * 获取minio的文件下载路径
     *
     * @return minio的文件下载路径
     */
    public String getFileDownloadUrlBySSO(String resourceType, String fileName, String bucketName) {
        return "/minio/" + (bucketName == null || bucketName.isEmpty() ? applicationService.getApplicationName() : bucketName)
                + getFileExtraPathBySSO(resourceType, bucketName) + fileName;
    }
}
