package com.af.v4.system.common.file.service;

import io.minio.*;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;


@Component
public class MinioService {

    private final MinioClient minioClient;

    public MinioService(MinioClient minioClient) {
        this.minioClient = minioClient;
    }

    /**
     * 上传至OSS服务
     * @param file          文件
     * @param fileName      文件名，带路径
     * @param bucketName    OSS的bucket名
     * @throws Exception    异常
     */
    public void upload(MultipartFile file, String fileName, String bucketName) throws Exception {
        // 检查bucket是否存在，不存在则建立bucket
        boolean bucketExists = minioClient.bucketExists(BucketExistsArgs.builder().bucket(bucketName).build());
        if (!bucketExists) {
            minioClient.makeBucket(MakeBucketArgs.builder().bucket(bucketName).build());
        }

        // 将原始文件名保存到metaData
        String originalFilename = file.getOriginalFilename();
        Map<String, String> metaData = new HashMap<>();
        metaData.put("originalFilename", originalFilename);

        // 保存
        minioClient.putObject(
                PutObjectArgs.builder()
                        .bucket(bucketName)
                        .object(fileName)
                        .userMetadata(metaData)
                        .stream(file.getInputStream(), file.getSize(), -1)
                        .contentType(file.getContentType())
                        .build()
        );
    }

    /**
     * 从OSS中下载文件
     * @param bucketName    OSS的bucket名
     * @param fileName      文件名，带路径
     * @return              返回inputStream流
     * @throws Exception    异常
     */
    public InputStream download(String bucketName, String fileName) throws Exception {
        if (fileName.startsWith("/")) {
            fileName = fileName.substring(1);
        }
        return minioClient.getObject(GetObjectArgs.builder().bucket(bucketName).object(fileName).build());
    }
}
